/**
 * MapView Core Module
 * Main class for handling the property map view functionality
 */
class MapView {
    constructor(options) {
        this.options = {
            mapContainer: 'advanced-map',
            apiEndpoint: '/properties/map-data',
            currencySymbol: '₹',
            propertyTypes: [],
            defaultCenter: [20.5937, 78.9629],
            defaultZoom: 6,
            clusterRadius: 50,
            animationDuration: 300,
            debounceDelay: 400,
            maxZoomForClustering: 15,
            ...options
        };
        
        this.map = null;
        this.markers = null;
        this.properties = [];
        this.filteredProperties = [];
        this.currentFilters = {};
        this.drawControl = null;
        this.drawnItems = null;
        this.heatmapLayer = null;
        this.propertyCache = new Map();
        this.searchCache = new Map();
        
        // Performance optimization
        this.requestQueue = [];
        this.isRequestInProgress = false;
        this.lastRequestTime = 0;
        this.minRequestInterval = 1000;
        
        // State management
        this.state = {
            isLoading: false,
            isInitialized: false,
            isInitializing: false,
            initRetryCount: 0,
            maxRetries: 3,
            isDarkMode: localStorage.getItem('map_dark_mode') === 'true',
            isFullscreen: false,
            selectedProperty: null,
            favorites: JSON.parse(localStorage.getItem('property_favorites') || '[]'),
            compareList: JSON.parse(localStorage.getItem('property_compare') || '[]'),
            autoFit: true,
            viewMode: 'markers', // markers, heatmap, satellite
            filterCount: 0,
            currentSort: 'newest',
            isAnimating: false
        };
        
        // Animation and interaction state
        this.animations = {
            propertyCards: new Map(),
            markers: new Map(),
            filters: null
        };
        
        // Bind methods
        this.loadProperties = this.loadProperties.bind(this);
        this.applyFilters = this.applyFilters.bind(this);
        this.onMapMove = this.onMapMove.bind(this);
        this.handleResize = this.handleResize.bind(this);
        // Debounced loader used by filter changes
        this.debouncedLoadProperties = this.debounce(() => this.loadProperties(), this.options.debounceDelay);
        
        // Initialize performance monitoring
        this.performance = {
            loadTimes: [],
            renderTimes: [],
            lastUpdate: Date.now()
        };
    }

    /**
     * Normalize backend property object so downstream code can rely on latitude/longitude
     */
    normalizeProperty(p) {
        if (!p || typeof p !== 'object') return p;
        // Try several possible keys for coordinates
        const candidates = [
            ['latitude', 'longitude'],
            ['lat', 'lng'],
            ['lat', 'lon'],
            ['map_lat', 'map_lng'],
            ['map_latitude', 'map_longitude'],
            ['geo_lat', 'geo_lng'],
            ['location_latitude', 'location_longitude']
        ];
        let lat = p.latitude, lng = p.longitude;
        if (!(this.isFiniteNumber(lat) && this.isFiniteNumber(lng))) {
            for (const [la, lo] of candidates) {
                const vlat = p[la];
                const vlng = p[lo];
                const flat = vlat != null ? parseFloat(vlat) : NaN;
                const flng = vlng != null ? parseFloat(vlng) : NaN;
                if (this.isFiniteNumber(flat) && this.isFiniteNumber(flng)) {
                    lat = flat; lng = flng; break;
                }
            }
        }
        // Assign normalized values if valid
        if (this.isFiniteNumber(lat) && this.isFiniteNumber(lng)) {
            p.latitude = lat;
            p.longitude = lng;
        }
        // Coerce common numeric fields to numbers to make comparisons stable
        if (p.price != null) {
            const n = parseFloat(p.price);
            if (this.isFiniteNumber(n)) p.price = n;
        }
        if (p.bedrooms != null) {
            const n = parseInt(p.bedrooms);
            if (this.isFiniteNumber(n)) p.bedrooms = n;
        }
        if (p.bathrooms != null) {
            const n = parseInt(p.bathrooms);
            if (this.isFiniteNumber(n)) p.bathrooms = n;
        }
        return p;
    }

    isFiniteNumber(v) {
        return typeof v === 'number' && isFinite(v);
    }
    
    /**
     * Initialize the map view - simplified for use with MapInitializer
     */
    async init() {
        // Prevent multiple simultaneous initializations
        if (this.state.isInitialized || this.state.isInitializing) {
            console.log('Map already initialized or initializing');
            return;
        }
        
        this.state.isInitializing = true;
        
        try {
            // Initialize core components (Google Maps API already verified by MapInitializer)
            await this.initializeMap();
            this.initializeControls();
            this.initializeEventListeners();
            this.initializePerformanceMonitoring();
            
            // Apply saved dark mode preference
            if (this.state.isDarkMode) {
                try {
                    document.body.classList.add('dark-mode');
                    this.map?.setOptions?.({ styles: this.darkMapStyle });
                } catch (_) {}
            }
            
            // Initialize modules with error handling
            await this.initializeModules();
            
            // Load initial properties
            await this.loadProperties();
            
            // Initialize advanced features
            this.initializeAdvancedFeatures();
            
            this.state.isInitialized = true;
            this.showNotification('Map initialized successfully', 'success');
            
        } catch (error) {
            console.error('Error initializing map:', error);
            throw error; // Let MapInitializer handle the error
        } finally {
            this.state.isInitializing = false;
        }
    }
    
    /**
     * Simple loading state management
     */
    showLoadingState(message = 'Loading...') {
        // Loading state is now handled by MapInitializer
        console.log('MapView:', message);
    }
    
    hideLoadingState() {
        // Loading state is now handled by MapInitializer
        console.log('MapView: Loading complete');
    }
    
    /**
     * Initialize modules with proper error handling
     */
    async initializeModules() {
        const modules = [
            { name: 'MapFilters', class: MapFilters, property: 'filters' },
            { name: 'MapMarkers', class: MapMarkers, property: 'markersManager' },
            { name: 'MapInteractions', class: MapInteractions, property: 'interactions' },
            { name: 'PropertyDetails', class: PropertyDetails, property: 'propertyDetails' },
            { name: 'MapUtils', class: MapUtils, property: 'utils' }
        ];
        
        for (const module of modules) {
            try {
                if (typeof module.class !== 'undefined') {
                    this[module.property] = new module.class(this);
                } else {
                    console.warn(`${module.name} module not available`);
                }
            } catch (error) {
                console.error(`Error initializing ${module.name}:`, error);
            }
        }
    }
    
    /**
     * Initialize performance monitoring
     */
    initializePerformanceMonitoring() {
        // Monitor frame rate
        let frameCount = 0;
        let lastTime = performance.now();
        
        const monitorFrameRate = () => {
            frameCount++;
            const currentTime = performance.now();
            
            if (currentTime - lastTime >= 1000) {
                const fps = Math.round((frameCount * 1000) / (currentTime - lastTime));
                this.performance.fps = fps;
                
                if (fps < 30) {
                    console.warn('Low frame rate detected:', fps);
                }
                
                frameCount = 0;
                lastTime = currentTime;
            }
            
            requestAnimationFrame(monitorFrameRate);
        };
        
        requestAnimationFrame(monitorFrameRate);
    }
    
    /**
     * Initialize advanced features
     */
    initializeAdvancedFeatures() {
        // Initialize keyboard shortcuts
        this.initializeKeyboardShortcuts();
        
        // Initialize gesture recognition for mobile
        this.initializeGestureRecognition();
        
        // Initialize real-time updates
        this.initializeRealTimeUpdates();
        
        // Initialize analytics
        this.initializeAnalytics();
    }
    
    /**
     * Simplified map initialization - Google Maps API verification handled by MapInitializer
     */
    
    /**
     * Initialize the Google Map
     */
    async initializeMap() {
        const container = document.getElementById(this.options.mapContainer);
        if (!container) {
            throw new Error(`Map container not found: ${this.options.mapContainer}`);
        }
        
        // Ensure container is visible and has dimensions
        if (container.offsetWidth === 0 || container.offsetHeight === 0) {
            throw new Error('Map container has no dimensions');
        }
        
        this.map = new google.maps.Map(container, {
            center: { lat: this.options.defaultCenter[0], lng: this.options.defaultCenter[1] },
            zoom: this.options.defaultZoom,
            mapId: 'DEMO_MAP_ID', // Required for Advanced Markers
            mapTypeControl: true,
            streetViewControl: false,
            fullscreenControl: false,
            gestureHandling: 'cooperative',
            mapTypeControlOptions: {
                style: google.maps.MapTypeControlStyle.HORIZONTAL_BAR,
                position: google.maps.ControlPosition.TOP_CENTER
            },
            zoomControlOptions: {
                position: google.maps.ControlPosition.RIGHT_CENTER
            }
        });
        
        // Wait for map to be fully loaded
        await new Promise((resolve, reject) => {
            const timeout = setTimeout(() => {
                reject(new Error('Map load timeout'));
            }, 10000);
            
            google.maps.event.addListenerOnce(this.map, 'idle', () => {
                clearTimeout(timeout);
                resolve();
            });
        });
        
        // Holder for Google markers and clusterer (managed by MapMarkers module)
        this.markers = [];
        this.markerClusterer = null;
        
        // Predefine dark mode style
        this.darkMapStyle = [
            { elementType: 'geometry', stylers: [{ color: '#242f3e' }] },
            { elementType: 'labels.text.stroke', stylers: [{ color: '#242f3e' }] },
            { elementType: 'labels.text.fill', stylers: [{ color: '#746855' }] },
            { featureType: 'administrative.locality', elementType: 'labels.text.fill', stylers: [{ color: '#d59563' }] },
            { featureType: 'poi', elementType: 'labels.text.fill', stylers: [{ color: '#d59563' }] },
            { featureType: 'poi.park', elementType: 'geometry', stylers: [{ color: '#263c3f' }] },
            { featureType: 'poi.park', elementType: 'labels.text.fill', stylers: [{ color: '#6b9a76' }] },
            { featureType: 'road', elementType: 'geometry', stylers: [{ color: '#38414e' }] },
            { featureType: 'road', elementType: 'geometry.stroke', stylers: [{ color: '#212a37' }] },
            { featureType: 'road', elementType: 'labels.text.fill', stylers: [{ color: '#9ca5b3' }] },
            { featureType: 'road.highway', elementType: 'geometry', stylers: [{ color: '#746855' }] },
            { featureType: 'road.highway', elementType: 'geometry.stroke', stylers: [{ color: '#1f2835' }] },
            { featureType: 'road.highway', elementType: 'labels.text.fill', stylers: [{ color: '#f3d19c' }] },
            { featureType: 'transit', elementType: 'geometry', stylers: [{ color: '#2f3948' }] },
            { featureType: 'transit.station', elementType: 'labels.text.fill', stylers: [{ color: '#d59563' }] },
            { featureType: 'water', elementType: 'geometry', stylers: [{ color: '#17263c' }] },
            { featureType: 'water', elementType: 'labels.text.fill', stylers: [{ color: '#515c6d' }] },
            { featureType: 'water', elementType: 'labels.text.stroke', stylers: [{ color: '#17263c' }] }
        ];
    }
    
    /**
     * Initialize custom controls
     */
    initializeControls() {
        // Google Maps shows zoom controls by default. We handle other controls via existing HTML buttons.
        // No-op for custom in-map controls for now.
    }
    
    /**
     * Initialize event listeners
     */
    initializeEventListeners() {
        // Map events
        this.map.addListener('idle', this.onMapMove);
        
        // UI event listeners
        this.initializeUIEventListeners();

        // Window resize -> gracefully notify map to recompute tiles
        window.addEventListener('resize', this.handleResize, { passive: true });
    }
    
    /**
     * Initialize UI event listeners
     */
    initializeUIEventListeners() {
        // Toggle filters panel
        const toggleFilters = document.getElementById('toggle-filters');
        if (toggleFilters) {
            toggleFilters.addEventListener('click', () => {
                this.toggleFiltersPanel();
            });
        }
        
        // Dark mode toggle
        const toggleDarkMode = document.getElementById('toggle-dark-mode');
        if (toggleDarkMode) {
            toggleDarkMode.addEventListener('click', () => {
                this.toggleDarkMode();
            });
        }
        
        // Fullscreen toggle
        const fullscreenMap = document.getElementById('fullscreen-map');
        if (fullscreenMap) {
            fullscreenMap.addEventListener('click', () => {
                const el = document.getElementById(this.options.mapContainer);
                if (!document.fullscreenElement) {
                    el.requestFullscreen?.();
                } else {
                    document.exitFullscreen?.();
                }
            });
        }
        // Also wire topbar fullscreen button
        const topbarFullscreen = document.getElementById('fullscreen-toggle');
        if (topbarFullscreen) {
            topbarFullscreen.addEventListener('click', () => {
                const el = document.getElementById(this.options.mapContainer);
                if (!document.fullscreenElement) {
                    el.requestFullscreen?.();
                } else {
                    document.exitFullscreen?.();
                }
            });
        }
        
        // Clear filters
        const clearFilters = document.getElementById('clear-filters');
        if (clearFilters) {
            clearFilters.addEventListener('click', () => {
                this.clearAllFilters();
            });
        }

        // Reset Filters inside list empty state
        const resetFiltersBtn = document.getElementById('reset-filters');
        if (resetFiltersBtn) {
            resetFiltersBtn.addEventListener('click', () => this.clearAllFilters());
        }

        // Sorting dropdown in property list header
        const listSort = document.getElementById('list-sort');
        if (listSort) {
            listSort.addEventListener('change', (e) => {
                this.state.currentSort = e.target.value || 'newest';
                this.sortFilteredProperties();
                this.updatePropertyDisplay();
            });
        }

        // My Location from topbar
        const myLocationBtn = document.getElementById('view-my-location');
        if (myLocationBtn) {
            myLocationBtn.addEventListener('click', () => {
                this.locateUser();
            });
        }
    }
    
    /**
     * Load properties from API
     */
    async loadProperties() {
        this.setLoading(true);
        
        try {
            const params = new URLSearchParams();
            // Append filters
            Object.entries(this.currentFilters || {}).forEach(([k, v]) => {
                if (v === undefined || v === null || v === '') return;
                if (typeof v === 'object') {
                    // For nested objects like userLocation, serialize as JSON
                    params.append(k, JSON.stringify(v));
                } else {
                    params.append(k, v);
                }
            });
            // Append bounds explicitly only when zoomed in enough; wide bounds often return empty
            const zoom = this.map?.getZoom?.();
            if (typeof zoom === 'number' && zoom >= 8) {
                const b = this.getMapBounds();
                if (b && typeof b.north === 'number') {
                    params.append('north', b.north);
                    params.append('south', b.south);
                    params.append('east', b.east);
                    params.append('west', b.west);
                }
            }
            
            const response = await fetch(`${this.options.apiEndpoint}?${params}`, {
                headers: { 'Accept': 'application/json' }
            });
            let data = null;
            try {
                data = await response.json();
            } catch (e) {
                console.warn('Non-JSON response for properties:', e);
                data = null;
            }
            
            // Normalize properties array from various shapes
            let props = [];
            if (Array.isArray(data)) {
                props = data;
            } else if (data && Array.isArray(data.properties)) {
                props = data.properties;
            } else if (data && data.properties && Array.isArray(data.properties.data)) {
                props = data.properties.data; // Laravel pagination style
            } else if (data && Array.isArray(data.data)) {
                props = data.data;
            } else if (data && Array.isArray(data.items)) {
                props = data.items;
            } else if (data && data.success && Array.isArray(data.results)) {
                props = data.results;
            }

            if (props && props.length >= 0) {
                // Normalize coordinates and other fields before storing
                this.properties = (props || []).map(p => this.normalizeProperty(p));
                // Diagnostics: count valid coordinates
                const withCoords = (this.properties || []).filter(p => this.isFiniteNumber(p?.latitude) && this.isFiniteNumber(p?.longitude));
                if (withCoords.length === 0) {
                    console.warn('No properties contain valid coordinates. Markers will not render.');
                    // Inform user once to check data configuration
                    this.showNotification('No mappable coordinates in results', 'warning');
                }
                // Apply current filters to freshly loaded data for consistent UX
                this.filteredProperties = (this.properties || []).filter(p => this.matchesFilters(p));
                this.updatePropertyDisplay();
                this.updateStats();
            } else {
                this.showNotification('Error loading properties', 'error');
            }
        } catch (error) {
            console.error('Error loading properties:', error);
            this.showNotification('Error loading properties', 'error');
        } finally {
            this.setLoading(false);
        }
    }
    
    /**
     * Apply current filters to properties
     */
    applyFilters() {
        // Re-enable auto fit when filters drive the updates (not user pan)
        this.state.autoFit = true;
        // If we already have local data, update UI immediately for responsiveness
        if (this.properties.length) {
            this.filteredProperties = this.properties.filter(property => this.matchesFilters(property));
            this.updatePropertyDisplay();
            this.updateStats();
        }
        // Always trigger a debounced refresh from server to stay in sync
        this.debouncedLoadProperties();
    }
    
    /**
     * Check if property matches current filters
     */
    matchesFilters(property) {
        // Search filter
        if (this.currentFilters.search) {
            const searchTerm = this.currentFilters.search.toLowerCase();
            const searchFields = [
                property.title,
                property.location,
                property.description,
                property.address
            ].join(' ').toLowerCase();
            
            if (!searchFields.includes(searchTerm)) {
                return false;
            }
        }
        
        // Property type filter
        if (this.currentFilters.type && property.type_id != this.currentFilters.type) {
            return false;
        }
        
        // Price range filter
        if (this.currentFilters.price_min && property.price < this.currentFilters.price_min) {
            return false;
        }
        
        if (this.currentFilters.price_max && property.price > this.currentFilters.price_max) {
            return false;
        }
        
        // Bedrooms filter
        if (this.currentFilters.bedrooms && property.bedrooms < this.currentFilters.bedrooms) {
            return false;
        }
        
        // Bathrooms filter
        if (this.currentFilters.bathrooms && property.bathrooms < this.currentFilters.bathrooms) {
            return false;
        }
        
        // Purpose filter
        if (this.currentFilters.purpose && property.purpose !== this.currentFilters.purpose) {
            return false;
        }

        // Status filter
        if (this.currentFilters.status && property.status !== this.currentFilters.status) {
            return false;
        }
        
        // County/City/Area filters (if provided by backend data)
        if (this.currentFilters.county && property.county && property.county !== this.currentFilters.county) {
            return false;
        }
        if (this.currentFilters.city && property.city && property.city !== this.currentFilters.city) {
            return false;
        }
        if (this.currentFilters.area && property.area_name && property.area_name !== this.currentFilters.area) {
            return false;
        }
        
        // Verified filter
        if (this.currentFilters.verified && !property.is_verified) {
            return false;
        }
        
        // Furnished filter
        if (this.currentFilters.furnished && !property.is_furnished) {
            return false;
        }
        
        // Radius + user location filter
        if (this.currentFilters.radius && this.currentFilters.userLocation && property.latitude && property.longitude) {
            const { lat, lng } = this.currentFilters.userLocation;
            const radiusKm = parseFloat(this.currentFilters.radius);
            if (!isNaN(radiusKm)) {
                let distance = null;
                if (this.utils && typeof this.utils.calculateDistance === 'function') {
                    distance = this.utils.calculateDistance(lat, lng, property.latitude, property.longitude);
                } else {
                    // Fallback simple Haversine
                    const toRad = (d) => d * Math.PI / 180;
                    const R = 6371;
                    const dLat = toRad(property.latitude - lat);
                    const dLng = toRad(property.longitude - lng);
                    const a = Math.sin(dLat/2)**2 + Math.cos(toRad(lat)) * Math.cos(toRad(property.latitude)) * Math.sin(dLng/2)**2;
                    const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1-a));
                    distance = R * c;
                }
                if (distance > radiusKm) {
                    return false;
                }
            }
        }
        
        // Polygon filter (point-in-polygon)
        if (Array.isArray(this.currentFilters.polygonPath) && this.currentFilters.polygonPath.length >= 3) {
            const path = this.currentFilters.polygonPath;
            if (!(property.latitude && property.longitude)) {
                return false;
            }
            try {
                // Build cached polygon if path changed
                const key = JSON.stringify(path);
                if (this._polygonCacheKey !== key) {
                    this._polygonCacheKey = key;
                    this._cachedPolygon = new google.maps.Polygon({
                        paths: path.map(p => new google.maps.LatLng(p.lat, p.lng))
                    });
                }
                if (google.maps.geometry && google.maps.geometry.poly && typeof google.maps.geometry.poly.containsLocation === 'function') {
                    const point = new google.maps.LatLng(property.latitude, property.longitude);
                    if (!google.maps.geometry.poly.containsLocation(point, this._cachedPolygon)) {
                        return false;
                    }
                } else {
                    // Fallback: quick bounding-box pre-check
                    const lats = path.map(p => p.lat);
                    const lngs = path.map(p => p.lng);
                    const minLat = Math.min(...lats), maxLat = Math.max(...lats);
                    const minLng = Math.min(...lngs), maxLng = Math.max(...lngs);
                    if (property.latitude < minLat || property.latitude > maxLat || property.longitude < minLng || property.longitude > maxLng) {
                        return false;
                    }
                }
            } catch (_) {
                // On any error, do not filter out
            }
        }
        
        return true;
    }
    
    /**
     * Update property display on map and list
     */
    updatePropertyDisplay() {
        if (this.markersManager) {
            const mappable = (this.filteredProperties || []).filter(p => this.isFiniteNumber(p?.latitude) && this.isFiniteNumber(p?.longitude));
            // Pass selected property ID to keep popup open
            const selectedId = this.state.selectedProperty?.id;
            this.markersManager.updateMarkers(mappable, selectedId);
        }
        
        // Ensure list uses current sort order
        this.sortFilteredProperties();
        this.updatePropertyList();
        this.updatePropertyCount();
    }

    /**
     * Sort filtered properties based on current selection
     */
    sortFilteredProperties() {
        if (!Array.isArray(this.filteredProperties) || this.filteredProperties.length === 0) return;
        const sort = this.state.currentSort || 'newest';
        const byNum = (v) => (typeof v === 'number' ? v : parseFloat(v) || 0);
        const byDate = (v) => (v ? new Date(v).getTime() : 0);
        const hasUserLoc = this.currentFilters.userLocation && this.utils && typeof this.utils.calculateDistance === 'function';

        switch (sort) {
            case 'price-low':
                this.filteredProperties.sort((a, b) => byNum(a.price) - byNum(b.price));
                break;
            case 'price-high':
                this.filteredProperties.sort((a, b) => byNum(b.price) - byNum(a.price));
                break;
            case 'distance':
                if (hasUserLoc) {
                    const { lat, lng } = this.currentFilters.userLocation;
                    this.filteredProperties.sort((a, b) => {
                        const da = (a.latitude && a.longitude) ? this.utils.calculateDistance(lat, lng, a.latitude, a.longitude) : Number.MAX_VALUE;
                        const db = (b.latitude && b.longitude) ? this.utils.calculateDistance(lat, lng, b.latitude, b.longitude) : Number.MAX_VALUE;
                        return da - db;
                    });
                }
                break;
            case 'relevance':
                // Placeholder: keep as-is or implement scoring later
                break;
            case 'newest':
            default:
                // Sort by created_at desc if present; fallback to id desc
                this.filteredProperties.sort((a, b) => (byDate(b.created_at) - byDate(a.created_at)) || (byNum(b.id) - byNum(a.id)));
        }
    }
    
    /**
     * Update property list in sidebar
     */
    updatePropertyList() {
        const container = document.getElementById('property-list-container');
        if (!container) return;
        
        const emptyState = document.getElementById('empty-state');
        
        if (this.filteredProperties.length === 0) {
            container.innerHTML = '';
            if (emptyState) emptyState.style.display = 'block';
            return;
        }
        
        if (emptyState) emptyState.style.display = 'none';
        
        const listHTML = this.filteredProperties.map(property => {
            return this.generatePropertyListItem(property);
        }).join('');
        
        container.innerHTML = listHTML;
        
        // Add event listeners to list items
        container.querySelectorAll('.property-list-item').forEach(item => {
            item.addEventListener('click', (e) => {
                const propertyId = item.dataset.propertyId;
                const property = this.filteredProperties.find(p => p.id == propertyId);
                if (property) {
                    this.selectProperty(property);
                }
            });
        });
    }
    
    /**
     * Generate HTML for property list item
     */
    generatePropertyListItem(property) {
        const isFavorite = this.state.favorites.includes(property.id);
        const price = this.formatPrice(property.price);
        const image = property.image || '/assets/images/property-placeholder.jpg';
        
        return `
            <div class="property-list-item" data-property-id="${property.id}">
                <div class="property-image">
                    <img src="${image}" alt="${property.title}" loading="lazy">
                    <div class="property-badges">
                        ${property.is_verified ? '<span class="badge badge-success">Verified</span>' : ''}
                        ${property.is_featured ? '<span class="badge badge-warning">Featured</span>' : ''}
                    </div>
                </div>
                <div class="property-details">
                    <h6 class="property-title">${property.title}</h6>
                    <p class="property-location">
                        <i class="fa fa-map-marker"></i> ${property.location}
                    </p>
                    <div class="property-specs">
                        <span><i class="fa fa-bed"></i> ${property.bedrooms}</span>
                        <span><i class="fa fa-bath"></i> ${property.bathrooms}</span>
                        <span><i class="fa fa-home"></i> ${property.area} sq ft</span>
                    </div>
                    <div class="property-price">
                        <strong>${price}</strong>
                        <span class="price-type">/${property.purpose === 'rent' ? 'month' : 'total'}</span>
                    </div>
                </div>
                <div class="property-actions">
                    <button class="action-btn favorite-btn ${isFavorite ? 'active' : ''}" 
                            data-property-id="${property.id}" title="Add to Favorites">
                        <i class="fa fa-heart${isFavorite ? '' : '-o'}"></i>
                    </button>
                    <button class="action-btn share-btn" 
                            data-property-id="${property.id}" title="Share">
                        <i class="fa fa-share-alt"></i>
                    </button>
                </div>
            </div>
        `;
    }
    
    /**
     * Update property count display
     */
    updatePropertyCount() {
        const count = this.filteredProperties.length;
        const countElements = document.querySelectorAll('#property-count, #list-count');
        
        countElements.forEach(element => {
            if (element.id === 'property-count') {
                element.textContent = `${count} properties`;
            } else {
                element.textContent = `(${count})`;
            }
        });

        // Update pagination info if present
        const showingCount = document.getElementById('showing-count');
        const totalCount = document.getElementById('total-count');
        if (showingCount) showingCount.textContent = String(this.filteredProperties.length);
        if (totalCount) totalCount.textContent = String(this.properties.length);
    }
    
    /**
     * Update map statistics
     */
    updateStats() {
        if (!this.filteredProperties.length) return;
        
        const prices = this.filteredProperties.map(p => p.price).filter(p => p > 0);
        const avgPrice = prices.reduce((a, b) => a + b, 0) / prices.length;
        const minPrice = Math.min(...prices);
        const maxPrice = Math.max(...prices);
        
        const visibleCount = document.getElementById('visible-count');
        const averagePrice = document.getElementById('average-price');
        const priceRange = document.getElementById('price-range');
        
        if (visibleCount) visibleCount.textContent = this.filteredProperties.length;
        if (averagePrice) averagePrice.textContent = this.formatPrice(avgPrice);
        if (priceRange) priceRange.textContent = `${this.formatPrice(minPrice)} - ${this.formatPrice(maxPrice)}`;
    }
    
    /**
     * Handle map move events
     */
    onMapMove() {
        // When user moves the map, stop auto-fitting on subsequent updates
        this.state.autoFit = false;
        // Debounce the property loading
        clearTimeout(this.moveTimeout);
        this.moveTimeout = setTimeout(() => {
            // Only reload when zoom is reasonably close to results
            const zoom = this.map?.getZoom?.();
            if (typeof zoom === 'number' && zoom >= 8) {
                this.loadProperties();
            }
        }, 700);
    }
    
    /**
     * Handle polygon drawing
     */
    onPolygonDrawn(layer) {
        this.currentFilters.polygon = layer;
        this.applyFilters();
    }
    
    /**
     * Get current map bounds
     */
    getMapBounds() {
        const bounds = this.map.getBounds();
        if (!bounds) return {};
        const ne = bounds.getNorthEast();
        const sw = bounds.getSouthWest();
        return {
            north: ne.lat(),
            south: sw.lat(),
            east: ne.lng(),
            west: sw.lng()
        };
    }
    
    /**
     * Select a property
     */
    selectProperty(property) {
        this.state.selectedProperty = property;
        
        // Highlight on map
        if (this.markersManager) {
            this.markersManager.highlightProperty(property.id);
        }
        
        // Show in drawer
        if (this.propertyDetails) {
            this.propertyDetails.showPropertyDetails(property);
        }
        
        // Center map on property
        this.map.setZoom(16);
        this.map.panTo({ lat: property.latitude, lng: property.longitude });
    }
    
    /**
     * Toggle filters panel
     */
    toggleFiltersPanel() {
        const panel = document.getElementById('map-right-panel');
        if (panel) {
            panel.classList.toggle('collapsed');
            // Trigger a resize after animation so tiles re-render correctly
            setTimeout(() => {
                if (this.map) {
                    google.maps.event.trigger(this.map, 'resize');
                }
            }, 300);
        }
    }
    
    /**
     * Toggle dark mode
     */
    toggleDarkMode() {
        this.state.isDarkMode = !this.state.isDarkMode;
        document.body.classList.toggle('dark-mode', this.state.isDarkMode);
        // Apply Google Maps dark style
        this.map.setOptions({ styles: this.state.isDarkMode ? this.darkMapStyle : null });
        try { localStorage.setItem('map_dark_mode', this.state.isDarkMode ? 'true' : 'false'); } catch (_) {}
    }
    
    /**
     * Clear all filters
     */
    clearAllFilters() {
        this.currentFilters = {};
        
        // Reset filter UI
        if (this.filters) {
            this.filters.resetFilters();
        }
        // ensure next marker update fits results nicely
        this.state.autoFit = true;
        this.loadProperties();
    }

    /**
     * Handle window resize to keep tiles crisp
     */
    handleResize() {
        clearTimeout(this.resizeTimeout);
        this.resizeTimeout = setTimeout(() => {
            if (this.map) {
                try { google.maps.event.trigger(this.map, 'resize'); } catch (_) {}
            }
        }, 150);
    }
    
    /**
     * Locate user
     */
    locateUser() {
        if (!navigator.geolocation) {
            this.showNotification('Geolocation is not supported', 'error');
            return;
        }
        
        navigator.geolocation.getCurrentPosition(
            (position) => {
                const lat = position.coords.latitude;
                const lng = position.coords.longitude;
                this.map.setZoom(14);
                this.map.panTo({ lat, lng });
                // Add user location marker
                new google.maps.Marker({
                    position: { lat, lng },
                    map: this.map,
                    title: 'Your location'
                });
                this.showNotification('Location found', 'success');
            },
            () => {
                this.showNotification('Unable to get location', 'error');
            }
        );
    }
    
    /**
     * Set loading state
     */
    setLoading(loading) {
        this.state.isLoading = loading;
        const loadingOverlay = document.getElementById('map-loading');
        
        if (loadingOverlay) {
            loadingOverlay.style.display = loading ? 'flex' : 'none';
        }
    }
    
    /**
     * Show notification
     */
    showNotification(message, type = 'info') {
        // Create notification element if it doesn't exist
        let notification = document.getElementById('map-notification');
        if (!notification) {
            notification = document.createElement('div');
            notification.id = 'map-notification';
            notification.className = 'map-notification';
            document.body.appendChild(notification);
        }
        
        notification.className = `map-notification ${type}`;
        notification.textContent = message;
        notification.style.display = 'block';
        
        // Auto hide after 3 seconds
        setTimeout(() => {
            notification.style.display = 'none';
        }, 3000);
    }

    /**
     * Loading helpers used during init
     */
    showLoadingState(message) {
        // Update loading overlay text if present
        const overlay = document.getElementById('map-loading');
        if (overlay) {
            const p = overlay.querySelector('p');
            if (p && typeof message === 'string') p.textContent = message;
        }
        this.setLoading(true);
    }

    hideLoadingState() {
        this.setLoading(false);
    }

    showErrorState(error) {
        console.error(error);
        this.setLoading(false);
    }

    /**
     * No-op advanced feature initializers (can be extended later)
     */
    initializeKeyboardShortcuts() { /* no-op */ }
    initializeGestureRecognition() { /* no-op */ }
    initializeRealTimeUpdates() { /* no-op */ }
    initializeAnalytics() { /* no-op */ }
    
    /**
     * Format price with currency
     */
    formatPrice(price) {
        if (!price) return 'Price on request';
        
        const formatter = new Intl.NumberFormat('en-IN', {
            style: 'currency',
            currency: 'INR',
            minimumFractionDigits: 0,
            maximumFractionDigits: 0
        });
        
        return formatter.format(price);
    }
    
    /**
     * Destroy map view
     */
    destroy() {
        // Clear timeouts
        if (this.moveTimeout) {
            clearTimeout(this.moveTimeout);
        }
        if (this.resizeTimeout) {
            clearTimeout(this.resizeTimeout);
        }
        // Clear markers
        if (this.markersManager) {
            this.markersManager.clearMarkers();
        }
        // Remove map listeners
        if (this.map) {
            google.maps.event.clearInstanceListeners(this.map);
        }
        // Remove window listeners
        window.removeEventListener('resize', this.handleResize);
        
        // Clear references
        this.map = null;
        this.markers = [];
        this.properties = [];
        this.filteredProperties = [];
    }

    /**
     * Simple notification system
     */
    showNotification(message, type = 'info') {
        // Create notification element if it doesn't exist
        let notificationContainer = document.getElementById('map-notifications');
        if (!notificationContainer) {
            notificationContainer = document.createElement('div');
            notificationContainer.id = 'map-notifications';
            notificationContainer.style.cssText = `
                position: fixed;
                top: 20px;
                right: 20px;
                z-index: 10000;
                max-width: 350px;
            `;
            document.body.appendChild(notificationContainer);
        }

        const notification = document.createElement('div');
        const colors = {
            success: '#28a745',
            error: '#dc3545',
            warning: '#ffc107',
            info: '#17a2b8'
        };

        notification.style.cssText = `
            background: ${colors[type] || colors.info};
            color: white;
            padding: 12px 16px;
            border-radius: 4px;
            margin-bottom: 10px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.15);
            font-size: 14px;
            opacity: 0;
            transform: translateX(100%);
            transition: all 0.3s ease;
        `;
        notification.textContent = message;

        notificationContainer.appendChild(notification);

        // Animate in
        setTimeout(() => {
            notification.style.opacity = '1';
            notification.style.transform = 'translateX(0)';
        }, 10);

        // Auto remove after 4 seconds
        setTimeout(() => {
            notification.style.opacity = '0';
            notification.style.transform = 'translateX(100%)';
            setTimeout(() => {
                if (notification.parentNode) {
                    notification.parentNode.removeChild(notification);
                }
            }, 300);
        }, 4000);
    }

    /**
     * Set loading state
     */
    setLoading(loading) {
        this.state.isLoading = loading;
        const loadingOverlay = document.getElementById('map-loading');
        if (loadingOverlay) {
            loadingOverlay.style.display = loading ? 'flex' : 'none';
        }
    }

    /**
     * Small debounce utility
     */
    debounce(fn, wait = 300) {
        let t;
        return (...args) => {
            clearTimeout(t);
            t = setTimeout(() => fn.apply(this, args), wait);
        };
    }
}

// Export for use in other modules
window.MapView = MapView;
