/**
 * MapFilters Module
 * Handles all filter-related functionality for the map view
 */
class MapFilters {
    constructor(mapView) {
        this.mapView = mapView;
        this.debounceTimeout = null;
        this.drawingManager = null;
        this.activePolygon = null;
        
        this.init();
    }
    
    /**
     * Initialize filter functionality
     */
    init() {
        this.initializeSearchFilters();
        this.initializePriceFilters();
        this.initializePropertyFilters();
        this.initializeFeatureFilters();
        this.initializeMapTools();
        this.initializeAreaFilters();
        this.initializeStatusAndSalePrice();
    }
    
    /**
     * Initialize search-related filters
     */
    initializeSearchFilters() {
        // Main search input
        const searchInput = document.getElementById('map-search');
        if (searchInput) {
            searchInput.addEventListener('input', (e) => {
                this.debounceFilter(() => {
                    this.mapView.currentFilters.search = e.target.value;
                    this.mapView.applyFilters();
                });
            });
            
            // Search suggestions
            searchInput.addEventListener('focus', () => {
                this.showSearchSuggestions();
            });
            
            searchInput.addEventListener('blur', () => {
                setTimeout(() => this.hideSearchSuggestions(), 200);
            });
        }
        
        // Location search
        const locationSearch = document.getElementById('location-search');
        if (locationSearch) {
            locationSearch.addEventListener('input', (e) => {
                this.debounceFilter(() => {
                    this.handleLocationSearch(e.target.value);
                });
            });
        }
        
        // GPS location button
        const gpsBtn = document.getElementById('use-gps');
        if (gpsBtn) {
            gpsBtn.addEventListener('click', () => {
                this.useCurrentLocation();
            });
        }
        
        // Search radius
        const radiusSlider = document.getElementById('search-radius');
        if (radiusSlider) {
            radiusSlider.addEventListener('input', (e) => {
                const value = e.target.value;
                document.getElementById('radius-value').textContent = `${value} km`;
                
                this.debounceFilter(() => {
                    this.mapView.currentFilters.radius = value;
                    this.mapView.applyFilters();
                });
            });
        }
    }
    
    /**
     * Initialize price-related filters
     */
    initializePriceFilters() {
        // Price input fields
        const priceMinInput = document.getElementById('price-min-input');
        const priceMaxInput = document.getElementById('price-max-input');
        
        if (priceMinInput) {
            priceMinInput.addEventListener('input', (e) => {
                this.updatePriceFilter('min', e.target.value);
            });
        }
        
        if (priceMaxInput) {
            priceMaxInput.addEventListener('input', (e) => {
                this.updatePriceFilter('max', e.target.value);
            });
        }
        
        // Price range sliders
        const priceMinSlider = document.getElementById('price-min');
        const priceMaxSlider = document.getElementById('price-max');
        
        if (priceMinSlider && priceMaxSlider) {
            priceMinSlider.addEventListener('input', (e) => {
                this.updatePriceSlider('min', e.target.value);
            });
            
            priceMaxSlider.addEventListener('input', (e) => {
                this.updatePriceSlider('max', e.target.value);
            });
        }
        
        // Price presets
        document.querySelectorAll('.preset-btn').forEach(btn => {
            btn.addEventListener('click', (e) => {
                const min = e.target.dataset.min;
                const max = e.target.dataset.max;
                this.setPriceRange(min, max);
                
                // Update active preset
                document.querySelectorAll('.preset-btn').forEach(b => b.classList.remove('active'));
                e.target.classList.add('active');
            });
        });
    }
    
    /**
     * Initialize property-related filters
     */
    initializePropertyFilters() {
        // Property type
        const propertyType = document.getElementById('map-property-type');
        if (propertyType) {
            propertyType.addEventListener('change', (e) => {
                this.mapView.currentFilters.type = e.target.value;
                this.mapView.applyFilters();
            });
        }
        
        // Bedrooms
        const bedrooms = document.getElementById('map-bedrooms');
        if (bedrooms) {
            bedrooms.addEventListener('change', (e) => {
                this.mapView.currentFilters.bedrooms = e.target.value;
                this.mapView.applyFilters();
            });
        }
        
        // Bathrooms
        const bathrooms = document.getElementById('map-bathrooms');
        if (bathrooms) {
            bathrooms.addEventListener('change', (e) => {
                this.mapView.currentFilters.bathrooms = e.target.value;
                this.mapView.applyFilters();
            });
        }
        
        // Purpose (Sale/Rent)
        document.querySelectorAll('input[name="purpose"]').forEach(radio => {
            radio.addEventListener('change', (e) => {
                this.mapView.currentFilters.purpose = e.target.value;
                this.mapView.applyFilters();
            });
        });
    }

    /**
     * Initialize county/city/area filters
     */
    initializeAreaFilters() {
        const county = document.getElementById('filter-county');
        const city = document.getElementById('filter-city');
        const area = document.getElementById('filter-area');

        if (county) {
            county.addEventListener('change', (e) => {
                this.mapView.currentFilters.county = e.target.value || '';
                this.debounceFilter(() => this.mapView.applyFilters());
            });
        }
        if (city) {
            city.addEventListener('change', (e) => {
                this.mapView.currentFilters.city = e.target.value || '';
                this.debounceFilter(() => this.mapView.applyFilters());
            });
        }
        if (area) {
            area.addEventListener('change', (e) => {
                this.mapView.currentFilters.area = e.target.value || '';
                this.debounceFilter(() => this.mapView.applyFilters());
            });
        }
    }

    /**
     * Initialize status and sale price filters
     */
    initializeStatusAndSalePrice() {
        const status = document.getElementById('filter-status');
        const priceRange = document.getElementById('filter-price');

        if (status) {
            status.addEventListener('change', (e) => {
                this.mapView.currentFilters.status = e.target.value || '';
                this.mapView.applyFilters();
            });
        }

        if (priceRange) {
            priceRange.addEventListener('change', (e) => {
                const val = e.target.value;
                if (!val) {
                    delete this.mapView.currentFilters.price_min;
                    delete this.mapView.currentFilters.price_max;
                    this.mapView.applyFilters();
                    return;
                }
                const [minStr, maxStr] = val.split('-');
                const min = parseInt(minStr) || 0;
                const max = parseInt(maxStr) || null;
                this.mapView.currentFilters.price_min = min || undefined;
                this.mapView.currentFilters.price_max = max || undefined;
                this.mapView.applyFilters();
            });
        }
    }
    
    /**
     * Initialize feature-related filters
     */
    initializeFeatureFilters() {
        // Verified properties
        const verifiedOnly = document.getElementById('verified-only');
        if (verifiedOnly) {
            verifiedOnly.addEventListener('change', (e) => {
                this.mapView.currentFilters.verified = e.target.checked;
                this.mapView.applyFilters();
            });
        }
        
        // Furnished properties
        const furnishedOnly = document.getElementById('furnished-only');
        if (furnishedOnly) {
            furnishedOnly.addEventListener('change', (e) => {
                this.mapView.currentFilters.furnished = e.target.checked;
                this.mapView.applyFilters();
            });
        }
        
        // Parking available
        const parkingAvailable = document.getElementById('parking-available');
        if (parkingAvailable) {
            parkingAvailable.addEventListener('change', (e) => {
                this.mapView.currentFilters.parking = e.target.checked;
                this.mapView.applyFilters();
            });
        }
    }
    
    /**
     * Initialize map tools
     */
    initializeMapTools() {
        // Draw polygon tool
        const drawPolygon = document.getElementById('draw-polygon');
        if (drawPolygon) {
            drawPolygon.addEventListener('click', () => {
                this.toggleDrawingMode();
            });
        }
        
        // Clear polygon tool
        const clearPolygon = document.getElementById('clear-polygon');
        if (clearPolygon) {
            clearPolygon.addEventListener('click', () => {
                this.clearDrawnPolygons();
            });
        }
        
        // Save search tool
        const saveSearch = document.getElementById('save-search');
        if (saveSearch) {
            saveSearch.addEventListener('click', () => {
                this.saveCurrentSearch();
            });
        }
    }
    
    /**
     * Show search suggestions
     */
    showSearchSuggestions() {
        const suggestions = document.getElementById('search-suggestions');
        if (!suggestions) return;
        
        // Sample suggestions - in real implementation, fetch from API
        const sampleSuggestions = [
            'Mumbai apartments',
            'Delhi houses',
            'Bangalore villas',
            'Pune commercial',
            'Gurgaon luxury homes'
        ];
        
        const suggestionsHTML = sampleSuggestions.map(suggestion => 
            `<div class="suggestion-item">${suggestion}</div>`
        ).join('');
        
        suggestions.innerHTML = suggestionsHTML;
        suggestions.style.display = 'block';
        
        // Add click handlers
        suggestions.querySelectorAll('.suggestion-item').forEach(item => {
            item.addEventListener('click', () => {
                document.getElementById('map-search').value = item.textContent;
                this.mapView.currentFilters.search = item.textContent;
                this.mapView.applyFilters();
                this.hideSearchSuggestions();
            });
        });
    }
    
    /**
     * Hide search suggestions
     */
    hideSearchSuggestions() {
        const suggestions = document.getElementById('search-suggestions');
        if (suggestions) {
            suggestions.style.display = 'none';
        }
    }
    
    /**
     * Handle location search
     */
    handleLocationSearch(location) {
        if (!location.trim()) return;
        
        // In real implementation, use geocoding service
        // For now, just set as search filter
        this.mapView.currentFilters.location = location;
        this.mapView.applyFilters();
    }
    
    /**
     * Use current GPS location
     */
    useCurrentLocation() {
        if (!navigator.geolocation) {
            this.mapView.showNotification('Geolocation not supported', 'error');
            return;
        }
        
        const gpsBtn = document.getElementById('use-gps');
        if (gpsBtn) {
            gpsBtn.innerHTML = '<i class="fa fa-spinner fa-spin"></i>';
        }
        
        navigator.geolocation.getCurrentPosition(
            (position) => {
                const lat = position.coords.latitude;
                const lng = position.coords.longitude;
                
                // Center map on user location (Google Maps)
                if (this.mapView.map) {
                    this.mapView.state.autoFit = false; // prevent jumpy refits
                    try { this.mapView.map.setZoom(14); } catch (_) {}
                    try { this.mapView.map.panTo({ lat, lng }); } catch (_) {}
                }
                
                // Update location filter
                this.mapView.currentFilters.userLocation = { lat, lng };
                this.mapView.applyFilters();
                
                // Update location input with reverse geocoding (simplified)
                const locationInput = document.getElementById('location-search');
                if (locationInput) {
                    locationInput.value = `${lat.toFixed(4)}, ${lng.toFixed(4)}`;
                }
                
                this.mapView.showNotification('Location updated', 'success');
                
                if (gpsBtn) {
                    gpsBtn.innerHTML = '<i class="fa fa-crosshairs"></i>';
                }
            },
            (error) => {
                this.mapView.showNotification('Unable to get location', 'error');
                if (gpsBtn) {
                    gpsBtn.innerHTML = '<i class="fa fa-crosshairs"></i>';
                }
            }
        );
    }
    
    /**
     * Update price filter
     */
    updatePriceFilter(type, value) {
        const numValue = parseInt(value) || 0;
        
        if (type === 'min') {
            this.mapView.currentFilters.price_min = numValue;
            const slider = document.getElementById('price-min');
            if (slider) slider.value = numValue;
        } else {
            this.mapView.currentFilters.price_max = numValue;
            const slider = document.getElementById('price-max');
            if (slider) slider.value = numValue;
        }
        
        this.updatePriceLabels();
        this.updatePriceSliderRange();
        
        this.debounceFilter(() => {
            this.mapView.applyFilters();
        });
    }
    
    /**
     * Update price slider
     */
    updatePriceSlider(type, value) {
        const numValue = parseInt(value);
        
        if (type === 'min') {
            this.mapView.currentFilters.price_min = numValue;
            const input = document.getElementById('price-min-input');
            if (input) input.value = numValue;
        } else {
            this.mapView.currentFilters.price_max = numValue;
            const input = document.getElementById('price-max-input');
            if (input) input.value = numValue;
        }
        
        this.updatePriceLabels();
        this.updatePriceSliderRange();
        
        this.debounceFilter(() => {
            this.mapView.applyFilters();
        });
    }
    
    /**
     * Set price range from preset
     */
    setPriceRange(min, max) {
        this.mapView.currentFilters.price_min = parseInt(min);
        this.mapView.currentFilters.price_max = parseInt(max);
        
        // Update inputs
        const minInput = document.getElementById('price-min-input');
        const maxInput = document.getElementById('price-max-input');
        const minSlider = document.getElementById('price-min');
        const maxSlider = document.getElementById('price-max');
        
        if (minInput) minInput.value = min;
        if (maxInput) maxInput.value = max;
        if (minSlider) minSlider.value = min;
        if (maxSlider) maxSlider.value = max;
        
        this.updatePriceLabels();
        this.updatePriceSliderRange();
        this.mapView.applyFilters();
    }
    
    /**
     * Update price labels
     */
    updatePriceLabels() {
        const minLabel = document.getElementById('price-min-label');
        const maxLabel = document.getElementById('price-max-label');
        
        if (minLabel) {
            const minValue = this.mapView.currentFilters.price_min || 0;
            minLabel.textContent = this.mapView.formatPrice(minValue);
        }
        
        if (maxLabel) {
            const maxValue = this.mapView.currentFilters.price_max || 10000000;
            maxLabel.textContent = maxValue >= 10000000 ? '₹1Cr+' : this.mapView.formatPrice(maxValue);
        }
    }
    
    /**
     * Update price slider range visual
     */
    updatePriceSliderRange() {
        const minSlider = document.getElementById('price-min');
        const maxSlider = document.getElementById('price-max');
        const sliderRange = document.querySelector('.slider-range');
        
        if (!minSlider || !maxSlider || !sliderRange) return;
        
        const min = parseInt(minSlider.value);
        const max = parseInt(maxSlider.value);
        const sliderMin = parseInt(minSlider.min);
        const sliderMax = parseInt(minSlider.max);
        
        const minPercent = ((min - sliderMin) / (sliderMax - sliderMin)) * 100;
        const maxPercent = ((max - sliderMin) / (sliderMax - sliderMin)) * 100;
        
        sliderRange.style.left = `${minPercent}%`;
        sliderRange.style.width = `${maxPercent - minPercent}%`;
    }
    
    /**
     * Toggle drawing mode
     */
    toggleDrawingMode() {
        const drawBtn = document.getElementById('draw-polygon');
        if (!drawBtn) return;
        
        // Ensure libraries and map exist
        if (!(window.google && google.maps && google.maps.drawing) || !this.mapView.map) {
            this.mapView.showNotification('Drawing tools are not available', 'warning');
            return;
        }

        // Toggle active state
        drawBtn.classList.toggle('active');

        // Initialize drawing manager lazily
        if (!this.drawingManager) {
            this.drawingManager = new google.maps.drawing.DrawingManager({
                drawingMode: google.maps.drawing.OverlayType.POLYGON,
                drawingControl: false,
                polygonOptions: {
                    fillColor: '#2563eb',
                    fillOpacity: 0.1,
                    strokeColor: '#2563eb',
                    strokeWeight: 2,
                    clickable: false,
                    editable: true,
                    zIndex: 10
                }
            });
            this.drawingManager.setMap(this.mapView.map);

            // When polygon completed, store and filter
            google.maps.event.addListener(this.drawingManager, 'overlaycomplete', (e) => {
                if (e.type === google.maps.drawing.OverlayType.POLYGON) {
                    // Remove previous polygon
                    if (this.activePolygon) {
                        try { this.activePolygon.setMap(null); } catch (_) {}
                    }
                    this.activePolygon = e.overlay;

                    // Persist path to filters for server/UI consistency
                    const path = this.activePolygon.getPath().getArray().map(latLng => ({ lat: latLng.lat(), lng: latLng.lng() }));
                    this.mapView.currentFilters.polygonPath = path;
                    this.mapView.applyFilters();

                    // Stop drawing mode after one polygon
                    this.drawingManager.setDrawingMode(null);
                    drawBtn.classList.add('active');
                    drawBtn.innerHTML = '<i class="fa fa-draw-polygon"></i><span>Drawing Active</span>';
                }
            });

            // Update filters when polygon is edited
            const updatePathFromPolygon = () => {
                if (!this.activePolygon) return;
                const path = this.activePolygon.getPath().getArray().map(latLng => ({ lat: latLng.lat(), lng: latLng.lng() }));
                this.mapView.currentFilters.polygonPath = path;
                this.mapView.applyFilters();
            };
            google.maps.event.addListener(this.drawingManager, 'polygoncomplete', (poly) => {
                google.maps.event.addListener(poly.getPath(), 'insert_at', updatePathFromPolygon);
                google.maps.event.addListener(poly.getPath(), 'remove_at', updatePathFromPolygon);
                google.maps.event.addListener(poly.getPath(), 'set_at', updatePathFromPolygon);
            });
        } else {
            // Toggle drawing mode on/off
            const mode = this.drawingManager.getDrawingMode();
            this.drawingManager.setDrawingMode(mode ? null : google.maps.drawing.OverlayType.POLYGON);
        }
    }
    
    /**
     * Clear drawn polygons
     */
    clearDrawnPolygons() {
        // Remove existing polygon from map if present
        if (this.activePolygon) {
            try { this.activePolygon.setMap(null); } catch (_) {}
            this.activePolygon = null;
        }
        // Clear filter and re-apply
        this.mapView.currentFilters.polygonPath = null;
        this.mapView.applyFilters();

        // Reset draw button label if present
        const drawBtn = document.getElementById('draw-polygon');
        if (drawBtn) {
            drawBtn.classList.remove('active');
            drawBtn.innerHTML = '<i class="fa fa-draw-polygon"></i><span>Draw Search Area</span>';
        }
    }
    
    /**
     * Save current search
     */
    saveCurrentSearch() {
        const searchData = {
            filters: { ...this.mapView.currentFilters },
            bounds: this.mapView.getMapBounds(),
            timestamp: new Date().toISOString()
        };
        
        // Save to localStorage
        const savedSearches = JSON.parse(localStorage.getItem('saved_searches') || '[]');
        savedSearches.push(searchData);
        
        // Keep only last 10 searches
        if (savedSearches.length > 10) {
            savedSearches.shift();
        }
        
        localStorage.setItem('saved_searches', JSON.stringify(savedSearches));
        
        this.mapView.showNotification('Search saved successfully', 'success');
    }
    
    /**
     * Reset all filters
     */
    resetFilters() {
        // Reset form elements
        const searchInput = document.getElementById('map-search');
        const locationInput = document.getElementById('location-search');
        const propertyType = document.getElementById('map-property-type');
        const bedrooms = document.getElementById('map-bedrooms');
        const bathrooms = document.getElementById('map-bathrooms');
        
        if (searchInput) searchInput.value = '';
        if (locationInput) locationInput.value = '';
        if (propertyType) propertyType.value = '';
        if (bedrooms) bedrooms.value = '';
        if (bathrooms) bathrooms.value = '';
        
        // Reset price inputs
        const priceMinInput = document.getElementById('price-min-input');
        const priceMaxInput = document.getElementById('price-max-input');
        const priceMinSlider = document.getElementById('price-min');
        const priceMaxSlider = document.getElementById('price-max');
        
        if (priceMinInput) priceMinInput.value = '';
        if (priceMaxInput) priceMaxInput.value = '';
        if (priceMinSlider) priceMinSlider.value = 0;
        if (priceMaxSlider) priceMaxSlider.value = 10000000;
        
        // Reset checkboxes
        document.querySelectorAll('input[type="checkbox"]').forEach(cb => {
            cb.checked = false;
        });
        
        // Reset radio buttons
        document.querySelectorAll('input[name="purpose"]').forEach(radio => {
            radio.checked = false;
        });
        
        // Reset preset buttons
        document.querySelectorAll('.preset-btn').forEach(btn => {
            btn.classList.remove('active');
        });
        
        // Reset radius
        const radiusSlider = document.getElementById('search-radius');
        if (radiusSlider) {
            radiusSlider.value = 10;
            document.getElementById('radius-value').textContent = '10 km';
        }
        
        // Update price labels and range
        this.updatePriceLabels();
        this.updatePriceSliderRange();
        
        // Clear drawn polygons
        this.clearDrawnPolygons();
    }
    
    /**
     * Debounce filter updates
     */
    debounceFilter(callback, delay = 300) {
        clearTimeout(this.debounceTimeout);
        this.debounceTimeout = setTimeout(callback, delay);
    }
}

// Export for use in other modules
window.MapFilters = MapFilters;
