/**
 * MapMarkers Module
 * Handles marker creation, clustering, and interactions on the map
 */
class MapMarkers {
    constructor(mapView) {
        this.mapView = mapView;
        this.propertyMarkers = new Map();
        this.highlightedMarker = null;
        this.infoWindow = new google.maps.InfoWindow();
        this.heatmapLayer = null;
        this.clusterer = null;
        this._clusterLibLoaded = false;
        this._clusterLibLoading = false;
        
        this.init();
    }
    
    /**
     * Initialize marker functionality
     */
    init() {
        this.createCustomIcons();
    }
    
    /**
     * Create custom marker icons
     */
    createCustomIcons() {
        // Using default Google markers for now; can be replaced with custom SVGs if desired
        this.icons = {
            sale: null,
            rent: null,
            verified: null,
            featured: null,
            highlighted: null
        };
    }
    
    /**
     * Create custom marker content for AdvancedMarkerElement
     */
    createMarkerContent(property) {
        const markerDiv = document.createElement('div');
        markerDiv.className = 'custom-marker';
        
        // Determine marker color based on property type
        const color = property.purpose === 'rent' ? '#28a745' : '#007bff';
        const price = this.mapView.formatPrice ? this.mapView.formatPrice(property.price) : `₹${property.price}`;
        
        markerDiv.innerHTML = `
            <div class="marker-content" style="
                background: ${color};
                color: white;
                padding: 8px 12px;
                border-radius: 20px;
                font-size: 12px;
                font-weight: bold;
                box-shadow: 0 2px 8px rgba(0,0,0,0.3);
                position: relative;
                white-space: nowrap;
                cursor: pointer;
                transition: all 0.2s ease;
            ">
                ${price}
                ${property.is_verified ? '<i class="fa fa-check-circle" style="margin-left: 4px;"></i>' : ''}
                <div style="
                    position: absolute;
                    bottom: -5px;
                    left: 50%;
                    transform: translateX(-50%);
                    width: 0;
                    height: 0;
                    border-left: 6px solid transparent;
                    border-right: 6px solid transparent;
                    border-top: 6px solid ${color};
                "></div>
            </div>
        `;
        
        // Add hover effects
        markerDiv.addEventListener('mouseenter', () => {
            const content = markerDiv.querySelector('.marker-content');
            content.style.transform = 'scale(1.1)';
            content.style.zIndex = '1000';
        });
        
        markerDiv.addEventListener('mouseleave', () => {
            const content = markerDiv.querySelector('.marker-content');
            content.style.transform = 'scale(1)';
            content.style.zIndex = 'auto';
        });
        
        return markerDiv;
    }
    
    /**
     * Update markers on the map
     * @param {Array} properties
     * @param {string|number} [selectedPropertyId] - Optional property ID to keep popup open
     */
    updateMarkers(properties, selectedPropertyId) {
        // Clear existing markers
        this.clearMarkers();
        this.propertyMarkers.clear();
        if (!properties || properties.length === 0) {
            return;
        }
        // Create markers for each property
        const googleMarkers = [];
        const bounds = new google.maps.LatLngBounds();
        properties.forEach(property => {
            const marker = this.createPropertyMarker(property);
            if (marker) {
                this.propertyMarkers.set(property.id, marker);
                googleMarkers.push(marker);
                // Handle different marker types for position
                const position = this.getMarkerPosition(marker);
                if (position) {
                    bounds.extend(position);
                }
            }
        });
        this.mapView.markers = googleMarkers;
        if (!bounds.isEmpty()) {
            // Only auto-fit if allowed by state to avoid jumpy UX after user pans/zooms
            if (this.mapView?.state?.autoFit) {
                this.mapView.map.fitBounds(bounds, { padding: 20 });
            }
        }
        // Update heatmap if enabled
        this.updateHeatmap(properties);
        // Apply clustering if library is available (loaded lazily)
        this.applyClustering(googleMarkers);
        // --- Keep InfoWindow open for selected property ---
        if (selectedPropertyId) {
            this.highlightProperty(selectedPropertyId);
        }
    }

    /**
     * Dynamically load MarkerClusterer library from CDN once
     */
    async ensureClustererLib() {
        if (this._clusterLibLoaded || window.markerClusterer || window.MarkerClusterer) return true;
        if (this._clusterLibLoading) {
            // Wait until loaded
            await new Promise(r => setTimeout(r, 100));
            return !!(window.markerClusterer || window.MarkerClusterer);
        }
        this._clusterLibLoading = true;
        try {
            await new Promise((resolve, reject) => {
                const s = document.createElement('script');
                s.src = 'https://unpkg.com/@googlemaps/markerclusterer/dist/index.min.js';
                s.async = true;
                s.onload = () => resolve();
                s.onerror = reject;
                document.head.appendChild(s);
            });
            this._clusterLibLoaded = true;
            return true;
        } catch (_) {
            // Fallback: no clustering
            return false;
        } finally {
            this._clusterLibLoading = false;
        }
    }

    /**
     * Apply clustering if the library is available, otherwise keep plain markers
     */
    async applyClustering(markers) {
        if (!Array.isArray(markers) || markers.length === 0) return;
        const ok = await this.ensureClustererLib();
        if (!ok) return;

        try {
            // Support both @googlemaps/markerclusterer IIFE (window.markerClusterer.MarkerClusterer)
            // and older MarkerClustererPlus global (window.MarkerClusterer)
            if (this.clusterer) {
                try { this.clusterer.clearMarkers?.(); } catch (_) {}
                try { this.clusterer.setMap?.(null); } catch (_) {}
                this.clusterer = null;
            }

            if (window.markerClusterer && window.markerClusterer.MarkerClusterer) {
                this.clusterer = new window.markerClusterer.MarkerClusterer({
                    map: this.mapView.map,
                    markers,
                    // Basic options; could be customized via mapView.options later
                    onClusterClick: undefined
                });
            } else if (window.MarkerClusterer) {
                // Legacy API: new MarkerClusterer(map, markers, options)
                this.clusterer = new window.MarkerClusterer(this.mapView.map, markers, {
                    maxZoom: this.mapView?.options?.maxZoomForClustering || 15
                });
            }
        } catch (e) {
            // If clustering fails, just leave markers as-is
            console.warn('Marker clustering not applied:', e);
        }
    }
    
    /**
     * Create a marker for a property using AdvancedMarkerElement
     */
    createPropertyMarker(property) {
        if (!property.latitude || !property.longitude) {
            return null;
        }
        
        // Create custom marker content
        const markerContent = this.createMarkerContent(property);
        const position = { lat: property.latitude, lng: property.longitude };
        
        let marker;
        
        // Use AdvancedMarkerElement if available, fallback to legacy Marker
        if (google.maps.marker && google.maps.marker.AdvancedMarkerElement) {
            marker = new google.maps.marker.AdvancedMarkerElement({
                position,
                map: this.mapView.map,
                title: property.title,
                content: markerContent
            });
        } else {
            // Fallback to legacy marker for older API versions
            marker = new google.maps.Marker({
                position,
                map: this.mapView.map,
                title: property.title
            });
        }
        
        // Click event
        marker.addListener('click', () => {
            this.onMarkerClick(property);
            const popupContent = this.createPopupContent(property);
            this.infoWindow.setContent(popupContent);
            this.infoWindow.open({ map: this.mapView.map, anchor: marker });
            // Attach popup action listeners after open
            setTimeout(() => this.attachPopupEventListeners(property), 50);
        });
        
        // Hover events (limited support in Google Maps)
        marker.addListener('mouseover', () => {
            this.onMarkerHover(property, marker);
        });
        marker.addListener('mouseout', () => {
            this.onMarkerLeave(property, marker);
        });
        
        return marker;
    }
    
    /**
     * Create popup content for property
     */
    createPopupContent(property) {
        const price = this.mapView.formatPrice(property.price);
        const image = property.image || '/assets/images/property-placeholder.jpg';
        const isFavorite = this.mapView.state.favorites.includes(property.id);
        // Modern, clean, improved popup with inline SVG icons and robust fallback for icons
        return `
            <style>
                .property-popup-modern {
                    font-family: 'Inter', 'Segoe UI', Arial, sans-serif;
                    background: #fff;
                    border-radius: 14px;
                    box-shadow: 0 4px 24px rgba(0,0,0,0.12);
                    min-width: 320px;
                    max-width: 350px;
                    padding: 0;
                    margin: 0;
                    overflow: hidden;
                }
                .property-popup-modern .popup-img-wrap {
                    position: relative;
                    width: 100%;
                    aspect-ratio: 4/2.1;
                    background: #f5f7fa;
                    overflow: hidden;
                }
                .property-popup-modern .popup-img {
                    width: 100%;
                    height: 100%;
                    object-fit: cover;
                    display: block;
                }
                .property-popup-modern .popup-badges {
                    position: absolute;
                    top: 10px;
                    left: 10px;
                    display: flex;
                    gap: 6px;
                    z-index: 2;
                }
                .property-popup-modern .badge-verified {
                    background: #eafaf1;
                    color: #28a745;
                    font-size: 12px;
                    border-radius: 6px;
                    padding: 2px 8px 2px 4px;
                    display: flex;
                    align-items: center;
                    gap: 3px;
                    font-weight: 500;
                }
                .property-popup-modern .badge-featured {
                    background: #fff8e1;
                    color: #b8860b;
                    font-size: 12px;
                    border-radius: 6px;
                    padding: 2px 8px 2px 4px;
                    display: flex;
                    align-items: center;
                    gap: 3px;
                    font-weight: 500;
                }
                .property-popup-modern .popup-price-badge {
                    position: absolute;
                    bottom: 10px;
                    left: 10px;
                    background: rgba(255,255,255,0.97);
                    border-radius: 8px;
                    padding: 4px 12px;
                    box-shadow: 0 2px 8px rgba(0,0,0,0.08);
                    display: flex;
                    align-items: center;
                    gap: 8px;
                    font-size: 15px;
                    font-weight: 600;
                }
                .property-popup-modern .popup-price-badge .price {
                    color: #222;
                }
                .property-popup-modern .popup-price-badge .purpose {
                    font-size: 12px;
                    font-weight: 500;
                    padding: 2px 7px;
                    border-radius: 6px;
                    background: #e9f5ff;
                    color: #007bff;
                    margin-left: 2px;
                }
                .property-popup-modern .popup-price-badge .purpose.rent {
                    background: #eafaf1;
                    color: #28a745;
                }
                .property-popup-modern .popup-main {
                    padding: 18px 18px 12px 18px;
                }
                .property-popup-modern .popup-header {
                    display: flex;
                    flex-direction: column;
                    gap: 2px;
                    margin-bottom: 8px;
                }
                .property-popup-modern .popup-title {
                    font-size: 1.08rem;
                    font-weight: 600;
                    color: #222;
                    margin: 0 0 2px 0;
                    line-height: 1.2;
                    white-space: nowrap;
                    overflow: hidden;
                    text-overflow: ellipsis;
                }
                .property-popup-modern .popup-location {
                    font-size: 13px;
                    color: #666;
                    display: flex;
                    align-items: center;
                    gap: 4px;
                }
                .property-popup-modern .popup-specs {
                    display: flex;
                    gap: 12px;
                    margin: 10px 0 12px 0;
                }
                .property-popup-modern .spec-chip {
                    background: #f5f7fa;
                    color: #444;
                    border-radius: 6px;
                    padding: 3px 8px;
                    font-size: 13px;
                    display: flex;
                    align-items: center;
                    gap: 4px;
                }
                .property-popup-modern .spec-label {
                    color: #888;
                    font-size: 12px;
                    margin-left: 2px;
                }
                .property-popup-modern .popup-actions {
                    display: flex;
                    gap: 10px;
                    margin-top: 10px;
                }
                .property-popup-modern .btn-modern {
                    display: flex;
                    align-items: center;
                    gap: 5px;
                    border: none;
                    border-radius: 6px;
                    padding: 7px 14px;
                    font-size: 14px;
                    font-weight: 500;
                    cursor: pointer;
                    background: #f5f7fa;
                    color: #222;
                    transition: background 0.15s, color 0.15s;
                    outline: none;
                    text-decoration: none;
                }
                .property-popup-modern .btn-modern svg {
                    display: inline-block;
                    vertical-align: middle;
                }
                .property-popup-modern .btn-modern.btn-primary {
                    background: #007bff;
                    color: #fff;
                }
                .property-popup-modern .btn-modern.btn-primary:hover {
                    background: #0056b3;
                }
                .property-popup-modern .btn-modern.btn-fav {
                    background: #fff0f0;
                    color: #e74c3c;
                }
                .property-popup-modern .btn-modern.btn-fav.active,
                .property-popup-modern .btn-modern.btn-fav:active {
                    background: #e74c3c;
                    color: #fff;
                }
                .property-popup-modern .btn-modern.btn-fav:hover {
                    background: #ffeaea;
                }
                .property-popup-modern .btn-modern.btn-share {
                    background: #e9f5ff;
                    color: #17a2b8;
                }
                .property-popup-modern .btn-modern.btn-share:hover {
                    background: #d0eafd;
                }
            </style>
            <div class="property-popup-modern">
                <div class="popup-img-wrap">
                    <img class="popup-img" src="${image}" alt="${property.title}" loading="lazy">
                    <div class="popup-badges">
                        ${property.is_verified ? `
                            <span class="badge-verified" title="Verified">
                                <svg width="16" height="16" viewBox="0 0 24 24" fill="none">
                                    <circle cx="12" cy="12" r="10" fill="#28a745"/>
                                    <path d="M8 12.5l2.5 2.5 5-5" stroke="#fff" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
                                </svg>
                                Verified
                            </span>
                        ` : ''}
                        ${property.is_featured ? `
                            <span class="badge-featured" title="Featured">
                                <svg width="16" height="16" viewBox="0 0 24 24" fill="none">
                                    <path d="M12 2l2.09 6.26L20 9.27l-5 3.64L16.18 20 12 16.77 7.82 20 9 12.91l-5-3.64 5.91-.91z" fill="#ffc107" stroke="#b8860b" stroke-width="1"/>
                                </svg>
                                Featured
                            </span>
                        ` : ''}
                    </div>
                    <div class="popup-price-badge">
                        <span class="price">${price}</span>
                        <span class="purpose ${property.purpose === 'rent' ? 'rent' : 'sale'}">
                            ${property.purpose === 'rent' ? 'For Rent' : 'For Sale'}
                        </span>
                    </div>
                </div>
                <div class="popup-main">
                    <div class="popup-header">
                        <div class="popup-title" title="${property.title}">${property.title}</div>
                        <span class="popup-location" title="${property.location}">
                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none">
                                <path d="M12 2C8.13 2 5 5.13 5 9c0 5.25 7 13 7 13s7-7.75 7-13c0-3.87-3.13-7-7-7zm0 10a2.5 2.5 0 110-5 2.5 2.5 0 010 5z" fill="#888"/>
                            </svg>
                            ${property.location}
                        </span>
                    </div>
                    <div class="popup-specs">
                        <span class="spec-chip" title="Bedrooms">
                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none">
                                <rect x="2" y="10" width="20" height="10" rx="2" fill="#f5f7fa" stroke="#007bff" stroke-width="2"/>
                                <path d="M4 10V7a4 4 0 018 0v3" stroke="#007bff" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
                            </svg>
                            ${property.bedrooms || 0} <span class="spec-label">Bed</span>
                        </span>
                        <span class="spec-chip" title="Bathrooms">
                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none">
                                <rect x="3" y="10" width="18" height="8" rx="2" fill="#f5f7fa" stroke="#17a2b8" stroke-width="2"/>
                                <path d="M7 10V7a5 5 0 0110 0v3" stroke="#17a2b8" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
                            </svg>
                            ${property.bathrooms || 0} <span class="spec-label">Bath</span>
                        </span>
                        <span class="spec-chip" title="Area">
                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none">
                                <rect x="3" y="7" width="18" height="10" rx="2" fill="#f5f7fa" stroke="#6c757d" stroke-width="2"/>
                                <path d="M3 17l18-10" stroke="#6c757d" stroke-width="2" stroke-linecap="round"/>
                            </svg>
                            ${property.area || 0} <span class="spec-label">sqft</span>
                        </span>
                    </div>
                    <div class="popup-actions">
                        <a class="btn-modern btn-primary view-details-btn" 
                           data-property-id="${property.id}" 
                           href="${property.url || '#'}" target="_blank" rel="noopener">
                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none">
                                <rect x="4" y="4" width="16" height="16" rx="3" stroke="#fff" stroke-width="2"/>
                                <path d="M12 8v4l3 3" stroke="#fff" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
                            </svg>
                            View
                        </a>
                        <button class="btn-modern btn-fav favorite-btn ${isFavorite ? 'active' : ''}" 
                                data-property-id="${property.id}" title="${isFavorite ? 'Remove from favorites' : 'Add to favorites'}">
                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none">
                                <path d="M12 21s-7-5.2-7-10.2A5 5 0 0112 5a5 5 0 017 5.8C19 15.8 12 21 12 21z" fill="${isFavorite ? '#e74c3c' : 'none'}" stroke="#e74c3c" stroke-width="2"/>
                            </svg>
                        </button>
                        <button class="btn-modern btn-share share-btn" 
                                data-property-id="${property.id}" title="Share">
                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none">
                                <path d="M4 12v7a1 1 0 001 1h14a1 1 0 001-1v-7" stroke="#17a2b8" stroke-width="2"/>
                                <path d="M16 6l-4-4-4 4" stroke="#17a2b8" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
                                <path d="M12 2v13" stroke="#17a2b8" stroke-width="2" stroke-linecap="round"/>
                            </svg>
                        </button>
                    </div>
                </div>
            </div>
        `;
    }
    
    /**
     * Handle marker click
     */
    onMarkerClick(property) {
        // Select property in main view
        this.mapView.selectProperty(property);
        
        // Update popup action buttons
        setTimeout(() => {
            this.attachPopupEventListeners(property);
        }, 100);
    }
    
    /**
     * Handle marker hover
     */
    onMarkerHover(property, marker) {
        // Add hover effect to corresponding list item
        const listItem = document.querySelector(`[data-property-id="${property.id}"]`);
        if (listItem) {
            listItem.classList.add('hovered');
        }
        
        // Optional: could show a lightweight tooltip; skipping for Google Maps
    }
    
    /**
     * Handle marker leave
     */
    onMarkerLeave(property, marker) {
        // Remove hover effect from list item
        const listItem = document.querySelector(`[data-property-id="${property.id}"]`);
        if (listItem) {
            listItem.classList.remove('hovered');
        }
        
        // No custom tooltip to hide in Google version
    }
    
    /**
     * Highlight a specific property marker
     */
    highlightProperty(propertyId) {
        // Remove previous highlight
        if (this.highlightedMarker) {
            this.unhighlightMarker(this.highlightedMarker);
        }
        
        // Find and highlight new marker
        const marker = this.propertyMarkers.get(propertyId);
        if (marker) {
            this.highlightMarker(marker);
            this.highlightedMarker = marker;
            
            // Open InfoWindow (Google Maps)
            const property = (this.mapView.filteredProperties || []).find(p => p.id == propertyId)
                || (this.mapView.properties || []).find(p => p.id == propertyId);
            if (property) {
                const popupContent = this.createPopupContent(property);
                this.infoWindow.setContent(popupContent);
                this.infoWindow.open({ map: this.mapView.map, anchor: marker });
                // Attach popup actions
                setTimeout(() => this.attachPopupEventListeners(property), 50);
            }
            
            // Animate to marker
            this.animateToMarker(marker);
        }
    }
    
    /**
     * Highlight a marker
     */
    highlightMarker(marker) {
        // Handle both AdvancedMarkerElement and legacy Marker
        if (marker.setZIndex && typeof marker.setZIndex === 'function') {
            // Legacy marker
            marker.setZIndex(google.maps.Marker.MAX_ZINDEX + 1);
        } else if (marker.content) {
            // AdvancedMarkerElement - add highlight class
            const content = marker.content.querySelector('.marker-content');
            if (content) {
                content.style.background = '#ff6b35';
                content.style.transform = 'scale(1.2)';
                content.style.zIndex = '1001';
            }
        }
    }
    
    /**
     * Remove highlight from marker
     */
    unhighlightMarker(marker) {
        // Handle both AdvancedMarkerElement and legacy Marker
        if (marker.setZIndex) {
            // Legacy marker
            marker.setZIndex(undefined);
        } else if (marker.content) {
            // AdvancedMarkerElement - remove highlight
            const content = marker.content.querySelector('.marker-content');
            if (content) {
                // Get property to restore original color
                const propertyId = Object.keys(this.propertyMarkers).find(key => 
                    this.propertyMarkers.get(key) === marker
                );
                if (propertyId) {
                    const property = this.mapView.filteredProperties?.find(p => p.id == propertyId) ||
                                   this.mapView.properties?.find(p => p.id == propertyId);
                    if (property) {
                        const color = property.purpose === 'rent' ? '#28a745' : '#007bff';
                        content.style.background = color;
                        content.style.transform = 'scale(1)';
                        content.style.zIndex = 'auto';
                    }
                }
            }
        }
    }
    
    /**
     * Get marker position (compatible with both AdvancedMarkerElement and legacy Marker)
     */
    getMarkerPosition(marker) {
        if (marker.position) {
            // AdvancedMarkerElement
            return marker.position;
        } else if (marker.getPosition) {
            // Legacy Marker
            return marker.getPosition();
        }
        return null;
    }
    
    /**
     * Animate to marker with pulse effect
     */
    animateToMarker(marker) {
        // Simple pan; CSS pulse not applicable to Google default markers
        const position = this.getMarkerPosition(marker);
        if (position) {
            this.mapView.map.panTo(position);
        }
    }
    
    /**
     * Show property tooltip
     */
    showPropertyTooltip(property, marker) { /* no-op for Google version */ }
    
    /**
     * Hide property tooltip
     */
    hidePropertyTooltip() { /* no-op for Google version */ }
    
    /**
     * Attach event listeners to popup buttons
     */
    attachPopupEventListeners(property) {
        // View details button
        const viewDetailsBtn = document.querySelector('.view-details-btn[data-property-id="' + property.id + '"]');
        if (viewDetailsBtn) {
            viewDetailsBtn.addEventListener('click', (e) => {
                e.stopPropagation();
                if (this.mapView.propertyDetails) {
                    this.mapView.propertyDetails.showPropertyDetails(property);
                }
            });
        }
        
        // Favorite button
        const favoriteBtn = document.querySelector('.favorite-btn[data-property-id="' + property.id + '"]');
        if (favoriteBtn) {
            favoriteBtn.addEventListener('click', (e) => {
                e.stopPropagation();
                this.toggleFavorite(property.id);
            });
        }
        
        // Share button
        const shareBtn = document.querySelector('.share-btn[data-property-id="' + property.id + '"]');
        if (shareBtn) {
            shareBtn.addEventListener('click', (e) => {
                e.stopPropagation();
                this.shareProperty(property);
            });
        }
    }
    
    /**
     * Toggle property favorite status
     */
    toggleFavorite(propertyId) {
        const favorites = this.mapView.state.favorites;
        const index = favorites.indexOf(propertyId);
        
        if (index > -1) {
            favorites.splice(index, 1);
        } else {
            favorites.push(propertyId);
        }
        
        // Update localStorage
        localStorage.setItem('property_favorites', JSON.stringify(favorites));
        
        // Update UI
        this.updateFavoriteButtons(propertyId);
        this.mapView.showNotification(
            index > -1 ? 'Removed from favorites' : 'Added to favorites',
            'success'
        );
    }
    
    /**
     * Update favorite button states
     */
    updateFavoriteButtons(propertyId) {
        const isFavorite = this.mapView.state.favorites.includes(propertyId);
        const buttons = document.querySelectorAll(`[data-property-id="${propertyId}"].favorite-btn`);
        
        buttons.forEach(btn => {
            const icon = btn.querySelector('i');
            if (isFavorite) {
                btn.classList.add('active');
                icon.className = 'fa fa-heart';
            } else {
                btn.classList.remove('active');
                icon.className = 'fa fa-heart-o';
            }
        });
    }
    
    /**
     * Share property
     */
    shareProperty(property) {
        if (navigator.share) {
            navigator.share({
                title: property.title,
                text: `Check out this property: ${property.title} in ${property.location}`,
                url: window.location.href + `?property=${property.id}`
            });
        } else {
            // Fallback: copy to clipboard
            const url = window.location.href + `?property=${property.id}`;
            navigator.clipboard.writeText(url).then(() => {
                this.mapView.showNotification('Property link copied to clipboard', 'success');
            });
        }
    }
    
    /**
     * Update heatmap layer
     */
    updateHeatmap(properties) {
        // Remove existing heatmap
        if (this.heatmapLayer) {
            this.heatmapLayer.setMap(null);
            this.heatmapLayer = null;
        }
        
        // Create heatmap data for Google Maps
        const heatmapData = properties
            .filter(p => p.latitude && p.longitude)
            .map(p => ({ location: new google.maps.LatLng(p.latitude, p.longitude), weight: this.getHeatmapIntensity(p) }));
        
        if (heatmapData.length > 0 && google.maps.visualization) {
            this.heatmapLayer = new google.maps.visualization.HeatmapLayer({
                data: heatmapData,
                radius: 25
            });
            this.heatmapLayer.setMap(this.mapView.map);
        }
    }
    
    /**
     * Get heatmap intensity for property
     */
    getHeatmapIntensity(property) {
        let intensity = 0.5;
        
        // Increase intensity for higher prices
        if (property.price > 5000000) intensity += 0.3;
        else if (property.price > 2000000) intensity += 0.2;
        else if (property.price > 1000000) intensity += 0.1;
        
        // Increase intensity for verified properties
        if (property.is_verified) intensity += 0.1;
        
        // Increase intensity for featured properties
        if (property.is_featured) intensity += 0.2;
        
        return Math.min(intensity, 1.0);
    }
    
    /**
     * Toggle heatmap visibility
     */
    toggleHeatmap() {
        if (!this.heatmapLayer) return;
        
        if (!this.heatmapLayer) return false;
        const visible = this.heatmapLayer.getMap() != null;
        this.heatmapLayer.setMap(visible ? null : this.mapView.map);
        return !visible;
    }
    
    /**
     * Get marker bounds for fitting map view
     */
    getMarkerBounds() {
        const bounds = new google.maps.LatLngBounds();
        this.propertyMarkers.forEach(marker => {
            const position = this.getMarkerPosition(marker);
            if (position) {
                bounds.extend(position);
            }
        });
        return bounds;
    }
    
    /**
     * Fit map to show all markers
     */
    fitToMarkers() {
        if (this.propertyMarkers.size > 0) {
            const bounds = this.getMarkerBounds();
            if (!bounds.isEmpty()) {
                this.mapView.map.fitBounds(bounds, { padding: 20 });
            }
        }
    }
    
    /**
     * Clear all markers
     */
    clearMarkers() {
        if (Array.isArray(this.mapView.markers)) {
            this.mapView.markers.forEach(m => m.setMap(null));
            this.mapView.markers = [];
        }
        this.propertyMarkers.clear();
        this.highlightedMarker = null;
        try { this.infoWindow?.close?.(); } catch (_) {}
        
        if (this.heatmapLayer) {
            this.heatmapLayer.setMap(null);
            this.heatmapLayer = null;
        }
        // Clear clusterer if present
        if (this.clusterer) {
            try { this.clusterer.clearMarkers?.(); } catch (_) {}
            try { this.clusterer.setMap?.(null); } catch (_) {}
            this.clusterer = null;
        }
    }
}

// Export for use in other modules
window.MapMarkers = MapMarkers;
