/**
 * MapUtils Module
 * Utility functions and helpers for the map view
 */
class MapUtils {
    constructor(mapView) {
        this.mapView = mapView;
    }
    
    /**
     * Calculate distance between two points using Haversine formula
     */
    calculateDistance(lat1, lng1, lat2, lng2) {
        const R = 6371; // Earth's radius in kilometers
        const dLat = this.toRadians(lat2 - lat1);
        const dLng = this.toRadians(lng2 - lng1);
        
        const a = Math.sin(dLat / 2) * Math.sin(dLat / 2) +
                  Math.cos(this.toRadians(lat1)) * Math.cos(this.toRadians(lat2)) *
                  Math.sin(dLng / 2) * Math.sin(dLng / 2);
        
        const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));
        return R * c;
    }
    
    /**
     * Convert degrees to radians
     */
    toRadians(degrees) {
        return degrees * (Math.PI / 180);
    }
    
    /**
     * Format distance for display
     */
    formatDistance(distance) {
        if (distance < 1) {
            return `${Math.round(distance * 1000)}m`;
        } else {
            return `${distance.toFixed(1)}km`;
        }
    }
    
    /**
     * Get bounds that include all given points with padding
     */
    getBoundsForPoints(points, padding = 0.01) {
        if (!points || points.length === 0) return null;
        
        let minLat = points[0].lat;
        let maxLat = points[0].lat;
        let minLng = points[0].lng;
        let maxLng = points[0].lng;
        
        points.forEach(point => {
            minLat = Math.min(minLat, point.lat);
            maxLat = Math.max(maxLat, point.lat);
            minLng = Math.min(minLng, point.lng);
            maxLng = Math.max(maxLng, point.lng);
        });
        
        return {
            south: minLat - padding,
            west: minLng - padding,
            north: maxLat + padding,
            east: maxLng + padding
        };
    }
    
    /**
     * Debounce function calls
     */
    debounce(func, wait, immediate) {
        let timeout;
        return function executedFunction(...args) {
            const later = () => {
                timeout = null;
                if (!immediate) func(...args);
            };
            const callNow = immediate && !timeout;
            clearTimeout(timeout);
            timeout = setTimeout(later, wait);
            if (callNow) func(...args);
        };
    }
    
    /**
     * Throttle function calls
     */
    throttle(func, limit) {
        let inThrottle;
        return function(...args) {
            if (!inThrottle) {
                func.apply(this, args);
                inThrottle = true;
                setTimeout(() => inThrottle = false, limit);
            }
        };
    }
    
    /**
     * Generate unique ID
     */
    generateId() {
        return Date.now().toString(36) + Math.random().toString(36).substr(2);
    }
    
    /**
     * Format number with Indian numbering system
     */
    formatIndianNumber(num) {
        if (num >= 10000000) {
            return (num / 10000000).toFixed(1) + ' Cr';
        } else if (num >= 100000) {
            return (num / 100000).toFixed(1) + ' L';
        } else if (num >= 1000) {
            return (num / 1000).toFixed(1) + 'K';
        }
        return num.toString();
    }
    
    /**
     * Parse search query for advanced filtering
     */
    parseSearchQuery(query) {
        const filters = {};
        const terms = [];
        
        // Extract price ranges (e.g., "under 50L", "above 1Cr", "between 25L and 50L")
        const pricePatterns = [
            /under\s+(\d+(?:\.\d+)?)\s*(cr|crore|l|lakh|k|thousand)/i,
            /above\s+(\d+(?:\.\d+)?)\s*(cr|crore|l|lakh|k|thousand)/i,
            /between\s+(\d+(?:\.\d+)?)\s*(cr|crore|l|lakh|k|thousand)\s+and\s+(\d+(?:\.\d+)?)\s*(cr|crore|l|lakh|k|thousand)/i
        ];
        
        pricePatterns.forEach(pattern => {
            const match = query.match(pattern);
            if (match) {
                const value1 = this.parseIndianNumber(match[1], match[2]);
                if (match[0].includes('under')) {
                    filters.price_max = value1;
                } else if (match[0].includes('above')) {
                    filters.price_min = value1;
                } else if (match[0].includes('between')) {
                    const value2 = this.parseIndianNumber(match[3], match[4]);
                    filters.price_min = Math.min(value1, value2);
                    filters.price_max = Math.max(value1, value2);
                }
                query = query.replace(match[0], '').trim();
            }
        });
        
        // Extract bedroom/bathroom counts
        const roomPatterns = [
            /(\d+)\s*(?:bed|bedroom|bhk)/i,
            /(\d+)\s*(?:bath|bathroom)/i
        ];
        
        roomPatterns.forEach((pattern, index) => {
            const match = query.match(pattern);
            if (match) {
                if (index === 0) {
                    filters.bedrooms = parseInt(match[1]);
                } else {
                    filters.bathrooms = parseInt(match[1]);
                }
                query = query.replace(match[0], '').trim();
            }
        });
        
        // Extract property types
        const typeKeywords = {
            'apartment': 'apartment',
            'flat': 'apartment',
            'house': 'house',
            'villa': 'villa',
            'plot': 'plot',
            'commercial': 'commercial',
            'office': 'commercial'
        };
        
        Object.keys(typeKeywords).forEach(keyword => {
            if (query.toLowerCase().includes(keyword)) {
                filters.type_keyword = typeKeywords[keyword];
                query = query.replace(new RegExp(keyword, 'gi'), '').trim();
            }
        });
        
        // Remaining query becomes search terms
        if (query.trim()) {
            terms.push(query.trim());
        }
        
        return { filters, terms: terms.join(' ') };
    }
    
    /**
     * Parse Indian number format to actual number
     */
    parseIndianNumber(value, unit) {
        const num = parseFloat(value);
        const unitLower = unit.toLowerCase();
        
        switch (unitLower) {
            case 'cr':
            case 'crore':
                return num * 10000000;
            case 'l':
            case 'lakh':
                return num * 100000;
            case 'k':
            case 'thousand':
                return num * 1000;
            default:
                return num;
        }
    }
    
    /**
     * Get color for property based on price range
     */
    getPropertyColor(property) {
        const price = property.price || 0;
        
        if (price < 1000000) return '#28a745'; // Green for low price
        if (price < 5000000) return '#ffc107'; // Yellow for medium price
        if (price < 10000000) return '#fd7e14'; // Orange for high price
        return '#dc3545'; // Red for very high price
    }
    
    /**
     * Generate property summary for sharing
     */
    generatePropertySummary(property) {
        const price = this.mapView.formatPrice(property.price);
        const specs = [];
        
        if (property.bedrooms) specs.push(`${property.bedrooms} bed`);
        if (property.bathrooms) specs.push(`${property.bathrooms} bath`);
        if (property.area) specs.push(`${property.area} sq ft`);
        
        return `${property.title} - ${price}\n${property.location}\n${specs.join(' • ')}`;
    }
    
    /**
     * Validate coordinates
     */
    isValidCoordinate(lat, lng) {
        return !isNaN(lat) && !isNaN(lng) && 
               lat >= -90 && lat <= 90 && 
               lng >= -180 && lng <= 180;
    }
    
    /**
     * Get map center from properties
     */
    getCenterFromProperties(properties) {
        if (!properties || properties.length === 0) {
            return this.mapView.options.defaultCenter;
        }
        
        const validProperties = properties.filter(p => 
            this.isValidCoordinate(p.latitude, p.longitude)
        );
        
        if (validProperties.length === 0) {
            return this.mapView.options.defaultCenter;
        }
        
        const avgLat = validProperties.reduce((sum, p) => sum + p.latitude, 0) / validProperties.length;
        const avgLng = validProperties.reduce((sum, p) => sum + p.longitude, 0) / validProperties.length;
        
        return [avgLat, avgLng];
    }
    
    /**
     * Get appropriate zoom level based on property spread
     */
    getZoomFromProperties(properties) {
        if (!properties || properties.length === 0) {
            return this.mapView.options.defaultZoom;
        }
        
        const validProperties = properties.filter(p => 
            this.isValidCoordinate(p.latitude, p.longitude)
        );
        
        if (validProperties.length === 0) {
            return this.mapView.options.defaultZoom;
        }
        
        if (validProperties.length === 1) {
            return 14; // Close zoom for single property
        }
        
        // Calculate bounds and determine appropriate zoom
        const bounds = this.getBoundsForPoints(
            validProperties.map(p => ({ lat: p.latitude, lng: p.longitude }))
        );
        
        if (!bounds) return this.mapView.options.defaultZoom;
        
        const latDiff = bounds.north - bounds.south;
        const lngDiff = bounds.east - bounds.west;
        const maxDiff = Math.max(latDiff, lngDiff);
        
        if (maxDiff > 10) return 6;
        if (maxDiff > 5) return 8;
        if (maxDiff > 2) return 10;
        if (maxDiff > 1) return 12;
        if (maxDiff > 0.5) return 13;
        return 14;
    }
    
    /**
     * Create custom marker HTML
     */
    createMarkerHTML(property, options = {}) {
        const price = this.formatIndianNumber(property.price);
        const color = options.color || this.getPropertyColor(property);
        const size = options.size || 'medium';
        
        return `
            <div class="custom-property-marker ${size}" style="border-color: ${color};">
                <div class="marker-icon">
                    <i class="fa fa-${property.purpose === 'rent' ? 'key' : 'home'}"></i>
                </div>
                <div class="marker-price" style="background-color: ${color};">
                    ₹${price}
                </div>
                ${property.is_verified ? '<div class="verified-badge"><i class="fa fa-shield"></i></div>' : ''}
            </div>
        `;
    }
    
    /**
     * Animate element with CSS classes
     */
    animateElement(element, animationClass, duration = 1000) {
        return new Promise(resolve => {
            element.classList.add(animationClass);
            setTimeout(() => {
                element.classList.remove(animationClass);
                resolve();
            }, duration);
        });
    }
    
    /**
     * Load image with fallback
     */
    loadImageWithFallback(src, fallbackSrc) {
        return new Promise(resolve => {
            const img = new Image();
            img.onload = () => resolve(src);
            img.onerror = () => resolve(fallbackSrc);
            img.src = src;
        });
    }
    
    /**
     * Check if device is mobile
     */
    isMobile() {
        return window.innerWidth <= 768 || /Android|iPhone|iPad|iPod|BlackBerry|IEMobile|Opera Mini/i.test(navigator.userAgent);
    }
    
    /**
     * Check if device supports touch
     */
    isTouchDevice() {
        return 'ontouchstart' in window || navigator.maxTouchPoints > 0;
    }
    
    /**
     * Get device pixel ratio
     */
    getPixelRatio() {
        return window.devicePixelRatio || 1;
    }
    
    /**
     * Convert RGB to Hex
     */
    rgbToHex(r, g, b) {
        return "#" + ((1 << 24) + (r << 16) + (g << 8) + b).toString(16).slice(1);
    }
    
    /**
     * Convert Hex to RGB
     */
    hexToRgb(hex) {
        const result = /^#?([a-f\d]{2})([a-f\d]{2})([a-f\d]{2})$/i.exec(hex);
        return result ? {
            r: parseInt(result[1], 16),
            g: parseInt(result[2], 16),
            b: parseInt(result[3], 16)
        } : null;
    }
    
    /**
     * Generate contrasting color
     */
    getContrastColor(hexColor) {
        const rgb = this.hexToRgb(hexColor);
        if (!rgb) return '#000000';
        
        const brightness = (rgb.r * 299 + rgb.g * 587 + rgb.b * 114) / 1000;
        return brightness > 128 ? '#000000' : '#ffffff';
    }
    
    /**
     * Save data to localStorage with expiry
     */
    saveToStorage(key, data, expiryHours = 24) {
        const item = {
            data: data,
            expiry: Date.now() + (expiryHours * 60 * 60 * 1000)
        };
        localStorage.setItem(key, JSON.stringify(item));
    }
    
    /**
     * Load data from localStorage with expiry check
     */
    loadFromStorage(key) {
        const itemStr = localStorage.getItem(key);
        if (!itemStr) return null;
        
        try {
            const item = JSON.parse(itemStr);
            if (Date.now() > item.expiry) {
                localStorage.removeItem(key);
                return null;
            }
            return item.data;
        } catch (e) {
            localStorage.removeItem(key);
            return null;
        }
    }
    
    /**
     * Deep clone object
     */
    deepClone(obj) {
        if (obj === null || typeof obj !== 'object') return obj;
        if (obj instanceof Date) return new Date(obj.getTime());
        if (obj instanceof Array) return obj.map(item => this.deepClone(item));
        if (typeof obj === 'object') {
            const clonedObj = {};
            for (const key in obj) {
                if (obj.hasOwnProperty(key)) {
                    clonedObj[key] = this.deepClone(obj[key]);
                }
            }
            return clonedObj;
        }
    }
    
    /**
     * Merge objects deeply
     */
    deepMerge(target, source) {
        const result = this.deepClone(target);
        
        for (const key in source) {
            if (source.hasOwnProperty(key)) {
                if (source[key] && typeof source[key] === 'object' && !Array.isArray(source[key])) {
                    result[key] = this.deepMerge(result[key] || {}, source[key]);
                } else {
                    result[key] = source[key];
                }
            }
        }
        
        return result;
    }
    
    /**
     * Format time ago
     */
    timeAgo(date) {
        const now = new Date();
        const diffInSeconds = Math.floor((now - new Date(date)) / 1000);
        
        const intervals = {
            year: 31536000,
            month: 2592000,
            week: 604800,
            day: 86400,
            hour: 3600,
            minute: 60
        };
        
        for (const [unit, seconds] of Object.entries(intervals)) {
            const interval = Math.floor(diffInSeconds / seconds);
            if (interval >= 1) {
                return `${interval} ${unit}${interval > 1 ? 's' : ''} ago`;
            }
        }
        
        return 'Just now';
    }
    
    /**
     * Escape HTML
     */
    escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }
    
    /**
     * Generate SEO-friendly slug
     */
    generateSlug(text) {
        return text
            .toLowerCase()
            .replace(/[^\w\s-]/g, '')
            .replace(/[\s_-]+/g, '-')
            .replace(/^-+|-+$/g, '');
    }
}

// Export for use in other modules
window.MapUtils = MapUtils;
