/**
 * PropertyDetails Module
 * Handles property detail display and interactions
 */
class PropertyDetails {
    constructor(mapView) {
        this.mapView = mapView;
        this.currentProperty = null;
        
        this.init();
    }
    
    /**
     * Initialize property details functionality
     */
    init() {
        this.initializeDrawer();
        this.initializeModals();
        this.initializeEventListeners();
    }
    
    /**
     * Initialize property drawer
     */
    initializeDrawer() {
        const drawer = document.getElementById('property-drawer');
        if (!drawer) return;
        
        // Close drawer button
        const closeBtn = document.getElementById('close-drawer');
        if (closeBtn) {
            closeBtn.addEventListener('click', () => {
                this.hidePropertyDetails();
            });
        }
        
        // Drawer action buttons
        const favoriteBtn = document.getElementById('drawer-favorite');
        const shareBtn = document.getElementById('drawer-share');
        const compareBtn = document.getElementById('drawer-compare');
        
        if (favoriteBtn) {
            favoriteBtn.addEventListener('click', () => {
                if (this.currentProperty) {
                    this.toggleFavorite(this.currentProperty.id);
                }
            });
        }
        
        if (shareBtn) {
            shareBtn.addEventListener('click', () => {
                if (this.currentProperty) {
                    this.shareProperty(this.currentProperty);
                }
            });
        }
        
        if (compareBtn) {
            compareBtn.addEventListener('click', () => {
                if (this.currentProperty) {
                    this.addToCompare(this.currentProperty.id);
                }
            });
        }
    }
    
    /**
     * Initialize modals
     */
    initializeModals() {
        // Contact agent form
        const contactForm = document.getElementById('contact-agent-form');
        if (contactForm) {
            contactForm.addEventListener('submit', (e) => {
                e.preventDefault();
                this.submitContactForm();
            });
        }
        
        // Contact agent button in modal
        const contactAgentBtn = document.getElementById('contact-agent-btn');
        if (contactAgentBtn) {
            contactAgentBtn.addEventListener('click', () => {
                this.showContactAgentModal();
            });
        }
        
        // Export comparison button
        const exportComparisonBtn = document.getElementById('export-comparison');
        if (exportComparisonBtn) {
            exportComparisonBtn.addEventListener('click', () => {
                this.exportComparison();
            });
        }
    }
    
    /**
     * Initialize event listeners
     */
    initializeEventListeners() {
        // Property list item clicks
        document.addEventListener('click', (e) => {
            const listItem = e.target.closest('.property-list-item');
            if (listItem) {
                const propertyId = listItem.dataset.propertyId;
                const property = this.mapView.filteredProperties.find(p => p.id == propertyId);
                if (property) {
                    this.showPropertyDetails(property);
                }
            }
        });
        
        // Favorite button clicks
        document.addEventListener('click', (e) => {
            if (e.target.closest('.favorite-btn')) {
                const btn = e.target.closest('.favorite-btn');
                const propertyId = btn.dataset.propertyId;
                this.toggleFavorite(propertyId);
            }
        });
        
        // Share button clicks
        document.addEventListener('click', (e) => {
            if (e.target.closest('.share-btn')) {
                const btn = e.target.closest('.share-btn');
                const propertyId = btn.dataset.propertyId;
                const property = this.mapView.filteredProperties.find(p => p.id == propertyId);
                if (property) {
                    this.shareProperty(property);
                }
            }
        });
    }
    
    /**
     * Show property details in drawer
     */
    showPropertyDetails(property) {
        this.currentProperty = property;
        
        const drawer = document.getElementById('property-drawer');
        const drawerContent = document.getElementById('drawer-content');
        const drawerTitle = document.getElementById('drawer-title');
        
        if (!drawer || !drawerContent) return;
        
        // Update drawer title
        if (drawerTitle) {
            drawerTitle.textContent = property.title;
        }
        
        // Show loading state
        drawerContent.innerHTML = `
            <div class="loading-state">
                <div class="spinner"></div>
                <p>Loading property details...</p>
            </div>
        `;
        
        // Open drawer
        drawer.classList.add('open');
        
        // Load detailed property information
        this.loadPropertyDetails(property.id).then(detailedProperty => {
            this.renderPropertyDetails(detailedProperty || property);
        }).catch(error => {
            console.error('Error loading property details:', error);
            this.renderPropertyDetails(property);
        });
        
        // Update favorite button state
        this.updateDrawerFavoriteButton(property.id);
    }
    
    /**
     * Hide property details drawer
     */
    hidePropertyDetails() {
        const drawer = document.getElementById('property-drawer');
        if (drawer) {
            drawer.classList.remove('open');
        }
        
        this.currentProperty = null;
        
        // Clear highlighted marker
        if (this.mapView.markersManager && this.mapView.markersManager.highlightedMarker) {
            this.mapView.markersManager.unhighlightMarker(this.mapView.markersManager.highlightedMarker);
            this.mapView.markersManager.highlightedMarker = null;
        }
    }
    
    /**
     * Load detailed property information
     */
    async loadPropertyDetails(propertyId) {
        try {
            const response = await fetch(`/api/properties/${propertyId}/details`);
            if (response.ok) {
                const data = await response.json();
                return data.property;
            }
        } catch (error) {
            console.error('Error fetching property details:', error);
        }
        return null;
    }
    
  
    
    /**
     * Initialize image gallery
     */
    initializeImageGallery(images) {
        let currentImageIndex = 0;
        
        window.selectImage = (index) => {
            currentImageIndex = index;
            document.getElementById('main-property-image').src = images[index];
            document.getElementById('current-image').textContent = index + 1;
            
            // Update thumbnail active state
            document.querySelectorAll('.thumbnail').forEach((thumb, i) => {
                thumb.classList.toggle('active', i === index);
            });
        };
        
        window.previousImage = () => {
            currentImageIndex = (currentImageIndex - 1 + images.length) % images.length;
            window.selectImage(currentImageIndex);
        };
        
        window.nextImage = () => {
            currentImageIndex = (currentImageIndex + 1) % images.length;
            window.selectImage(currentImageIndex);
        };
        
        // Keyboard navigation
        document.addEventListener('keydown', (e) => {
            if (document.getElementById('property-drawer').classList.contains('open')) {
                if (e.key === 'ArrowLeft') {
                    window.previousImage();
                } else if (e.key === 'ArrowRight') {
                    window.nextImage();
                }
            }
        });
    }
    
    /**
     * Attach event listeners to action buttons
     */
    attachActionListeners(property) {
        // Contact agent button
        const contactBtn = document.querySelector('.contact-agent-btn');
        if (contactBtn) {
            contactBtn.addEventListener('click', () => {
                this.showContactAgentModal(property);
            });
        }
        
        // Schedule visit button
        const scheduleBtn = document.querySelector('.schedule-btn');
        if (scheduleBtn) {
            scheduleBtn.addEventListener('click', () => {
                this.scheduleVisit(property);
            });
        }
        
        // Virtual tour button
        const virtualTourBtn = document.querySelector('.virtual-tour-btn');
        if (virtualTourBtn) {
            virtualTourBtn.addEventListener('click', () => {
                this.startVirtualTour(property);
            });
        }
        
        // Report issue button
        const reportBtn = document.querySelector('.report-btn');
        if (reportBtn) {
            reportBtn.addEventListener('click', () => {
                this.reportIssue(property);
            });
        }
        
        // Directions button
        window.getDirections = (lat, lng) => {
            this.getDirections(lat, lng);
        };
    }
    
    /**
     * Show contact agent modal
     */
    showContactAgentModal(property = this.currentProperty) {
        const modal = document.getElementById('contact-agent-modal');
        const agentInfo = document.getElementById('agent-info');
        
        if (!modal || !property) return;
        
        // Populate agent information
        if (agentInfo && property.agent) {
            agentInfo.innerHTML = `
                <div class="agent-card">
                    <img src="${property.agent.avatar || '/assets/images/agent-placeholder.jpg'}" 
                         alt="${property.agent.name}" class="agent-avatar">
                    <div class="agent-details">
                        <h6>${property.agent.name}</h6>
                        <p>${property.agent.company || 'Real Estate Agent'}</p>
                        <div class="agent-rating">
                            ${this.generateStarRating(property.agent.rating || 4.5)}
                            <span>(${property.agent.reviews || 0} reviews)</span>
                        </div>
                    </div>
                </div>
            `;
        }
        
        // Pre-fill message
        const messageField = document.getElementById('contact-message');
        if (messageField) {
            messageField.value = `Hi, I'm interested in the property "${property.title}" located at ${property.location}. Could you please provide more information?`;
        }
        
        // Show modal
        $(modal).modal('show');
    }
    
    /**
     * Submit contact form
     */
    async submitContactForm() {
        const form = document.getElementById('contact-agent-form');
        if (!form || !this.currentProperty) return;
        
        const formData = new FormData(form);
        const data = {
            property_id: this.currentProperty.id,
            name: document.getElementById('contact-name').value,
            email: document.getElementById('contact-email').value,
            phone: document.getElementById('contact-phone').value,
            message: document.getElementById('contact-message').value
        };
        
        try {
            const response = await fetch('/api/contact-agent', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content')
                },
                body: JSON.stringify(data)
            });
            
            if (response.ok) {
                this.mapView.showNotification('Message sent successfully!', 'success');
                $('#contact-agent-modal').modal('hide');
                form.reset();
            } else {
                this.mapView.showNotification('Failed to send message. Please try again.', 'error');
            }
        } catch (error) {
            console.error('Error sending message:', error);
            this.mapView.showNotification('Failed to send message. Please try again.', 'error');
        }
    }
    
    /**
     * Toggle favorite status
     */
    toggleFavorite(propertyId) {
        const favorites = this.mapView.state.favorites;
        const index = favorites.indexOf(propertyId);
        
        if (index > -1) {
            favorites.splice(index, 1);
        } else {
            favorites.push(propertyId);
        }
        
        // Update localStorage
        localStorage.setItem('property_favorites', JSON.stringify(favorites));
        
        // Update UI
        this.updateFavoriteButtons(propertyId);
        this.updateDrawerFavoriteButton(propertyId);
        
        this.mapView.showNotification(
            index > -1 ? 'Removed from favorites' : 'Added to favorites',
            'success'
        );
    }
    
    /**
     * Update favorite button states
     */
    updateFavoriteButtons(propertyId) {
        const isFavorite = this.mapView.state.favorites.includes(propertyId);
        const buttons = document.querySelectorAll(`[data-property-id="${propertyId}"].favorite-btn`);
        
        buttons.forEach(btn => {
            const icon = btn.querySelector('i');
            if (isFavorite) {
                btn.classList.add('active');
                if (icon) icon.className = 'fa fa-heart';
            } else {
                btn.classList.remove('active');
                if (icon) icon.className = 'fa fa-heart-o';
            }
        });
    }
    
    /**
     * Update drawer favorite button
     */
    updateDrawerFavoriteButton(propertyId) {
        const drawerFavoriteBtn = document.getElementById('drawer-favorite');
        if (!drawerFavoriteBtn) return;
        
        const isFavorite = this.mapView.state.favorites.includes(propertyId);
        const icon = drawerFavoriteBtn.querySelector('i');
        
        if (isFavorite) {
            drawerFavoriteBtn.classList.add('active');
            drawerFavoriteBtn.title = 'Remove from Favorites';
            if (icon) icon.className = 'fa fa-heart';
        } else {
            drawerFavoriteBtn.classList.remove('active');
            drawerFavoriteBtn.title = 'Add to Favorites';
            if (icon) icon.className = 'fa fa-heart-o';
        }
    }
    
    /**
     * Share property
     */
    shareProperty(property) {
        const shareData = {
            title: property.title,
            text: `Check out this property: ${property.title} in ${property.location}`,
            url: `${window.location.origin}/property/${property.id}`
        };
        
        if (navigator.share) {
            navigator.share(shareData);
        } else {
            // Fallback: copy to clipboard
            navigator.clipboard.writeText(shareData.url).then(() => {
                this.mapView.showNotification('Property link copied to clipboard', 'success');
            });
        }
    }
    
    /**
     * Add property to comparison
     */
    addToCompare(propertyId) {
        const compareList = this.mapView.state.compareList;
        
        if (compareList.includes(propertyId)) {
            this.mapView.showNotification('Property already in comparison', 'warning');
            return;
        }
        
        if (compareList.length >= 4) {
            this.mapView.showNotification('Maximum 4 properties can be compared', 'warning');
            return;
        }
        
        compareList.push(propertyId);
        localStorage.setItem('property_compare', JSON.stringify(compareList));
        
        this.mapView.showNotification('Added to comparison', 'success');
        this.updateCompareButton();
    }
    
    /**
     * Update compare button
     */
    updateCompareButton() {
        const compareBtn = document.getElementById('compare-favorites');
        if (compareBtn) {
            const count = this.mapView.state.compareList.length;
            compareBtn.disabled = count < 2;
            compareBtn.textContent = `Compare Selected (${count})`;
        }
    }
    
    /**
     * Schedule visit
     */
    scheduleVisit(property) {
        // This would typically open a scheduling modal or redirect to a booking system
        this.mapView.showNotification('Visit scheduling feature coming soon!', 'info');
    }
    
    /**
     * Start virtual tour
     */
    startVirtualTour(property) {
        // This would typically open a virtual tour viewer
        this.mapView.showNotification('Virtual tour feature coming soon!', 'info');
    }
    
    /**
     * Report issue
     */
    reportIssue(property) {
        // This would typically open a report form
        this.mapView.showNotification('Issue reporting feature coming soon!', 'info');
    }
    
    /**
     * Get directions to property
     */
    getDirections(lat, lng) {
        if (navigator.geolocation) {
            navigator.geolocation.getCurrentPosition((position) => {
                const userLat = position.coords.latitude;
                const userLng = position.coords.longitude;
                
                const url = `https://www.google.com/maps/dir/${userLat},${userLng}/${lat},${lng}`;
                window.open(url, '_blank');
            });
        } else {
            const url = `https://www.google.com/maps/place/${lat},${lng}`;
            window.open(url, '_blank');
        }
    }
    
    /**
     * Generate star rating HTML
     */
    generateStarRating(rating) {
        const fullStars = Math.floor(rating);
        const hasHalfStar = rating % 1 !== 0;
        const emptyStars = 5 - fullStars - (hasHalfStar ? 1 : 0);
        
        let html = '';
        
        for (let i = 0; i < fullStars; i++) {
            html += '<i class="fa fa-star"></i>';
        }
        
        if (hasHalfStar) {
            html += '<i class="fa fa-star-half-o"></i>';
        }
        
        for (let i = 0; i < emptyStars; i++) {
            html += '<i class="fa fa-star-o"></i>';
        }
        
        return html;
    }
    
    /**
     * Export comparison
     */
    exportComparison() {
        // This would generate a PDF or Excel file with property comparison
        this.mapView.showNotification('Export feature coming soon!', 'info');
    }
}

// Export for use in other modules
window.PropertyDetails = PropertyDetails;
