<?php

namespace App\Factories;

use App\Contracts\OTPServiceInterface;
use App\Services\TwilioService;
use App\Services\Fast2SMSService;

class OTPServiceFactory
{
    /**
     * Create the appropriate OTP service based on configuration
     *
     * @return OTPServiceInterface
     * @throws \Exception
     */
    public static function create(): OTPServiceInterface
    {
        $provider = strtolower(env('OTP_PROVIDER', 'twilio'));

        return match ($provider) {
            'twilio' => new TwilioService(),
            'fast2sms' => new Fast2SMSService(),
            default => throw new \Exception("Unsupported OTP provider: {$provider}. Supported providers: twilio, fast2sms")
        };
    }

    /**
     * Get available OTP providers
     *
     * @return array
     */
    public static function getAvailableProviders(): array
    {
        return [
            'twilio' => 'Twilio',
            'fast2sms' => 'Fast2SMS'
        ];
    }

    /**
     * Check if the current provider is properly configured
     *
     * @return bool
     */
    public static function isConfigured(): bool
    {
        $provider = strtolower(env('OTP_PROVIDER', 'twilio'));

        return match ($provider) {
            'twilio' => !empty(env('TWILIO_SID')) && !empty(env('TWILIO_TOKEN')) && !empty(env('TWILIO_FROM_NUMBER')),
            'fast2sms' => !empty(env('FAST2SMS_API_KEY')),
            default => false
        };
    }

    /**
     * Get the current provider name
     *
     * @return string
     */
    public static function getCurrentProvider(): string
    {
        return strtolower(env('OTP_PROVIDER', 'twilio'));
    }
}
