<?php

namespace App\Http\Controllers\Admin;

use App\CustomPlanRequest;
use App\SubscriptionPlan;
use App\Transactions;
use App\User;
use App\Property;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\URL;

class CustomPlanAdminController extends MainAdminController
{
    public function index()
    {
        if (Auth::User()->usertype != "Admin") {
            Session::flash('flash_message', trans('words.access_denied'));
            return redirect('dashboard');
        }

        $page_title = 'Custom Plan Requests';
        $requests = CustomPlanRequest::with(['user', 'plan'])
            ->orderByDesc('id')
            ->paginate(15);

        return view('admin.pages.custom_plans.index', compact('page_title', 'requests'));
    }

    public function show($id)
    {
        if (Auth::User()->usertype != "Admin") {
            Session::flash('flash_message', trans('words.access_denied'));
            return redirect('dashboard');
        }

        $page_title = 'Review Custom Plan Request';
        $requestModel = CustomPlanRequest::with(['user', 'plan'])->findOrFail($id);

        return view('admin.pages.custom_plans.show', compact('page_title', 'requestModel'));
    }

    public function approve($id, Request $request)
    {
        if (Auth::User()->usertype != "Admin") {
            Session::flash('flash_message', trans('words.access_denied'));
            return redirect('dashboard');
        }

        $data = $request->all();
        $rules = [
            'approved_property_limit' => 'required|integer|min:0',
            'approved_ads_limit' => 'nullable|integer|min:0',
            'approved_price' => 'required|numeric|min:0',
            'approved_validity_days' => 'required|integer|min:1',
            'admin_notes' => 'nullable|string|max:2000',
        ];
        $validator = \Validator::make($data, $rules);
        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator->messages())->withInput();
        }

        $cpr = CustomPlanRequest::findOrFail($id);

        // Create a custom subscription plan row
        $plan = new SubscriptionPlan();
        $plan->plan_name = 'Custom Plan - ' . $cpr->user->name . ' #' . $cpr->id;
        $plan->plan_duration = (int)$data['approved_validity_days'];
        $plan->plan_duration_type = 1; // days
        $plan->plan_days = (int)$data['approved_validity_days'];
        $plan->plan_price = (float)$data['approved_price'];
        $plan->plan_property_limit = (int)$data['approved_property_limit'];
        $plan->plan_ads_limit = isset($data['approved_ads_limit']) ? (int)$data['approved_ads_limit'] : null;
        $plan->status = 1;
        $plan->is_custom = 1;
        $plan->save();

        // Update request
        $cpr->approved_property_limit = (int)$data['approved_property_limit'];
        $cpr->approved_ads_limit = isset($data['approved_ads_limit']) ? (int)$data['approved_ads_limit'] : null;
        $cpr->approved_price = (float)$data['approved_price'];
        $cpr->approved_validity_days = (int)$data['approved_validity_days'];
        $cpr->admin_notes = $data['admin_notes'] ?? null;
        $cpr->approved_by = Auth::id();
        $cpr->approved_at = now();
        $cpr->generated_plan_id = $plan->id;

        if ($cpr->payment_preference === 'online') {
            $cpr->status = 'awaiting_payment';
            $cpr->payment_gateway = 'Online';
            $cpr->payment_link = URL::to('payment_method/' . $plan->id . '?cpr=' . $cpr->id);
            $cpr->save();
        } else {
            // Cash: immediately activate the plan for the user
            $cpr->status = 'activated';
            $cpr->payment_gateway = 'Cash';
            $cpr->payment_link = null;
            $cpr->paid_at = now();
            $cpr->activated_at = now();
            $cpr->save();

            // Assign plan to user
            $user = User::findOrFail($cpr->user_id);
            $plan_days = $cpr->approved_validity_days ?: 0;
            $plan_amount = $cpr->approved_price ?: 0;

            // Carry-over remaining posts from ALL previous activated CPR cycles
            $plan = SubscriptionPlan::find($cpr->generated_plan_id);
            $carryOverTotal = 0;
            $priorCprs = CustomPlanRequest::where('user_id', $user->id)
                ->where(function($q){
                    $q->whereIn('status', ['activated','ACTIVATED'])
                      ->orWhereNotNull('activated_at');
                })
                ->where('id', '!=', $cpr->id)
                ->orderBy('activated_at')
                ->orderBy('id')
                ->get();
            foreach ($priorCprs as $index => $pc) {
                if (empty($pc->activated_at)) { continue; }
                $start = $pc->activated_at;
                $end = null;
                // next cycle start is next CPR's activation or current CPR's activation
                if (isset($priorCprs[$index+1]) && !empty($priorCprs[$index+1]->activated_at)) {
                    $end = $priorCprs[$index+1]->activated_at;
                } else {
                    $end = $cpr->activated_at ?: now();
                }
                $limit = (int)($pc->approved_property_limit ?? $pc->property_posts_requested ?? 0);
                if ($limit <= 0) { continue; }
                $used = Property::where('user_id', $user->id)
                    ->whereNotNull('approved_at')
                    ->where('approved_at', '>=', $start)
                    ->where('approved_at', '<', $end)
                    ->count();
                $carryOverTotal += max(0, $limit - $used);
            }
            if ($plan && $carryOverTotal > 0) {
                $plan->plan_property_limit = (int)$plan->plan_property_limit + $carryOverTotal;
                $plan->save();
            }

            $user->plan_id = $plan->id;
            $user->start_date = strtotime(date('m/d/Y'));
            // No-expiry: do not set exp_date
            $user->plan_amount = $plan_amount;
            $user->save();

            // Log transaction as cash
            $payment_trans = new Transactions;
            $payment_trans->user_id = $user->id;
            $payment_trans->email = $user->email;
            $payment_trans->plan_id = $plan->id;
            $payment_trans->gateway = 'Cash';
            $payment_trans->payment_amount = $plan_amount;
            $payment_trans->payment_id = $cpr->payment_id ?: '-';
            $payment_trans->transaction_type = 'subscription';
            $payment_trans->payment_status = 'completed';
            $payment_trans->payment_date = now();
            $payment_trans->date = strtotime(date('m/d/Y H:i:s'));
            $payment_trans->save();
        }

        Session::flash('flash_message', 'Custom plan approved successfully.');
        return redirect()->back();
    }

    public function reject($id, Request $request)
    {
        if (Auth::User()->usertype != "Admin") {
            Session::flash('flash_message', trans('words.access_denied'));
            return redirect('dashboard');
        }

        $cpr = CustomPlanRequest::findOrFail($id);
        $cpr->status = 'rejected';
        $cpr->admin_notes = $request->get('admin_notes');
        $cpr->approved_by = Auth::id();
        $cpr->approved_at = now();
        $cpr->save();

        Session::flash('flash_message', 'Request rejected.');
        return redirect()->back();
    }

    public function markPaid($id, Request $request)
    {
        if (Auth::User()->usertype != "Admin") {
            Session::flash('flash_message', trans('words.access_denied'));
            return redirect('dashboard');
        }

        $cpr = CustomPlanRequest::findOrFail($id);
        $cpr->status = 'paid';
        $cpr->paid_at = now();
        $cpr->payment_id = $request->get('payment_id');
        $cpr->save();

        Session::flash('flash_message', 'Marked as paid. You can now activate the plan for user.');
        return redirect()->back();
    }

    public function activate($id)
    {
        if (Auth::User()->usertype != "Admin") {
            Session::flash('flash_message', trans('words.access_denied'));
            return redirect('dashboard');
        }

        $cpr = CustomPlanRequest::findOrFail($id);
        if (!$cpr->generated_plan_id) {
            Session::flash('flash_message', 'No generated plan to activate. Approve first.');
            return redirect()->back();
        }

        $user = User::findOrFail($cpr->user_id);

        $plan_days = $cpr->approved_validity_days ?: 0;
        $plan_amount = $cpr->approved_price ?: 0;
        $plan_id = $cpr->generated_plan_id;

        // Carry-over remaining posts from ALL previous activated CPR cycles
        $plan = SubscriptionPlan::find($plan_id);
        $carryOverTotal = 0;
        $priorCprs = CustomPlanRequest::where('user_id', $user->id)
            ->where(function($q){
                $q->whereIn('status', ['activated','ACTIVATED'])
                  ->orWhereNotNull('activated_at');
            })
            ->where('id', '!=', $cpr->id)
            ->orderBy('activated_at')
            ->orderBy('id')
            ->get();
        foreach ($priorCprs as $index => $pc) {
            if (empty($pc->activated_at)) { continue; }
            $start = $pc->activated_at;
            $end = null;
            // next cycle start is next CPR's activation or current CPR's activation
            if (isset($priorCprs[$index+1]) && !empty($priorCprs[$index+1]->activated_at)) {
                $end = $priorCprs[$index+1]->activated_at;
            } else {
                $end = $cpr->activated_at ?: now();
            }
            $limit = (int)($pc->approved_property_limit ?? $pc->property_posts_requested ?? 0);
            if ($limit <= 0) { continue; }
            $used = Property::where('user_id', $user->id)
                ->whereNotNull('approved_at')
                ->where('approved_at', '>=', $start)
                ->where('approved_at', '<', $end)
                ->count();
            $carryOverTotal += max(0, $limit - $used);
        }
        if ($plan && $carryOverTotal > 0) {
            $plan->plan_property_limit = (int)$plan->plan_property_limit + $carryOverTotal;
            $plan->save();
        }

        // Assign plan to user
        $user->plan_id = $plan_id;
        $user->start_date = strtotime(date('m/d/Y'));
        // No-expiry: do not set exp_date
        $user->plan_amount = $plan_amount;
        $user->save();

        // Log transaction
        $payment_trans = new Transactions;
        $payment_trans->user_id = $user->id;
        $payment_trans->email = $user->email;
        $payment_trans->plan_id = $plan_id;
        $payment_trans->gateway = $cpr->payment_gateway ?: 'Manual';
        $payment_trans->payment_amount = $plan_amount;
        $payment_trans->payment_id = $cpr->payment_id ?: '-';
        $payment_trans->transaction_type = 'subscription';
        $payment_trans->payment_status = 'completed';
        $payment_trans->payment_date = now();
        $payment_trans->date = strtotime(date('m/d/Y H:i:s'));
        $payment_trans->save();

        $cpr->status = 'activated';
        $cpr->activated_at = now();
        $cpr->save();

        Session::flash('flash_message', 'Plan activated for user.');
        return redirect()->back();
    }

    public function syncLimits($id)
    {
        if (Auth::User()->usertype != "Admin") {
            Session::flash('flash_message', trans('words.access_denied'));
            return redirect('dashboard');
        }

        $cpr = CustomPlanRequest::findOrFail($id);
        if (!$cpr->generated_plan_id) {
            Session::flash('flash_message', 'No generated plan to sync. Approve first.');
            return redirect()->back();
        }

        $plan = SubscriptionPlan::find($cpr->generated_plan_id);
        if (!$plan) {
            Session::flash('flash_message', 'Generated plan not found.');
            return redirect()->back();
        }

        // Prefer approved limits; fallback to requested if approved missing
        $propLimit = (int)($cpr->approved_property_limit ?? 0);
        $adsLimit = $cpr->approved_ads_limit !== null ? (int)$cpr->approved_ads_limit : null;
        if ($propLimit === 0) {
            $propLimit = (int)($cpr->property_posts_requested ?? 0);
        }
        if ($adsLimit === null) {
            $adsLimit = $cpr->ads_posts_requested !== null ? (int)$cpr->ads_posts_requested : null;
        }

        if ($propLimit > 0) {
            $plan->plan_property_limit = $propLimit;
        }
        $plan->plan_ads_limit = $adsLimit; // can be null
        $plan->save();

        Session::flash('flash_message', 'Plan limits synced from Custom Plan Request.');
        return redirect()->back();
    }
}
