<?php

namespace App\Http\Controllers\Admin;

use Auth;
use Illuminate\Http\Request;
use App\Models\WebAdvertisement;
use App\Models\User;
use App\Location;

class WebAdvertisementController extends MainAdminController
{
    public function index(Request $request, string $position = null)
    {
        if(Auth::User()->usertype!="Admin"){
            \Session::flash('flash_message', trans('words.access_denied'));
            return redirect('admin/dashboard');
        }

        // Normalize and validate requested position
        $allowedPositions = ['home-top','home-right','home-left','home-bottom','property-bottom'];
        if ($position === null) {
            $position = 'home-top';
        }
        if (!in_array($position, $allowedPositions, true)) {
            abort(404);
        }

        // Translate route position to DB fields
        if (str_starts_with($position, 'home-')) {
            $page = 'home';
            $dbPosition = str_replace('home-', '', $position); // top/right/left/bottom
        } else {
            // property-*
            $page = 'property';
            $dbPosition = str_replace('property-', '', $position); // bottom
        }

        $ads = WebAdvertisement::query()
            ->where('page', $page)
            ->where('position', $dbPosition)
            ->orderByDesc('id')
            ->paginate(10);

        $page_title = 'Banner Ads - '.ucwords(str_replace('-', ' ', $position));

        $banners = $ads;
        return view('admin.pages.banner_ads.index', compact('page_title','ads','banners','position'));
    }

    public function create(Request $request, string $position)
    {
        if(Auth::User()->usertype!="Admin"){
            \Session::flash('flash_message', trans('words.access_denied'));
            return redirect('admin/dashboard');
        }

        $allowedPositions = ['home-top','home-right','home-left','home-bottom','property-bottom'];
        if (!in_array($position, $allowedPositions, true)) {
            abort(404);
        }

        $page_title = 'Add Banner - '.ucwords(str_replace('-', ' ', $position));
        $users = User::orderBy('name')->get(['id','name','email']);
        $locations = Location::orderBy('name')->get(['id','name']);
        return view('admin.pages.banner_ads.create', compact('page_title','position','users','locations'));
    }

    public function store(Request $request, string $position)
    {
        if(Auth::User()->usertype!="Admin"){
            \Session::flash('flash_message', trans('words.access_denied'));
            return redirect('admin/dashboard');
        }

        $allowedPositions = ['home-top','home-right','home-left','home-bottom','property-bottom'];
        if (!in_array($position, $allowedPositions, true)) {
            abort(404);
        }

        if (str_starts_with($position, 'home-')) {
            $dbPosition = str_replace('home-', '', $position); // top/right/left/bottom
        } else {
            $dbPosition = str_replace('property-', '', $position); // bottom
        }

        $data = $request->validate([
            'name' => 'required|string|max:255',
            'redirect_url' => 'nullable|url|max:1000',
            'description' => 'nullable|string',
            'cost' => 'nullable|numeric',
            'status' => 'nullable|boolean',
            'starts_at' => 'nullable|date',
            'ends_at' => 'nullable|date|after_or_equal:starts_at',
            'user_id' => 'required|exists:users,id',
            'location_id' => 'nullable|exists:location,id',
            'lat' => 'required|numeric',
            'lng' => 'required|numeric',
            'image_file' => 'required|image|max:4096',
        ]);

        $banner = new WebAdvertisement();
        $banner->page = str_starts_with($position, 'home-') ? 'home' : 'property';
        $banner->position = $dbPosition;
        $banner->name = $data['name'];
        $banner->user_id = $data['user_id'];
        // handle upload if present
        // upload image (required)
        $file = $request->file('image_file');
        $destination = public_path('site_assets/images/advertisement');
        if (!is_dir($destination)) {
            @mkdir($destination, 0755, true);
        }
        $extension = $file->getClientOriginalExtension();
        $filename = 'banner_'.time().'_'.uniqid().'.'.$extension;
        $file->move($destination, $filename);
        $banner->image_path = 'site_assets/images/advertisement/'.$filename;
        $banner->redirect_url = $data['redirect_url'] ?? null;
        $banner->description = $data['description'] ?? null;
        $banner->cost = $data['cost'] ?? null;
        $banner->status = $data['status'] ?? 0;
        $banner->starts_at = $data['starts_at'] ?? null;
        $banner->ends_at = $data['ends_at'] ?? null;
        $banner->location_id = $data['location_id'] ?? null;
        $banner->lat = round((float)$data['lat'], 6);
        $banner->lng = round((float)$data['lng'], 6);
        $banner->save();

        \Session::flash('flash_message', trans('words.successfully_added'));
        return redirect('admin/banner-ads/'.$position);
    }

    public function edit(Request $request, string $position, int $id)
    {
        if(Auth::User()->usertype!="Admin"){
            \Session::flash('flash_message', trans('words.access_denied'));
            return redirect('admin/dashboard');
        }

        $allowedPositions = ['home-top','home-right','home-left','home-bottom','property-bottom'];
        if (!in_array($position, $allowedPositions, true)) {
            abort(404);
        }

        $banner = WebAdvertisement::findOrFail($id);
        $page_title = 'Edit Banner - '.ucwords(str_replace('-', ' ', $position));
        $users = User::orderBy('name')->get(['id','name','email']);
        $locations = Location::orderBy('name')->get(['id','name']);
        return view('admin.pages.banner_ads.edit', compact('page_title','position','banner','users','locations'));
    }

    public function update(Request $request, string $position, int $id)
    {
        if(Auth::User()->usertype!="Admin"){
            \Session::flash('flash_message', trans('words.access_denied'));
            return redirect('admin/dashboard');
        }

        $allowedPositions = ['home-top','home-right','home-left','home-bottom','property-bottom'];
        if (!in_array($position, $allowedPositions, true)) {
            abort(404);
        }

        $data = $request->validate([
            'name' => 'required|string|max:255',
            'redirect_url' => 'nullable|url|max:1000',
            'description' => 'nullable|string',
            'cost' => 'nullable|numeric',
            'status' => 'nullable|boolean',
            'starts_at' => 'nullable|date',
            'ends_at' => 'nullable|date|after_or_equal:starts_at',
            'user_id' => 'required|exists:users,id',
            'location_id' => 'nullable|exists:location,id',
            'lat' => 'required|numeric',
            'lng' => 'required|numeric',
            'image_file' => 'nullable|image|max:4096',
        ]);

        $banner = WebAdvertisement::findOrFail($id);
        $banner->name = $data['name'];
        $banner->user_id = $data['user_id'];
        if ($request->hasFile('image_file')) {
            $file = $request->file('image_file');
            $destination = public_path('site_assets/images/advertisement');
            if (!is_dir($destination)) { @mkdir($destination, 0755, true); }
            $extension = $file->getClientOriginalExtension();
            $filename = 'banner_'.time().'_'.uniqid().'.'.$extension;
            $file->move($destination, $filename);
            $banner->image_path = 'site_assets/images/advertisement/'.$filename;
        }
        $banner->redirect_url = $data['redirect_url'] ?? null;
        $banner->description = $data['description'] ?? null;
        $banner->cost = $data['cost'] ?? null;
        $banner->status = $data['status'] ?? 0;
        $banner->starts_at = $data['starts_at'] ?? null;
        $banner->ends_at = $data['ends_at'] ?? null;
        $banner->location_id = $data['location_id'] ?? null;
        $banner->lat = round((float)$data['lat'], 6);
        $banner->lng = round((float)$data['lng'], 6);
        $banner->save();

        \Session::flash('flash_message', trans('words.successfully_updated'));
        return redirect('admin/banner-ads/'.$position);
    }

    public function destroy(Request $request, string $position, int $id)
    {
        if(Auth::User()->usertype!="Admin"){
            \Session::flash('flash_message', trans('words.access_denied'));
            return redirect('admin/dashboard');
        }

        $banner = WebAdvertisement::findOrFail($id);
        $banner->delete();

        \Session::flash('flash_message', trans('words.deleted'));
        return redirect('admin/banner-ads/'.$position);
    }
}


