<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Contracts\OTPServiceInterface;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Validator;
use App\Models\User;
use Illuminate\Support\Facades\Log;

class PhoneAuthController extends Controller
{
    protected OTPServiceInterface $otpService;

    public function __construct(OTPServiceInterface $otpService)
    {
        $this->otpService = $otpService;
    }

    /**
     * Show phone login form
     */
    public function showLogin()
    {
        if (Auth::check()) {
            return redirect('dashboard');
        }
        return view('pages.user.phone_login');
    }

    /**
     * Show phone signup form
     */
    public function showSignup()
    {
        if (Auth::check()) {
            return redirect('dashboard');
        }
        return view('pages.user.phone_signup');
    }

    /**
     * Send OTP for login
     */
    public function sendLoginOtp(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'phone' => 'required|string|regex:/^[6-9][0-9]{9}$/'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Please enter a valid 10-digit mobile number'
            ], 422);
        }

        // Validate CSRF token for AJAX requests (check both header and request body)
        $hasValidToken = false;
        if ($request->ajax()) {
            $headerToken = $request->header('X-CSRF-TOKEN');
            $bodyToken = $request->input('_token');
            $hasValidToken = !empty($headerToken) || !empty($bodyToken);
        }

        if ($request->ajax() && !$hasValidToken) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid request. Please refresh the page and try again.'
            ], 419);
        }

        $phone = $request->input('phone');

        // Rate limiting: Max 3 OTP requests per minute per phone number
        $rateLimitKey = 'otp_requests_' . $phone;
        $otpRequests = Cache::get($rateLimitKey, 0);

        if ($otpRequests >= 3) {
            $resetTime = Cache::get($rateLimitKey . '_reset', now()->addMinute());
            $remainingTime = now()->diffInMinutes($resetTime, false);

            if ($remainingTime > 0) {
                return response()->json([
                    'success' => false,
                    'message' => "Too many OTP requests. Please try again after {$remainingTime} minutes."
                ], 429);
            } else {
                // Reset the counter
                Cache::forget($rateLimitKey);
                Cache::forget($rateLimitKey . '_reset');
                $otpRequests = 0;
            }
        }

        // Check if user exists with this phone number (checking both phone and mobile fields)
        $user = User::where('phone', $phone)->orWhere('mobile', $phone)->first();

        if (!$user) {
            return response()->json([
                'success' => false,
                'message' => 'No account found with this phone number. Please sign up first.'
            ], 404);
        }
        // Send OTP
        $result = $this->otpService->sendVerificationOtp($phone);

        if ($result['success']) {
            // Store OTP in session for verification
            Session::put('login_otp', $result['otp']);
            Session::put('login_phone', $phone);
            Session::put('login_user_id', $user->id);
            Session::put('otp_sent_at', now()->timestamp);

            // Update rate limiting cache
            Cache::put($rateLimitKey, $otpRequests + 1, 60); // 1 minute expiry
            if ($otpRequests === 0) {
                Cache::put($rateLimitKey . '_reset', now()->addMinute(), 60);
            }

            // Debug logging (without exposing OTP)
            Log::info('Login OTP Sent Successfully', [
                'phone' => $phone,
                'user_id' => $user->id,
                'otp_length' => strlen($result['otp']),
                'session_stored' => [
                    'login_phone' => Session::get('login_phone'),
                    'login_user_id' => Session::get('login_user_id'),
                    'otp_sent_at' => Session::get('otp_sent_at'),
                    'session_id' => Session::getId()
                ]
            ]);

            return response()->json([
                'success' => true,
                'message' => 'OTP sent successfully to your phone number'
            ]);
        }

        return response()->json([
            'success' => false,
            'message' => $result['message']
        ], 500);
    }

    /**
     * Send OTP for signup
     */
    public function sendSignupOtp(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'phone' => 'required|string|regex:/^[6-9][0-9]{9}$/|unique:users,phone|unique:users,mobile'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Please enter a valid 10-digit mobile number that is not already registered'
            ], 422);
        }

        // Validate CSRF token for AJAX requests (check both header and request body)
        $hasValidToken = false;
        if ($request->ajax()) {
            $headerToken = $request->header('X-CSRF-TOKEN');
            $bodyToken = $request->input('_token');
            $hasValidToken = !empty($headerToken) || !empty($bodyToken);
        }

        if ($request->ajax() && !$hasValidToken) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid request. Please refresh the page and try again.'
            ], 419);
        }

        $phone = $request->input('phone');

        // Rate limiting: Max 3 OTP requests per minute per phone number
        $rateLimitKey = 'otp_requests_' . $phone;
        $otpRequests = Cache::get($rateLimitKey, 0);

        if ($otpRequests >= 3) {
            $resetTime = Cache::get($rateLimitKey . '_reset', now()->addMinute());
            $remainingTime = now()->diffInMinutes($resetTime, false);

            if ($remainingTime > 0) {
                return response()->json([
                    'success' => false,
                    'message' => "Too many OTP requests. Please try again after {$remainingTime} minutes."
                ], 429);
            } else {
                // Reset the counter
                Cache::forget($rateLimitKey);
                Cache::forget($rateLimitKey . '_reset');
                $otpRequests = 0;
            }
        }

        // Send OTP
        $result = $this->otpService->sendVerificationOtp($phone);

        if ($result['success']) {
            // Store OTP in session for verification
            Session::put('signup_otp', $result['otp']);
            Session::put('signup_phone', $phone);
            Session::put('otp_sent_at', now()->timestamp);

            // Debug logging (without exposing OTP)
            Log::info('Signup OTP Sent Successfully', [
                'phone' => $phone,
                'otp_length' => strlen($result['otp']),
                'session_stored' => [
                    'signup_phone' => Session::get('signup_phone'),
                    'otp_sent_at' => Session::get('otp_sent_at'),
                    'session_id' => Session::getId()
                ]
            ]);

            // Update rate limiting cache
            Cache::put($rateLimitKey, $otpRequests + 1, 60); // 1 minute expiry
            if ($otpRequests === 0) {
                Cache::put($rateLimitKey . '_reset', now()->addMinute(), 60);
            }

            return response()->json([
                'success' => true,
                'message' => 'OTP sent successfully to your phone number'
            ]);
        }

        return response()->json([
            'success' => false,
            'message' => $result['message']
        ], 500);
    }

    /**
     * Verify OTP for login
     */
    public function verifyLoginOtp(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'phone' => 'required|string',
            'otp' => 'required|string|size:4'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid input data'
            ], 422);
        }

        $phone = $request->input('phone');
        $otp = $request->input('otp');

        // Get stored OTP data
        $storedOtp = Session::get('login_otp');
        $storedPhone = Session::get('login_phone');
        $userId = Session::get('login_user_id');
        $otpSentAt = Session::get('otp_sent_at');

        // Debug logging (without exposing OTP)
        Log::info('Login OTP Verification Debug', [
            'phone' => $phone,
            'otp_length' => strlen($otp),
            'stored_otp_length' => $storedOtp ? strlen($storedOtp) : 0,
            'stored_phone' => $storedPhone,
            'user_id' => $userId,
            'otp_sent_at' => $otpSentAt,
            'current_time' => now()->timestamp,
            'phone_match' => $phone === $storedPhone,
            'session_data' => [
                'has_otp' => Session::has('login_otp'),
                'has_phone' => Session::has('login_phone'),
                'has_user_id' => Session::has('login_user_id'),
                'has_otp_sent_at' => Session::has('otp_sent_at')
            ]
        ]);

        // Check if session data exists
        if (!$storedOtp || !$storedPhone || !$userId) {
            Log::warning('Login OTP session data missing', [
                'stored_otp' => $storedOtp ? 'exists' : 'null',
                'stored_phone' => $storedPhone ? 'exists' : 'null',
                'user_id' => $userId ? 'exists' : 'null',
                'session_keys' => Session::allKeys()
            ]);
            return response()->json([
                'success' => false,
                'message' => 'OTP session expired. Please request a new OTP.'
            ], 422);
        }

        // More flexible phone number comparison (trim and normalize)
        $normalizedPhone = preg_replace('/\D/', '', $phone);
        $normalizedStoredPhone = preg_replace('/\D/', '', $storedPhone);

        if ($normalizedPhone !== $normalizedStoredPhone) {
            Log::warning('Phone number mismatch', [
                'input_phone' => $phone,
                'normalized_input' => $normalizedPhone,
                'stored_phone' => $storedPhone,
                'normalized_stored' => $normalizedStoredPhone
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Phone number mismatch'
            ], 422);
        }

        // Check account lockout (max 5 failed attempts per hour)
        $lockoutKey = 'login_attempts_' . $phone;
        $failedAttempts = Cache::get($lockoutKey, 0);

        if ($failedAttempts >= 5) {
            $lockoutReset = Cache::get($lockoutKey . '_reset', now()->addMinute());
            $remainingTime = now()->diffInMinutes($lockoutReset, false);

            if ($remainingTime > 0) {
                return response()->json([
                    'success' => false,
                    'message' => "Account temporarily locked due to too many failed attempts. Try again after {$remainingTime} minutes."
                ], 423);
            } else {
                // Reset lockout
                Cache::forget($lockoutKey);
                Cache::forget($lockoutKey . '_reset');
                $failedAttempts = 0;
            }
        }

        // Check OTP expiry (10 minutes)
        if ((now()->timestamp - $otpSentAt) > 600) {
            Session::forget(['login_otp', 'login_phone', 'login_user_id', 'otp_sent_at']);
            return response()->json([
                'success' => false,
                'message' => 'OTP expired. Please request a new OTP.'
            ], 422);
        }

        if ($otp === (string)$storedOtp) {
            // OTP verified, log in the user
            $user = User::find($userId);
            Auth::login($user);

            // Clear OTP session data to prevent reuse
            Session::forget(['login_otp', 'login_phone', 'login_user_id', 'otp_sent_at']);

            // Regenerate session to prevent session fixation
            Session::regenerate();

            // Reset failed attempts on successful login
            $lockoutKey = 'login_attempts_' . $phone;
            Cache::forget($lockoutKey);
            Cache::forget($lockoutKey . '_reset');

            // Update phone_verified_at if not already set
            if (!$user->phone_verified_at) {
                $user->phone_verified_at = now();
                $user->save();
            }

            return response()->json([
                'success' => true,
                'message' => 'Login successful',
                'redirect' => route('dashboard')
            ]);
        }

        // OTP verification failed - increment counter
        $lockoutKey = 'login_attempts_' . $phone;
        $failedAttempts = Cache::get($lockoutKey, 0);
        Cache::put($lockoutKey, $failedAttempts + 1, 60);
        if ($failedAttempts === 0) {
            Cache::put($lockoutKey . '_reset', now()->addMinute(), 60);
        }

        return response()->json([
            'success' => false,
            'message' => 'Invalid OTP'
        ], 422);
    }

    /**
     * Verify OTP for signup
     */
    public function verifySignupOtp(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'phone' => 'required|string',
            'otp' => 'required|string|size:4',
            'name' => 'required|string|max:255',
            'business_type' => 'required|in:1,2'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid input data'
            ], 422);
        }

        $phone = $request->input('phone');
        $otp = $request->input('otp');
        $name = $request->input('name');
        $businessType = $request->input('business_type');

        // Get stored OTP data
        $storedOtp = Session::get('signup_otp');
        $storedPhone = Session::get('signup_phone');
        $otpSentAt = Session::get('otp_sent_at');

        // Debug logging (without exposing OTP)
        Log::info('Signup OTP Verification Debug', [
            'phone' => $phone,
            'otp_length' => strlen($otp),
            'name' => $name,
            'stored_otp_length' => $storedOtp ? strlen($storedOtp) : 0,
            'stored_phone' => $storedPhone,
            'otp_sent_at' => $otpSentAt,
            'current_time' => now()->timestamp,
            'phone_match' => $phone === $storedPhone,
            'session_data' => [
                'has_otp' => Session::has('signup_otp'),
                'has_phone' => Session::has('signup_phone'),
                'has_otp_sent_at' => Session::has('otp_sent_at')
            ]
        ]);

        // Check if session data exists
        if (!$storedOtp || !$storedPhone) {
            Log::warning('Signup OTP session data missing', [
                'stored_otp' => $storedOtp ? 'exists' : 'null',
                'stored_phone' => $storedPhone ? 'exists' : 'null',
                'session_keys' => Session::allKeys()
            ]);
            return response()->json([
                'success' => false,
                'message' => 'OTP session expired. Please request a new OTP.'
            ], 422);
        }

        // More flexible phone number comparison (trim and normalize)
        $normalizedPhone = preg_replace('/\D/', '', $phone);
        $normalizedStoredPhone = preg_replace('/\D/', '', $storedPhone);

        if ($normalizedPhone !== $normalizedStoredPhone) {
            Log::warning('Signup phone number mismatch', [
                'input_phone' => $phone,
                'normalized_input' => $normalizedPhone,
                'stored_phone' => $storedPhone,
                'normalized_stored' => $normalizedStoredPhone
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Phone number mismatch'
            ], 422);
        }

        // Check OTP expiry (10 minutes)
        if ((now()->timestamp - $otpSentAt) > 600) {
            Session::forget(['signup_otp', 'signup_phone', 'otp_sent_at']);
            return response()->json([
                'success' => false,
                'message' => 'OTP expired. Please request a new OTP.'
            ], 422);
        }

        if ($otp === (string)$storedOtp) {
            try {
                Log::info('User creation attempt', [
                    'name' => $name,
                    'phone' => $phone,
                    'business_type' => $businessType,
                    'email' => $phone . '@phone.auth',
                    'password_generated' => true,
                    'fillable_fields' => (new User())->getFillable()
                ]);

                // Generate a secure random password
                $randomPassword = 'PhoneAuth_' . $phone . '_' . time() . '_' . rand(1000, 9999);
                $hashedPassword = bcrypt($randomPassword);

                // Create new user
                $user = User::create([
                    'name' => $name,
                    'phone' => $phone,
                    'email' => $phone . '@phone.auth', // Provide a default email based on phone
                    'password' => $hashedPassword,
                    'business_type' => $businessType,
                    'phone_verified_at' => now(),
                    'usertype' => 'User',
                    'status' => 1
                ]);

                Log::info('User created successfully', [
                    'user_id' => $user->id,
                    'name' => $user->name,
                    'phone' => $user->phone,
                    'business_type' => $user->business_type,
                    'email' => $user->email
                ]);

                // Log in the user
                Auth::login($user);

                // Clear OTP session data to prevent reuse
                Session::forget(['signup_otp', 'signup_phone', 'otp_sent_at']);

                // Regenerate session to prevent session fixation
                Session::regenerate();

                return response()->json([
                    'success' => true,
                    'message' => 'Account created and login successful',
                    'redirect' => route('dashboard')
                ]);
            } catch (\Exception $e) {
                Log::error('User creation failed', [
                    'error' => $e->getMessage(),
                    'name' => $name,
                    'phone' => $phone
                ]);

                return response()->json([
                    'success' => false,
                    'message' => 'Failed to create account: ' . $e->getMessage()
                ], 500);
            }
        }

        return response()->json([
            'success' => false,
            'message' => 'Invalid OTP'
        ], 422);
    }

    /**
     * Resend OTP
     */
    public function resendOtp(Request $request)
    {
        $type = $request->input('type', 'login'); // login or signup
        $phone = $request->input('phone');

        if ($type === 'login') {
            return $this->sendLoginOtp($request);
        } else {
            return $this->sendSignupOtp($request);
        }
    }

    /**
     * Logout
     */
    public function logout(Request $request)
    {
        try {
            // Clear all authentication-related session data
            Session::forget(['login_otp', 'login_phone', 'login_user_id', 'signup_otp', 'signup_phone', 'otp_sent_at']);

            // Logout the user
            Auth::logout();

            // Regenerate session to prevent session fixation
            Session::regenerate();

            // Invalidate the session token
            Session::invalidate();

            // Flush all session data
            Session::flush();

            return redirect('/');
        } catch (\Exception $e) {
            // Log the error but still redirect to home
            Log::error('Logout error: ' . $e->getMessage());
            return redirect('/');
        }
    }
}