<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Session;
use App\CustomPlanRequest;

class CustomPlanRequestController extends Controller
{
    public function index()
    {
        if (!Auth::check()) {
            Session::flash('error_flash_message', trans('words.access_denied'));
            return redirect('login');
        }

        $requests = CustomPlanRequest::with('plan')
            ->where('user_id', Auth::id())
            ->orderByDesc('id')
            ->paginate(12);

        return view('pages.user.custom_plan_requests', compact('requests'));
    }

    public function create()
    {
        if (!Auth::check()) {
            Session::flash('error_flash_message', trans('words.access_denied'));
            return redirect('login');
        }

        return view('pages.payment.custom_plan_request');
    }

    public function store(Request $request)
    {
        if (!Auth::check()) {
            Session::flash('error_flash_message', trans('words.access_denied'));
            return redirect('login');
        }

        $data = $request->all();

        $rules = [
            'property_posts_needed' => 'required|integer|min:1',
            'ads_posts_needed' => 'nullable|integer|min:0',
            'notes' => 'nullable|string|max:2000',
            'payment_preference' => 'required|in:online,cash',
        ];

        $validator = \Validator::make($data, $rules);
        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator->messages())->withInput();
        }

        $cpr = CustomPlanRequest::create([
            'user_id' => Auth::id(),
            'property_posts_requested' => (int)$data['property_posts_needed'],
            'ads_posts_requested' => isset($data['ads_posts_needed']) ? (int)$data['ads_posts_needed'] : null,
            'notes' => $data['notes'] ?? null,
            'payment_preference' => $data['payment_preference'],
            'status' => 'pending',
        ]);

        return redirect()->to('custom-plan/request/' . $cpr->id . '/submitted');
    }

    public function submitted($id)
    {
        if (!Auth::check()) {
            Session::flash('error_flash_message', trans('words.access_denied'));
            return redirect('login');
        }

        $requestModel = CustomPlanRequest::where('id', $id)->where('user_id', Auth::id())->firstOrFail();

        return view('pages.payment.custom_plan_submitted', [
            'request' => $requestModel,
        ]);
    }
}
