<?php

namespace App\Http\Controllers;

use App\Property;
use App\Models\Lead;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;
use App\Factories\OTPServiceFactory;

class LeadsController extends Controller
{
    public function store(Request $request)
    {
        $action = $request->input('action');

        // Conditional validation: for OTP flows we don't require name/email
        if ($action === 'verify_phone') {
            $v = Validator::make($request->all(), [
                'phone' => 'required|string|max:50',
                'action' => 'required|string'
            ]);
        } elseif ($action === 'verify_otp') {
            $v = Validator::make($request->all(), [
                'phone' => 'required|string|max:50',
                'otp' => 'required|string|size:4',
                'action' => 'required|string'
            ]);
        } else {
            $v = Validator::make($request->all(), [
                'name' => 'required|string|max:191',
                'phone' => 'required|string|max:50',
                'email' => 'nullable|email|max:191',
                'action' => 'required|string',
                'property_id' => 'nullable|integer',
            ]);
        }

        if ($v->fails()) {
            return response()->json(['status' => 'error', 'errors' => $v->errors()], 422);
        }

        // If frontend requests to send OTP for phone verification
        if ($request->input('action') === 'verify_phone') {
            return $this->verifyPhone($request);
        }

        // Handle OTP verification
        if ($request->input('action') === 'verify_otp') {
            return $this->verifyOtp($request);
        }

        // For other actions (view/download) require phone verification first
        return response()->json(['status' => 'ok', 'message' => 'Phone verification required']);
    }

    private function captureLead(array $data)
    {
        try {
            $lead = Lead::create([
                'name' => isset($data['name']) ? $data['name'] : null,
                'phone' => isset($data['phone']) ? $data['phone'] : null,
                'email' => isset($data['email']) ? $data['email'] : null,
                'action' => isset($data['action']) ? $data['action'] : null,
                'property_id' => isset($data['property_id']) ? $data['property_id'] : null,
                'ip' => isset($data['ip']) ? $data['ip'] : null,
                'user_id' => \Auth::check() ? \Auth::id() : (isset($data['user_id']) ? $data['user_id'] : null),
                'meta' => isset($data['user_agent']) ? json_encode(['user_agent' => $data['user_agent']]) : null
            ]);

            Log::info('Lead captured successfully', [
                'name' => isset($data['name']) ? $data['name'] : null,
                'phone' => isset($data['phone']) ? $data['phone'] : null,
                'action' => isset($data['action']) ? $data['action'] : null,
                'property_id' => isset($data['property_id']) ? $data['property_id'] : null,
                'lead_id' => $lead->id
            ]);

            return $lead;
        } catch (\Exception $e) {
            Log::error('Failed to capture lead', [
                'error' => $e->getMessage(),
                'data' => $data
            ]);
            return null;
        }
    }

    public function verifyPhone(Request $request)
    {
        try {
            $phone = preg_replace('/\D/', '', $request->input('phone'));
            // Normalize phone: remove leading plus and leading zeros
            $phone = ltrim($phone, '+');
            $phone = preg_replace('/^0+/', '', $phone);
            $name = $request->input('name');
            $email = $request->input('email');
            $requestedAction = $request->input('requested_action', $request->input('action'));
            $propertyId = $request->input('property_id');

            // Normalize phone: if 10 digits, prefix country code if needed (assume 91 for India)
            if (strlen($phone) == 10) {
                $phone = '91' . $phone;
            }

            // If number is 11 and started with country-less 0 it will now be 10 after stripping; else if it's 11 and still starts with '91' etc, leave as-is.

            // Generate OTP
            $otp = rand(1000, 9999);

            // Store OTP in cache for verification
            \Cache::put('otp_' . $phone, $otp, 600); // 10 minutes

            // Store pending lead data in cache so we can create lead only after verification
            $pending = [
                'name' => $name,
                'phone' => $phone,
                'email' => $email,
                'requested_action' => $requestedAction,
                'property_id' => $propertyId,
                'owner_id' => $request->input('owner_id'),
                'ip' => $request->ip(),
                'user_agent' => $request->userAgent(),
                'created_at' => now()->toDateTimeString()
            ];

            \Cache::put('pending_lead_' . $phone, $pending, 600);

            // Send OTP via configured provider (Twilio or Fast2SMS)
            try {
                $smsService = OTPServiceFactory::create();
            } catch (\Exception $e) {
                Log::error('OTP provider creation failed', ['error' => $e->getMessage()]);
                return response()->json(['status' => 'error', 'message' => 'OTP provider misconfigured'], 500);
            }

            $message = "Hello {$name}, your verification code is: {$otp}. This code will expire in 10 minutes.";
            $result = $smsService->sendMessage($phone, $message);

            if ($result['success']) {
                return response()->json([
                    'status' => 'ok',
                    'message' => 'OTP sent successfully to your phone number',
                    'phone' => $phone
                ]);
            } else {
                // cleanup pending cache on failure
                \Cache::forget('otp_' . $phone);
                \Cache::forget('pending_lead_' . $phone);

                return response()->json([
                    'status' => 'error',
                    'message' => 'Failed to send OTP: ' . (isset($result['message']) ? $result['message'] : 'Unknown')
                ], 500);
            }
        } catch (\Exception $e) {
            Log::error('Phone verification error', ['error' => $e->getMessage()]);
            return response()->json([
                'status' => 'error',
                'message' => 'Error sending OTP: ' . $e->getMessage()
            ], 500);
        }
    }

    public function verifyOtp(Request $request)
    {
        try {
            $phone = preg_replace('/\D/', '', $request->input('phone'));
            $phone = ltrim($phone, '+');
            $phone = preg_replace('/^0+/', '', $phone);
            $otp = $request->input('otp');

            if (strlen($phone) == 10) {
                $phone = '91' . $phone;
            }

            // Get stored OTP and pending lead data from cache
            $storedOtp = \Cache::get('otp_' . $phone);
            $pending = \Cache::get('pending_lead_' . $phone);

            if (!$storedOtp) {
                Log::warning('OTP not found during verification', ['phone' => $phone]);
                return response()->json([
                    'status' => 'error',
                    'message' => 'OTP expired or not found. Please request a new OTP.'
                ], 400);
            }

            // Validate OTP length
            if (strlen($otp) !== 4) {
                Log::warning('Invalid OTP length during verification', ['phone' => $phone, 'otp_length' => strlen($otp)]);
                return response()->json([
                    'status' => 'error',
                    'message' => 'Invalid OTP format. Please enter a 4-digit OTP.'
                ], 400);
            }

            if (!$pending) {
                Log::warning('Pending lead not found during OTP verification', ['phone' => $phone]);
                return response()->json([
                    'status' => 'error',
                    'message' => 'No pending verification found for this phone. Please request a new OTP.'
                ], 400);
            }

            if ($otp == $storedOtp) {
                // OTP is valid, clear it from cache
                \Cache::forget('otp_' . $phone);
                \Cache::forget('pending_lead_' . $phone);

                // Use pending data for lead creation; fallback to request data if missing
                $leadData = [
                    'name' => isset($pending['name']) ? $pending['name'] : $request->input('name', ''),
                    'phone' => isset($pending['phone']) ? $pending['phone'] : $phone,
                    'email' => isset($pending['email']) ? $pending['email'] : $request->input('email', ''),
                    'action' => isset($pending['requested_action']) ? $pending['requested_action'] : 'phone_verified',
                    'property_id' => isset($pending['property_id']) ? $pending['property_id'] : $request->input('property_id'),
                    'ip' => isset($pending['ip']) ? $pending['ip'] : $request->ip(),
                    'user_agent' => isset($pending['user_agent']) ? $pending['user_agent'] : $request->userAgent(),
                ];

                $lead = $this->captureLead($leadData);

                // Prepare reveal data depending on requested action
                $reveal = [];
                // If a property_id was provided, prefer property-based reveal
                if (!empty($leadData['property_id'])) {
                    $property = Property::find($leadData['property_id']);
                    if ($property) {
                        if ($leadData['action'] === 'view_number' || $leadData['action'] === 'phone_verified') {
                            $reveal['phone'] = $property->phone;
                        }
                        if ($leadData['action'] === 'download_brochure') {
                            if ($property->brochure) {
                                $reveal['brochure_url'] = url('/' . $property->brochure);
                            }
                        }
                    }
                }

                // If owner_id was provided in the pending lead (sidebar flow), reveal owner's phone
                if (empty($reveal['phone']) && !empty($pending['owner_id'])) {
                    try {
                        $ownerPhone = get_user_info($pending['owner_id'], 'phone');
                        if (!empty($ownerPhone)) {
                            $reveal['phone'] = $ownerPhone;
                        }
                    } catch (\Exception $e) {
                        Log::warning('Failed to fetch owner phone for reveal', ['owner_id' => $pending['owner_id'], 'error' => $e->getMessage()]);
                    }
                }

                Log::info('Phone verified and lead captured', [
                    'phone' => $phone,
                    'lead_id' => $lead ? $lead->id : null
                ]);

                return response()->json([
                    'status' => 'ok',
                    'message' => 'OTP verified successfully',
                    'reveal' => $reveal
                ]);
            } else {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Invalid OTP'
                ], 400);
            }
        } catch (\Exception $e) {
            Log::error('OTP verification error', ['error' => $e->getMessage()]);
            return response()->json([
                'status' => 'error',
                'message' => 'Error verifying OTP: ' . $e->getMessage()
            ], 500);
        }
    }

    // Admin listing
    public function adminIndex()
    {
        // Fetch paginated leads with property relation
        try {
            $leads = Lead::with('property')->orderBy('created_at', 'desc')->paginate(20);
            $total = Lead::count();
            Log::info('Admin leads loaded', ['total_leads' => $total]);
        } catch (\Exception $e) {
            Log::error('Failed to load leads for admin', ['error' => $e->getMessage()]);
            $leads = collect();
            $total = 0;
        }

        return view('admin.pages.leads.index', compact('leads', 'total'));
    }

    // User listing (optional)
    public function userIndex()
    {
        if (!Auth::check()) {
            return redirect('/login');
        }

        $user = Auth::user();

        try {
            // Leads where user is the owner of the property
            $leads = Lead::whereHas('property', function ($q) use ($user) {
                $q->where('user_id', $user->id);
            })->with('property')->orderBy('created_at', 'desc')->paginate(20);
        } catch (\Exception $e) {
            Log::error('Failed to load user leads', ['error' => $e->getMessage(), 'user_id' => $user->id]);
            $leads = collect();
        }

        return view('user.leads.index', compact('leads'));
    }
}
