<?php

namespace App\Http\Controllers;

use Auth;
use App\User;
use App\Property;
use App\Type;
use App\Location;
use App\Favourite;
use App\PropertyGallery;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;

class MapViewController extends Controller
{
    public function __construct()
    {
        // Middleware is already applied at route level: Route::post('map/favorite', 'MapViewController@toggleFavorite')->middleware('auth');
    }

    /**
     * Display the map view
     */
    public function index(Request $request)
    {
        try {
            // Get filter parameters
            $search = $request->get('search', '');
            $type_id = $request->get('type_id', '');
            $location_id = $request->get('location_id', '');
            $purpose = $request->get('purpose', '');
            $bedrooms = $request->get('bedrooms', '');
            $bathrooms = $request->get('bathrooms', '');
            $price_min = $request->get('price_min', '');
            $price_max = $request->get('price_max', '');
            $furnished = $request->get('furnished', '');
            $verified = $request->get('verified', '');

            // Build the query for initial data
            $query = Property::with(['types', 'locations', 'users'])
                             ->where('status', 1)
                             ->where('approval_status', 'approved')
                             ->whereNotNull('latitude')
                             ->whereNotNull('longitude')
                             ->where('latitude', '!=', '')
                             ->where('longitude', '!=', '');

            // Apply filters (same as PropertyController map_view method)
            if (!empty($search)) {
                $query->where(function($q) use ($search) {
                    $q->where('title', 'like', '%' . $search . '%')
                      ->orWhere('address', 'like', '%' . $search . '%')
                      ->orWhere('description', 'like', '%' . $search . '%');
                });
            }

            if (!empty($type_id)) {
                $query->where('type_id', $type_id);
            }

            if (!empty($location_id)) {
                $query->where('location_id', $location_id);
            }

            if (!empty($purpose)) {
                $query->where('purpose', $purpose);
            }

            if (!empty($bedrooms)) {
                $query->where('bedrooms', '>=', $bedrooms);
            }

            if (!empty($bathrooms)) {
                $query->where('bathrooms', '>=', $bathrooms);
            }

            if (!empty($price_min)) {
                $query->where('price', '>=', $price_min);
            }

            if (!empty($price_max)) {
                $query->where('price', '<=', $price_max);
            }

            if (!empty($furnished)) {
                $query->where('furnishing', $furnished);
            }

            if (!empty($verified)) {
                $query->where('verified', 1);
            }

            // Get initial property count
            $total_properties = $query->count();

            // Get property types for filter dropdown
            $property_types = Type::where('status', 1)->orderBy('type_name')->get();

            // Get locations for filter dropdown
            $locations = Location::where('status', 1)->orderBy('name')->get();

            return view('pages.property.map', compact(
                'total_properties',
                'property_types',
                'locations',
                'search',
                'type_id',
                'location_id',
                'purpose',
                'bedrooms',
                'bathrooms',
                'price_min',
                'price_max',
                'furnished',
                'verified'
            ));
        } catch (\Exception $e) {
            Log::error('Map view error', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            // Fallback to basic view without data
            return view('pages.property.map', [
                'total_properties' => 0,
                'property_types' => collect([]),
                'locations' => collect([]),
                'search' => '',
                'type_id' => '',
                'location_id' => '',
                'purpose' => '',
                'bedrooms' => '',
                'bathrooms' => '',
                'price_min' => '',
                'price_max' => '',
                'furnished' => '',
                'verified' => ''
            ]);
        }
    }

    /**
     * Render comparison page with selected properties
     */
    public function compareView(Request $request)
    {
        try {
            $idsParam = $request->query('ids', '');
            if (is_string($idsParam)) {
                // Parse comma-separated ids string like "1,2,3"
                $idArray = array_filter(array_map('trim', explode(',', $idsParam)), function($v) { return $v !== ''; });
            } else if (is_array($idsParam)) {
                $idArray = $idsParam;
            } else {
                $idArray = [];
            }

            // Coerce to integers and ensure unique, limit to 4
            $propertyIds = array_values(array_unique(array_map(function($v){ return (int)$v; }, $idArray)));
            if (empty($propertyIds)) {
                return redirect()->to(url('properties'))
                    ->with('error', 'Please select at least two properties to compare.');
            }

            $properties = Property::with(['types', 'locations', 'users'])
                ->whereIn('id', $propertyIds)
                ->where('status', 1)
                ->get();

            if ($properties->count() < 2) {
                return redirect()->to(url('properties'))
                    ->with('error', 'Please select at least two properties to compare.');
            }

            return view('pages.property.compare', [
                'properties' => $properties,
                'ids' => $propertyIds
            ]);
        } catch (\Throwable $e) {
            return redirect()->to(url('properties'))
                ->with('error', 'Unable to load comparison page.');
        }
    }

    /**
     * Get map data for AJAX requests with enhanced caching and debugging
     */
    public function getMapData(Request $request)
    {
        try {
            // Validate and sanitize input parameters
            $request->validate([
                'search' => 'nullable|string|max:255',
                'type_id' => 'nullable|integer|exists:types,id',
                'location_id' => 'nullable|integer|exists:locations,id',
                'purpose' => 'nullable|in:Sale,Rent',
                'bedrooms' => 'nullable|integer|min:0|max:10',
                'bathrooms' => 'nullable|integer|min:0|max:10',
                'price_min' => 'nullable|numeric|min:0',
                'price_max' => 'nullable|numeric|min:0',
                'furnishing' => 'nullable|in:Furnished,Semi-Furnished,Unfurnished',
                'verified' => 'nullable|boolean',
                'lat' => 'nullable|numeric|between:-90,90',
                'lng' => 'nullable|numeric|between:-180,180',
                'radius' => 'nullable|numeric|min:0|max:100', // Made optional
            ]);

            // Get filter parameters
            $filters = $request->only([
                'search', 'type_id', 'location_id', 'purpose', 'bedrooms',
                'bathrooms', 'price_min', 'price_max', 'furnishing', 'verified',
                'lat', 'lng', 'radius'
            ]);

            // Sanitize search input
            if (!empty($filters['search'])) {
                $filters['search'] = strip_tags(trim($filters['search']));
            }

            // Debug logging
            Log::info('Map data request received', [
                'filters' => $filters,
                'request_params' => $request->all(),
                'user_agent' => $request->header('User-Agent'),
                'ip' => $request->ip()
            ]);

            // Create cache key based on filters
            $cacheKey = 'map_data_' . md5(serialize($filters));

            // Cache results for 30 minutes
            $properties = Cache::remember($cacheKey, 1800, function () use ($filters) {
                return $this->getPropertiesFromDatabase($filters);
            });

            // Check if we have properties with coordinates
            $propertiesWithCoords = collect($properties)->filter(function($p) {
                return isset($p['lat']) && isset($p['lng']) &&
                       is_numeric($p['lat']) && is_numeric($p['lng']) &&
                       $p['lat'] >= -90 && $p['lat'] <= 90 &&
                       $p['lng'] >= -180 && $p['lng'] <= 180;
            });

            if ($propertiesWithCoords->count() === 0 && count($properties) > 0) {
                Log::warning('No properties have valid coordinates', [
                    'total_properties' => count($properties),
                    'filters' => $filters
                ]);
            }

            Log::info('Map data request processed', [
                'filters' => $filters,
                'results_count' => count($properties),
                'coordinates_count' => $propertiesWithCoords->count(),
                'cached' => true
            ]);

            return response()->json([
                'data' => $properties,
                'total' => count($properties),
                'coordinates_count' => $propertiesWithCoords->count(),
                'filters_applied' => $filters,
                'cached' => true,
                'debug' => config('app.debug') ? [
                    'cache_key' => $cacheKey,
                    'filters' => $filters,
                    'execution_time' => microtime(true) - LARAVEL_START
                ] : null
            ]);
        } catch (\Throwable $e) {
            Log::error('Map data request error', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'request_data' => $request->all()
            ]);

            return response()->json([
                'error' => true,
                'message' => 'An error occurred while fetching map data',
                'debug' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Get properties from database with all filtering logic
     */
    private function getPropertiesFromDatabase($filters)
    {
        try {
            // Start with base query
            $query = Property::where('status', 1)
                ->where('approval_status', 'approved')
                ->whereNotNull('latitude')
                ->whereNotNull('longitude')
                ->where('latitude', '!=', '')
                ->where('longitude', '!=', '');

            // Apply filters with logging
            if (!empty($filters['search'])) {
                $query->where('title', 'LIKE', '%' . $filters['search'] . '%');
                Log::debug('Applied search filter', ['search' => $filters['search']]);
            }

            if (!empty($filters['type_id'])) {
                $query->where('type_id', $filters['type_id']);
                Log::debug('Applied type filter', ['type_id' => $filters['type_id']]);
            }

            if (!empty($filters['location_id'])) {
                $query->where('location_id', $filters['location_id']);
                Log::debug('Applied location filter', ['location_id' => $filters['location_id']]);
            }

            if (!empty($filters['purpose'])) {
                $query->where('purpose', $filters['purpose']);
                Log::debug('Applied purpose filter', ['purpose' => $filters['purpose']]);
            }

            if (!empty($filters['bedrooms'])) {
                if ($filters['bedrooms'] == 4) {
                    $query->where('bedrooms', '>=', $filters['bedrooms']);
                } else {
                    $query->where('bedrooms', $filters['bedrooms']);
                }
                Log::debug('Applied bedrooms filter', ['bedrooms' => $filters['bedrooms']]);
            }

            if (!empty($filters['bathrooms'])) {
                if ($filters['bathrooms'] == 4) {
                    $query->where('bathrooms', '>=', $filters['bathrooms']);
                } else {
                    $query->where('bathrooms', $filters['bathrooms']);
                }
                Log::debug('Applied bathrooms filter', ['bathrooms' => $filters['bathrooms']]);
            }

            if (!empty($filters['furnishing'])) {
                $query->where('furnishing', $filters['furnishing']);
                Log::debug('Applied furnishing filter', ['furnishing' => $filters['furnishing']]);
            }

            if (!empty($filters['verified'])) {
                $query->where('verified', $filters['verified']);
                Log::debug('Applied verified filter', ['verified' => $filters['verified']]);
            }

            // Price range filter
            if (!empty($filters['price_min'])) {
                $query->where('price', '>=', $filters['price_min']);
                Log::debug('Applied price_min filter', ['price_min' => $filters['price_min']]);
            }

            if (!empty($filters['price_max'])) {
                $query->where('price', '<=', $filters['price_max']);
                Log::debug('Applied price_max filter', ['price_max' => $filters['price_max']]);
            }

            // Location-based filtering (radius search) - Only apply when explicitly requested
            if (
                isset($filters['lat'], $filters['lng'], $filters['radius']) &&
                is_numeric($filters['lat']) && is_numeric($filters['lng']) && is_numeric($filters['radius']) &&
                $filters['radius'] > 0 // Only apply radius filter when explicitly set to a positive value
            ) {
                $lat = (float) $filters['lat'];
                $lng = (float) $filters['lng'];
                $radius = (float) $filters['radius']; // in kilometers
                $query->selectRaw(
                    '*, (6371 * acos(cos(radians(?)) * cos(radians(latitude)) * cos(radians(longitude) - radians(?)) + sin(radians(?)) * sin(radians(latitude)))) AS distance',
                    [$lat, $lng, $lat]
                )
                ->having('distance', '<=', $radius)
                ->orderBy('distance');

                Log::debug('Applied radius filter', [
                    'lat' => $lat,
                    'lng' => $lng,
                    'radius' => $radius
                ]);
            }

            // Get count before executing query
            $totalCount = $query->count();
            Log::debug('Query count before execution', ['count' => $totalCount]);

            // Get properties with relationships - optimized query
            $properties = $query->with(['types:id,type_name', 'locations:id,name', 'users:id,name,phone,email,user_image'])
                ->select('id', 'slug', 'title', 'latitude', 'longitude', 'price', 'purpose', 'image',
                        'bedrooms', 'bathrooms', 'area', 'furnishing', 'verified', 'address',
                        'type_id', 'location_id', 'user_id')
                ->orderBy('id', 'DESC')
                ->get(); // Removed limit for better performance - show all properties

            Log::debug('Properties fetched from database', [
                'total_found' => $properties->count(),
                'filters' => $filters
            ]);

            // Check if we got any properties
            if ($properties->count() === 0) {
                Log::info('No properties found with current filters', ['filters' => $filters]);
            }

            // Transform properties to array format
            $propertiesArray = $properties->map(function ($p) {
                return [
                    'id' => $p->id,
                    'title' => $p->title,
                    'slug' => $p->slug,
                    'lat' => (float) $p->latitude,
                    'lng' => (float) $p->longitude,
                    'price' => $p->price,
                    'purpose' => $p->purpose,
                    'image' => asset('/' . ltrim($p->image, '/')),
                    'url' => url('properties/' . $p->slug . '/' . $p->id),
                        'bedrooms' => $p->bedrooms,
                        'bathrooms' => $p->bathrooms,
                        'area' => $p->area,
                        'furnishing' => $p->furnishing,
                        'verified' => $p->verified,
                        'address' => $p->address,
                        'type' => $p->types ? $p->types->type_name : null,
                        'location' => $p->locations ? $p->locations->name : null,
                        'agent' => $p->users ? [
                            'id' => $p->users->id,
                            'name' => $p->users->name,
                            'phone' => $p->users->phone,
                            'email' => $p->users->email,
                            'image' => $p->users->user_image ? asset('/upload/' . $p->users->user_image) : null
                        ] : null,
                        'distance' => isset($p->distance) ? round($p->distance, 2) : null
                ];
            })->toArray();

            Log::debug('Properties transformed to array', [
                'total_properties' => count($propertiesArray),
                'first_property' => $propertiesArray[0] ?? null
            ]);

            return $propertiesArray;

        } catch (\Exception $e) {
            Log::error('Database query error in getPropertiesFromDatabase', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'filters' => $filters
            ]);

            // Return empty array instead of throwing error
            return [];
        }
    }

    /**
     * Get property details for map popup
     */
    public function getPropertyDetails($id)
    {
        try {
            $property = Property::with(['types', 'locations', 'users'])
                ->where('status', 1)
                ->where('approval_status', 'approved')
                ->where('id', $id)
                ->first();

            if (!$property) {
                Log::warning('Property not found for map details', ['property_id' => $id]);
                return response()->json(['error' => 'Property not found'], 404);
            }

            Log::info('Property details requested', ['property_id' => $id]);

            return response()->json([
                'id' => $property->id,
                'title' => $property->title,
                'slug' => $property->slug,
                'price' => $property->price,
                'purpose' => $property->purpose,
                'description' => $property->description,
                'bedrooms' => $property->bedrooms,
                'bathrooms' => $property->bathrooms,
                'area' => $property->area,
                'furnishing' => $property->furnishing,
                'verified' => $property->verified,
                'address' => $property->address,
                'latitude' => $property->latitude,
                'longitude' => $property->longitude,
                'image' => asset('/' . ltrim($property->image, '/')),
                'type' => $property->types ? $property->types->type_name : null,
                'location' => $property->locations ? $property->locations->name : null,
                'agent' => $property->users ? [
                    'id' => $property->users->id,
                    'name' => $property->users->name,
                    'phone' => $property->users->phone,
                    'email' => $property->users->email,
                    'image' => $property->users->user_image ? asset('/upload/' . $property->users->user_image) : null
                ] : null,
                'galleries' => [],
                'url' => url('properties/' . $property->slug . '/' . $property->id)
            ]);
        } catch (\Throwable $e) {
            Log::error('Property details error', [
                'property_id' => $id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'error' => true,
                'message' => 'An error occurred while fetching property details'
            ], 500);
        }
    }

    /**
     * Toggle favorite property
     */
    public function toggleFavorite(Request $request)
    {
        try {
            $request->validate([
                'property_id' => 'required|integer|exists:property,id'
            ]);

            $user = Auth::user();
            $propertyId = $request->property_id;

            // Check if already favorited
            $favorite = Favourite::where('user_id', $user->id)
                ->where('post_id', $propertyId)
                ->where('post_type', 'Property')
                ->first();

            if ($favorite) {
                // Remove favorite
                $favorite->delete();
                return response()->json([
                    'success' => true,
                    'action' => 'removed',
                    'message' => 'Property removed from favorites'
                ]);
            } else {
                // Add favorite
                Favourite::create([
                    'user_id' => $user->id,
                    'post_id' => $propertyId,
                    'post_type' => 'Property'
                ]);
                return response()->json([
                    'success' => true,
                    'action' => 'added',
                    'message' => 'Property added to favorites'
                ]);
            }
        } catch (\Throwable $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get comparison data
     */
    public function getCompareData(Request $request)
    {
        try {
            $propertyIds = $request->input('ids', []);
            if (empty($propertyIds)) {
                return response()->json(['error' => 'No property IDs provided'], 400);
            }

            $properties = Property::with(['types', 'locations', 'users'])
                ->whereIn('id', $propertyIds)
                ->where('status', 1)
                ->get();

            return response()->json([
                'success' => true,
                'properties' => $properties->map(function($property) {
                    return [
                        'id' => $property->id,
                        'title' => $property->title,
                        'price' => $property->price,
                        'purpose' => $property->purpose,
                        'bedrooms' => $property->bedrooms,
                        'bathrooms' => $property->bathrooms,
                        'area' => $property->area,
                        'image' => asset('/' . ltrim($property->image, '/')),
                        'type' => $property->types ? $property->types->type_name : null,
                        'location' => $property->locations ? $property->locations->name : null,
                        'agent' => $property->users ? $property->users->name : null
                    ];
                })
            ]);
        } catch (\Throwable $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Search locations by query with caching
     */
    public function searchByLocation(Request $request)
    {
        try {
            $query = $request->get('q', '');
            if (empty($query)) {
                return response()->json([]);
            }

            // Cache location search results for 1 hour
            $cacheKey = 'location_search_' . md5($query);
            $locations = Cache::remember($cacheKey, 3600, function () use ($query) {
                return Location::where('status', 1)
                    ->where('name', 'LIKE', '%' . $query . '%')
                    ->select('id', 'name', 'latitude', 'longitude')
                    ->limit(10)
                    ->get();
            });

            Log::info('Location search performed', [
                'query' => $query,
                'results_count' => $locations->count()
            ]);

            return response()->json($locations);
        } catch (\Throwable $e) {
            Log::error('Location search error', [
                'query' => $query,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'error' => true,
                'message' => 'An error occurred while searching locations'
            ], 500);
        }
    }
}
