<?php

namespace App\Http\Controllers;

use Auth;
use App\User;
use App\Property;
use App\Models\PostProperty;
use App\PropertyGallery;
use App\Reports;
use App\Meeting;
use App\Type;
use App\Location;
use Carbon\Carbon;

use App\Http\Controllers\Controller;
use Session;
use Illuminate\Http\Request;
use App\Mail\MeetingScheduled;
use App\Mail\MeetingConfirmed;
use App\Mail\MeetingDeclined;
use Illuminate\Support\Facades\Mail;
use Validator;

class PropertyController extends Controller
{

    public function __construct()
    {
        $this->pagination_limit = getcong('pagination_limit') ? getcong('pagination_limit') : 10;
    }

    public function properties(Request $request)
    {

        $sort_by = $request->input('sort_by', 'New');

        $property_list = Property::where('status', 1);

        switch ($sort_by) {
            case 'Old':
                $property_list = $property_list->orderBy('id', 'ASC');
                break;

            case 'High':
                $property_list = $property_list->orderBy('price', 'DESC');
                break;

            case 'Low':
                $property_list = $property_list->orderBy('price', 'ASC');
                break;

            case 'New':
            default:
                $property_list = $property_list->orderBy('id', 'DESC');
                break;
        }

        $property_list = $property_list->with(['types', 'locations', 'users'])->paginate(10)->appends(['sort_by' => $sort_by]);


        return view('pages.property.list', compact('property_list'));
    }
    
     public function map_data(Request $request)
    {
        try {
            // Validate and sanitize input parameters
            $request->validate([
                'search' => 'nullable|string|max:255',
                'type_id' => 'nullable|integer|exists:types,id',
                'location_id' => 'nullable|integer|exists:locations,id',
                'purpose' => 'nullable|in:Sale,Rent',
                'bedrooms' => 'nullable|integer|min:0|max:10',
                'bathrooms' => 'nullable|integer|min:0|max:10',
                'price_min' => 'nullable|numeric|min:0',
                'price_max' => 'nullable|numeric|min:0',
                'furnishing' => 'nullable|in:Furnished,Semi-Furnished,Unfurnished',
                'verified' => 'nullable|boolean',
                'lat' => 'nullable|numeric|between:-90,90',
                'lng' => 'nullable|numeric|between:-180,180',
                'radius' => 'nullable|numeric|min:0.1|max:100',
                'bounds' => 'nullable|json'
            ]);

            // Get filter parameters
            $filters = $request->only([
                'search', 'type_id', 'location_id', 'purpose', 'bedrooms', 
                'bathrooms', 'price_min', 'price_max', 'furnishing', 'verified',
                'lat', 'lng', 'radius', 'bounds'
            ]);

            // Sanitize search input
            if (!empty($filters['search'])) {
                $filters['search'] = strip_tags(trim($filters['search']));
            }

            // Start with base query
            $query = Property::where('status', 1)
            ->where('approval_status', 'approved')
            ->whereNotNull('latitude')
            ->whereNotNull('longitude')
            ->where('latitude', '!=', '')
                ->where('longitude', '!=', '');

            // Apply filters
            if (!empty($filters['search'])) {
                $query->where('title', 'LIKE', '%' . $filters['search'] . '%');
            }
            if (!empty($filters['type_id'])) {
                $query->where('type_id', $filters['type_id']);
            }
            if (!empty($filters['location_id'])) {
                $query->where('location_id', $filters['location_id']);
            }
            if (!empty($filters['purpose'])) {
                $query->where('purpose', $filters['purpose']);
            }
            if (!empty($filters['bedrooms'])) {
                if ($filters['bedrooms'] == 4) {
                    $query->where('bedrooms', '>=', $filters['bedrooms']);
                } else {
                    $query->where('bedrooms', $filters['bedrooms']);
                }
            }
            if (!empty($filters['bathrooms'])) {
                if ($filters['bathrooms'] == 4) {
                    $query->where('bathrooms', '>=', $filters['bathrooms']);
                } else {
                    $query->where('bathrooms', $filters['bathrooms']);
                }
            }
            if (!empty($filters['furnishing'])) {
                $query->where('furnishing', $filters['furnishing']);
            }
            if (!empty($filters['verified'])) {
                $query->where('verified', $filters['verified']);
            }
            // Price range filter
            if (!empty($filters['price_min'])) {
                $query->where('price', '>=', $filters['price_min']);
            }
            if (!empty($filters['price_max'])) {
                $query->where('price', '<=', $filters['price_max']);
            }
            // Location-based filtering (radius search)
            if (
                isset($filters['lat'], $filters['lng'], $filters['radius']) &&
                is_numeric($filters['lat']) && is_numeric($filters['lng']) && is_numeric($filters['radius'])
            ) {
                $lat = (float) $filters['lat'];
                $lng = (float) $filters['lng'];
                $radius = (float) $filters['radius']; // in kilometers
                $query->selectRaw(
                    '*, (6371 * acos(cos(radians(?)) * cos(radians(latitude)) * cos(radians(longitude) - radians(?)) + sin(radians(?)) * sin(radians(latitude)))) AS distance',
                    [$lat, $lng, $lat]
                )
                ->having('distance', '<=', $radius)
                ->orderBy('distance');
            }
            // Bounds filtering (for map viewport)
            if (!empty($filters['bounds'])) {
                $bounds = json_decode($filters['bounds'], true);
                if (
                    isset($bounds['north'], $bounds['south'], $bounds['east'], $bounds['west']) &&
                    is_numeric($bounds['north']) && is_numeric($bounds['south']) &&
                    is_numeric($bounds['east']) && is_numeric($bounds['west'])
                ) {
                    $query->whereBetween('latitude', [$bounds['south'], $bounds['north']])
                          ->whereBetween('longitude', [$bounds['west'], $bounds['east']]);
                }
            }
            // Get properties with relationships - optimized query
            $properties = $query->with(['types:id,type_name', 'locations:id,name', 'users:id,name,phone,email,user_image'])
                ->select('id', 'slug', 'title', 'latitude', 'longitude', 'price', 'purpose', 'image', 
                        'bedrooms', 'bathrooms', 'area', 'furnishing', 'verified', 'address',
                        'type_id', 'location_id', 'user_id')
                ->orderBy('id', 'DESC')
                ->limit(500) // Reduced limit for better performance
            ->get()
            ->map(function ($p) {
                return [
                    'id' => $p->id,
                    'title' => $p->title,
                    'slug' => $p->slug,
                    'lat' => (float) $p->latitude,
                    'lng' => (float) $p->longitude,
                    'price' => $p->price,
                    'purpose' => $p->purpose,
                    'image' => url('/' . ltrim($p->image, '/')),
                    'url' => url('properties/' . $p->slug . '/' . $p->id),
                        'bedrooms' => $p->bedrooms,
                        'bathrooms' => $p->bathrooms,
                        'area' => $p->area,
                        'furnishing' => $p->furnishing,
                        'verified' => $p->verified,
                        'address' => $p->address,
                        'type' => $p->types ? $p->types->type_name : null,
                        'location' => $p->locations ? $p->locations->name : null,
                        'agent' => $p->users ? [
                            'id' => $p->users->id,
                            'name' => $p->users->name,
                            'phone' => $p->users->phone,
                            'email' => $p->users->email,
                            'image' => $p->users->user_image ? url('/upload/' . $p->users->user_image) : null
                        ] : null,
                        'distance' => isset($p->distance) ? round($p->distance, 2) : null
                ];
            });
        return response()->json([
            'data' => $properties,
                'total' => $properties->count(),
                'filters_applied' => $filters
            ]);
        } catch (\Throwable $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ], 500);
        }
    }
    
    /**
     * Display the dedicated map view for properties
     */
    public function map_view(Request $request)
    {
        // Get filter parameters
        $search = $request->get('search', '');
        $type_id = $request->get('type_id', '');
        $location_id = $request->get('location_id', '');
        $purpose = $request->get('purpose', '');
        $bedrooms = $request->get('bedrooms', '');
        $bathrooms = $request->get('bathrooms', '');
        $price_min = $request->get('price_min', '');
        $price_max = $request->get('price_max', '');
        $furnished = $request->get('furnished', '');
        $verified = $request->get('verified', '');

        // Build the query
        $query = Property::with(['type', 'location', 'user'])
                         ->where('status', 1)
                         ->whereNotNull('latitude')
                         ->whereNotNull('longitude')
                         ->where('latitude', '!=', '')
                         ->where('longitude', '!=', '');

        // Apply filters
        if (!empty($search)) {
            $query->where(function($q) use ($search) {
                $q->where('title', 'like', '%' . $search . '%')
                  ->orWhere('address', 'like', '%' . $search . '%')
                  ->orWhere('description', 'like', '%' . $search . '%');
            });
        }

        if (!empty($type_id)) {
            $query->where('type_id', $type_id);
        }

        if (!empty($location_id)) {
            $query->where('location_id', $location_id);
        }

        if (!empty($purpose)) {
            $query->where('purpose', $purpose);
        }

        if (!empty($bedrooms)) {
            $query->where('bedrooms', '>=', $bedrooms);
        }

        if (!empty($bathrooms)) {
            $query->where('bathrooms', '>=', $bathrooms);
        }

        if (!empty($price_min)) {
            $query->where('price', '>=', $price_min);
        }

        if (!empty($price_max)) {
            $query->where('price', '<=', $price_max);
        }

        if (!empty($furnished)) {
            $query->where('furnished', $furnished);
        }

        if (!empty($verified)) {
            $query->where('verified', 1);
        }

        // Get initial property count for display
        $total_properties = $query->count();

        // Get property types for filter dropdown
        $property_types = Type::where('status', 1)->orderBy('type_name')->get();

        // Get locations for filter dropdown
        $locations = Location::where('status', 1)->orderBy('name')->get();

        // Pass data to the view
        return view('pages.property.map', compact(
            'total_properties',
            'property_types',
            'locations',
            'search',
            'type_id',
            'location_id',
            'purpose',
            'bedrooms',
            'bathrooms',
            'price_min',
            'price_max',
            'furnished',
            'verified'
        ));
    }
    

    public function property_details($slug, $id)
    {
        $property_info = Property::where('status', 1)->where('id', $id)->first();

        if (empty($property_info)) {
            Session::flash('error_flash_message', trans('words.access_denied'));

            return redirect()->back();
        }

        $gallery_images = PropertyGallery::where('post_id', $property_info->id)->orderBy('id')->get();

        $latest_list = Property::where('status', 1)->orderby('id', 'DESC')->limit(5)->get();

        $type_id = $property_info->type_id;

        $related_list = Property::where('status', 1)->where('type_id', $type_id)->orderby('id', 'DESC')->limit(5)->get();

        $user_id = $property_info->user_id;

        //View Update
        post_views_save($id, 'Property');

        return view('pages.property.details', compact('property_info', 'gallery_images', 'related_list', 'latest_list', 'user_id'));
    }

    // public function get_purpose_list(Request $request)
    // {
    //     $type_id = $request->input('type_id');

    //     if ($type_id != '') {
    //         $purpose_list = Property::where('type_id', $type_id)->distinct()->pluck('purpose')->toArray();
    //     } else {
    //         $purpose_list = Property::where('type_id', '!=' , '')->distinct()->pluck('purpose')->toArray();
    //     }
    //     var_dump($purpose_list); exit;
    //     return response()->json($purpose_list);
    // }
    public function get_purpose_list(Request $request)
    {
        $typeId = $request->input('type_id');

        $q = Property::query();
        if (!empty($typeId)) {
            $q->where('type_id', $typeId);
        }

        $items = $q->whereNotNull('purpose')
            ->where('purpose', '<>', '')
            ->select('purpose')
            ->distinct()
            ->orderByRaw('LOWER(purpose) DESC')   // <= ASC sort
            ->pluck('purpose')
            ->toArray();

        $html = '';
        foreach ($items as $p) {
            $value = trim($p);
            $label = (strcasecmp($value, 'Sale') === 0) ? 'Buy' : $value;
            $html .= '<option value="' . e($value) . '">' . e($label) . '</option>';
        }

        return response($html, 200)->header('Content-Type', 'text/html; charset=utf-8');
    }



    public function property_search()
    {
        $property_query = Property::with(['types', 'locations', 'users'])->where('status', 1)
            ->where(function ($query) {
                // Search text
                if ($search_text = request()->get('search_text')) {
                    $query->where('title', 'LIKE', '%' . trim($search_text) . '%');
                }

                // Purpose
                if ($purpose = request()->get('purpose')) {
                    $query->where('purpose', $purpose);
                }

                // Type
                if ($type_id = request()->get('type_id')) {
                    $query->where('type_id', $type_id);
                }
                
                // Location (city) - prefer explicit param, else cookie default
                if (!empty($location_id)) {
                    $query->where('location_id', $location_id);
                }

                // Locality (city_id on property) - from new dropdown
                if ($city_id = request()->get('city_id')) {
                    $query->where('city_id', $city_id);
                }


                // Location
                if ($location_id = request()->get('location_id')) {
                    $query->where('location_id', $location_id);
                }

                // Bedrooms
                if ($bedrooms = request()->get('bedrooms')) {

                    if ($bedrooms == 4) {
                        $query->where('bedrooms', '>=', $bedrooms);
                    } else {
                        $query->where('bedrooms', $bedrooms);
                    }
                }

                // Bathrooms
                if ($bathrooms = request()->get('bathrooms')) {

                    if ($bathrooms == 4) {
                        $query->where('bathrooms', '>=', $bathrooms);
                    } else {
                        $query->where('bathrooms', $bathrooms);
                    }
                }

                // Furnishing
                if ($furnishing = request()->get('furnishing')) {
                    $query->where('furnishing', $furnishing);
                }

                // Verified
                if ($verified = request()->get('verified')) {
                    $query->where('verified', $verified);
                }

                // Price range

                if ($price_range = request()->get('price_range')) {

                    $price_parts = explode('-', $price_range);
                    $price_start = substr($price_parts[0], 1);
                    $price_end = substr($price_parts[1], 2);

                    $query->whereBetween('price', [$price_start, $price_end]);
                }
            });

        $total_records = $property_query->count();

        $property_list = $property_query->orderBy('id', 'DESC')->paginate($this->pagination_limit);

        $property_list->appends(request()->input())->links();

        $total_property = $total_records;


        return view('pages.property.search', compact('property_list', 'total_property'));
    }

    public function properties_contact(Request $request)
    {

        $data =  \Request::except(array('_token'));

        $inputs = $request->all();

        if (getcong('recaptcha_on_contact_us')) {
            $rule = array(
                'name' => 'required',
                'email' => 'required|email|max:100',
                'g-recaptcha-response' => 'required'
            );
        } else {
            $rule = array(
                'name' => 'required',
                'email' => 'required|email|max:100'
            );
        }



        $validator = \Validator::make($data, $rule);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator->messages());
        }

        //check reCaptcha
        if (getcong('recaptcha_on_contact_us')) {

            $recaptcha_response = $inputs['g-recaptcha-response'];

            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, "https://www.google.com/recaptcha/api/siteverify");
            curl_setopt($ch, CURLOPT_HEADER, 0);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
            curl_setopt($ch, CURLOPT_POST, 1);
            curl_setopt($ch, CURLOPT_POSTFIELDS, [
                'secret' => getcong('recaptcha_secret_key'),
                'response' => $recaptcha_response,
                'remoteip' => $_SERVER['REMOTE_ADDR']
            ]);

            $resp = json_decode(curl_exec($ch));
            curl_close($ch);


            if ($resp->success != true) {

                Session::flash('error_flash_message', 'Captcha timeout or duplicate');
                return \Redirect::back();
            }
        }

        $user_id = $inputs['property_owner_id'];

        $user_info = User::where('id', $user_id)->first();

        $to_name = $user_info->name;
        $to_email = $user_info->email;

        $data = array(
            'name' => $inputs['name'],
            'email' => $inputs['email'],
            'phone' => $inputs['phone'],
            'user_message' => $inputs['message'],
        );

        $subject = $inputs['property_title'] . ' - Contact';

        $from_email = getenv("MAIL_FROM_ADDRESS");

        try {

            \Mail::send('emails.property_contact', $data, function ($message) use ($subject, $from_email, $to_name, $to_email) {

                $message->from($from_email, getcong('site_name'));

                $message->to($to_email, $to_name)->subject($subject);
            });
        } catch (\Throwable $e) {

            \Log::info($e->getMessage());

            Session::flash('flash_message', $e->getMessage());
            return \Redirect::back();
        }

        Session::flash('flash_message', trans('words.contact_msg'));
        return \Redirect::back();
    }

    public function properties_report(Request $request)
    {

        $data =  \Request::except(array('_token'));

        $inputs = $request->all();

        $rule = array(
            'report_text' => 'required'
        );


        $validator = \Validator::make($data, $rule);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator->messages());
        }

        $user_id = Auth::User()->id;

        $post_id = $inputs['property_id'];
        $message = $inputs['report_text'];

        $re_obj = new Reports;

        $re_obj->post_type = 'Property';
        $re_obj->post_id = $post_id;
        $re_obj->user_id = $user_id;
        $re_obj->message = $message;
        $re_obj->date = strtotime(date('m/d/Y H:i:s'));
        $re_obj->save();

        Session::flash('flash_message', trans('words.reports_success'));
        return \Redirect::back();
    }

    public function properties_owner($owner_id)
    {

        $user_info = User::where('id', $owner_id)->first();

        $property_list = $user_info->userproperty()->where('status', 1)->orderby('id', 'DESC')->paginate(10);

        $user_plan_id = $user_info->plan_id;
        $user_plan_exp_date = $user_info->exp_date;


        if ($user_info->usertype == "User") {
            if ($user_plan_id == 0 or strtotime(date('m/d/Y')) >= $user_plan_exp_date) {
                Session::flash('error_flash_message', trans('words.access_denied'));

                return \Redirect::back();
            }
        }

        $latest_list = Property::with(['types', 'locations', 'users'])->where('status', 1)->orderby('id', 'DESC')->limit(5)->get();


        return view('pages.property.owner_list', compact('property_list', 'owner_id', 'latest_list'));
    }

    public function schedule_meeting(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'property_id' => 'required|exists:property,id',
            'agent_id' => 'required|exists:users,id',
            'meeting_date' => 'required|date|after_or_equal:today',
            'meeting_time' => 'required|date_format:H:i',
            'message' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        $property = Property::findOrFail($request->property_id);
        $agent = User::findOrFail($request->agent_id);
        $user = Auth::user();

        if (!$user) {
            Session::flash('error_flash_message', trans('words.login_required_for_meeting'));
            return redirect()->back();
        }

        $meeting = Meeting::create([
            'property_id' => $property->id,
            'user_id' => $user->id,
            'agent_id' => $agent->id,
            'meeting_date' => $request->meeting_date,
            'meeting_time' => $request->meeting_time,
            'message' => $request->message,
            'status' => 'pending',
        ]);

        // Send email notification to the agent
        try {
            Mail::to($agent->email)->send(new MeetingScheduled($meeting, $property, $user, $agent));
        } catch (\Exception $e) {
            \Log::error("Error sending meeting scheduled email to agent: " . $e->getMessage());
        }

        // Send confirmation email to the user
        try {
            Mail::to($user->email)->send(new MeetingScheduled($meeting, $property, $user, $agent));
        } catch (\Exception $e) {
            \Log::error("Error sending meeting scheduled email to user: " . $e->getMessage());
        }

        Session::flash('flash_message', trans('words.meeting_request_sent'));
        return redirect()->back();
    }

    public function confirm_meeting($id)
    {
        $meeting = Meeting::findOrFail($id);

        // Only the agent or admin can confirm a meeting
        if (Auth::user()->id !== $meeting->agent_id && Auth::user()->usertype !== 'Admin') {
            Session::flash('error_flash_message', trans('words.unauthorized_action'));
            return redirect()->back();
        }

        $meeting->status = 'confirmed';
        $meeting->save();

        $property = Property::findOrFail($meeting->property_id);
        $user = User::findOrFail($meeting->user_id);
        $agent = User::findOrFail($meeting->agent_id);

        // Send email notification to the user about confirmation
        try {
            Mail::to($user->email)->send(new MeetingConfirmed($meeting, $property, $user, $agent));
        } catch (\Exception $e) {
            \Log::error("Error sending meeting confirmed email to user: " . $e->getMessage());
        }

        Session::flash('flash_message', trans('words.meeting_confirmed'));
        return redirect()->back();
    }

    public function decline_meeting(Request $request, $id)
    {
        $meeting = Meeting::findOrFail($id);

        // Only the agent or admin can decline a meeting
        if (Auth::user()->id !== $meeting->agent_id && Auth::user()->usertype !== 'Admin') {
            Session::flash('error_flash_message', trans('words.unauthorized_action'));
            return redirect()->back();
        }

        $validator = Validator::make($request->all(), [
            'reason' => 'required|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        $meeting->status = 'declined';
        $meeting->message = $request->reason; // Store the reason for declining in the message field
        $meeting->save();

        $property = Property::findOrFail($meeting->property_id);
        $user = User::findOrFail($meeting->user_id);
        $agent = User::findOrFail($meeting->agent_id);

        // Send email notification to the user about declination
        try {
            Mail::to($user->email)->send(new MeetingDeclined($meeting, $property, $user, $agent, $request->reason));
        } catch (\Exception $e) {
            \Log::error("Error sending meeting declined email to user: " . $e->getMessage());
        }

        Session::flash('flash_message', trans('words.meeting_declined'));
        return redirect()->back();
    }
    
    public function post_property()
    {

        return view('pages.post_property');
    }
    
    public function post_property_submit(Request $request)
    {
        // Validation rules
        $rules = [
            'property_purpose' => 'required|in:sale,rent',
            'property_category' => 'required|in:residential,commercial,land',
            'property_type' => 'required_if:property_category,commercial',
            'name' => 'required|string|min:3|max:50|regex:/^[a-zA-Z ]{3,50}$/',
            'email' => 'required|email',
            'mobile' => 'required|regex:/^[6-9]\d{9}$/',
            'city' => 'required'
        ];

        // Custom validation messages
        $messages = [
            'property_purpose.required' => 'Please select a property purpose',
            'property_category.required' => 'Please select a property category',
            'property_type.required_if' => 'Please select a property type for commercial properties',
            'name.required' => 'Name is required',
            'name.regex' => 'Please enter a valid name (3-50 characters, letters only)',
            'email.required' => 'Email is required',
            'email.email' => 'Please enter a valid email address',
            'mobile.required' => 'Mobile number is required',
            'mobile.regex' => 'Please enter a valid 10-digit mobile number',
            'city.required' => 'Please select a city'
        ];

        // Validate the request
        $validator = Validator::make($request->all(), $rules, $messages);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            // Create the property
            $property = new PostProperty();
            // $property->user_id = $user->id;
            $property->property_purpose = $request->property_purpose;
            $property->property_category = $request->property_category;
            if ($request->property_category === 'commercial' && $request->has('property_type')) {
                $property->property_type = $request->property_type;
                $property->title = ucfirst($request->property_type) . ' for ' . ucfirst($request->property_purpose);
            } else {
                $property->title = ucfirst($request->property_category) . ' Property for ' . ucfirst($request->property_purpose);
            }
            $property->name = $request->name;
            $property->email = $request->email;
            $property->mobile = $request->mobile;
            $property->city = $request->city;
            $property->status = 'new'; // Pending
            $property->date = strtotime(date('m/d/Y H:i:s'));
            $property->save();

            return response()->json([
                'status' => 'success',
                'message' => 'Property posted successfully! Our team will contact you shortly.',
            ]);
        } catch (\Exception $e) {
            Log::error('Property submission error: ' . $e->getMessage());

            return response()->json([
                'status' => 'error',
                'message' => 'An error occurred while processing your request. Please try again.',
            ], 500);
        }
    }
}
