<?php

namespace App\Http\Controllers;

use Auth;
use App\User;
use App\SubscriptionPlan;
use App\Transactions;
use App\Property;
use App\Favourite;
use Carbon\Carbon;
use App\Models\Lead;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Intervention\Image\Facades\Image;
use Illuminate\Support\Str;
use Session;

class UserController extends Controller
{

    public function dashboard()
    {
        if (!Auth::check()) {

            Session::flash('error_flash_message', trans('words.access_denied'));

            return redirect('login');
        }

        if (Auth::user()->usertype == 'Admin' or Auth::user()->usertype == 'Sub_Admin') {
            return redirect('admin/dashboard');
        }

        $user_id = Auth::user()->id;
        $user = User::findOrFail($user_id);

        $property_total = Property::where('user_id', $user_id)->count();
        $property_active = Property::where('status', 1)->where('user_id', $user_id)->count();
        $property_pending = Property::where('status', 0)->where('user_id', $user_id)->count();

        $favourite_total = Favourite::where('user_id', $user_id)->count();

        $transactions_list = Transactions::where('user_id', $user_id)->orderBy('id', 'DESC')->paginate(10);

        // Latest custom plan request (if any)
        $latest_cpr = \App\CustomPlanRequest::where('user_id', $user_id)->orderByDesc('id')->first();

        // Usage stats
        $property_limit = 0;
        // We'll compute properties_used since the latest activation date (CPR activated_at or user's start_date)
        $properties_used = 0;
        $properties_left = 0;
        $ads_limit = null;
        $ads_used = null;
        $ads_left = null;
        // No-expiry flow: do not compute or show expiry
        $expiry_date = null;

        // Determine latest activated CPR once
        $activeCpr = \App\CustomPlanRequest::where('user_id', $user_id)
            ->where(function($q){
                $q->whereIn('status', ['activated','ACTIVATED'])
                  ->orWhereNotNull('activated_at');
            })
            ->orderByDesc('activated_at')
            ->orderByDesc('id')
            ->first();

        // Base from plan if present
        $plan = $user->plan_id ? SubscriptionPlan::find($user->plan_id) : null;
        if ($plan) {
            $property_limit = (int) $plan->plan_property_limit;
            if (isset($plan->plan_ads_limit)) {
                $ads_limit = $plan->plan_ads_limit;
            }
        }

        // Override from active CPR if available
        if ($activeCpr) {
            $cprProp = (int)($activeCpr->approved_property_limit ?? 0);
            $cprAds  = $activeCpr->approved_ads_limit !== null ? (int)$activeCpr->approved_ads_limit : null;
            if ($cprProp === 0) {
                $cprProp = (int)($activeCpr->property_posts_requested ?? 0);
            }
            if ($cprAds === null) {
                $cprAds = $activeCpr->ads_posts_requested !== null ? (int)$activeCpr->ads_posts_requested : null;
            }
            // Prefer the plan's stored limits (may include carry-over) but ensure it's at least the CPR amount
            if ($cprProp > 0) {
                $property_limit = max((int)$property_limit, (int)$cprProp);
            }
            if ($cprAds !== null) {
                $ads_limit = $ads_limit !== null ? max((int)$ads_limit, (int)$cprAds) : (int)$cprAds;
            }
        }

        // Determine activation baseline: prefer CPR activation time, else user's plan start_date
        $activationStart = null;
        if ($activeCpr && !empty($activeCpr->activated_at)) {
            $activationStart = $activeCpr->activated_at; // already a Carbon instance via casts
        } elseif (!empty($user->start_date)) {
            $activationStart = Carbon::createFromTimestamp($user->start_date);
        }

        // Usage = Approved since activation + Pending created since activation
        $approved_since_activation = Property::where('user_id', $user_id)
            ->where('status', 1)
            ->where('approval_status', 'approved')
            ->when($activationStart, function ($q) use ($activationStart) {
                $q->whereNotNull('approved_at')
                  ->where('approved_at', '>=', $activationStart);
            })
            ->count();
        $pending_since_activation = Property::where('user_id', $user_id)
            ->where(function($q){
                $q->where('approval_status', 'pending')->orWhereNull('approved_at');
            })
            ->when($activationStart, function ($q) use ($activationStart) {
                $q->whereNotNull('created_at')
                  ->where('created_at', '>=', $activationStart);
            })
            ->count();
        $properties_used = $approved_since_activation + $pending_since_activation;

        $properties_left = max(0, (int)$property_limit - (int)$properties_used);
        // If you add an Ads model later, compute $ads_used
        $ads_used = null;
        $ads_left = $ads_limit !== null ? max(0, $ads_limit - (int) ($ads_used ?: 0)) : null;

        // Load recent leads for properties owned by this user (show latest 10)
        try {
            $leads = Lead::whereHas('property', function ($q) use ($user_id) {
                $q->where('user_id', $user_id);
            })->with('property')->orderBy('created_at', 'desc')->take(10)->get();
        } catch (\Exception $e) {
            $leads = collect();
        }

        return view('pages.user.dashboard', compact(
            'user',
            'property_total',
            'property_active',
            'property_pending',
            'favourite_total',
            'transactions_list',
            'leads',
            'property_limit',
            'properties_used',
            'properties_left',
            'ads_limit',
            'ads_used',
            'ads_left',
            'expiry_date',
            'latest_cpr'
        ));
    }

    public function profile()
    {

        if (!Auth::check()) {

            Session::flash('error_flash_message', trans('words.access_denied'));

            return redirect('login');
        }

        if (Auth::User()->usertype == "Admin" or Auth::User()->usertype == "Sub_Admin") {
            return redirect('admin');
        }

        $user_id = Auth::user()->id;
        $user = User::findOrFail($user_id);

        return view('pages.user.profile', compact('user'));
    }


    public function editprofile(Request $request)
    {

        $id = Auth::user()->id;
        $user = User::findOrFail($id);

        $data =  \Request::except(array('_token'));

        $rule = array(
            'name' => 'required',
            'email' => 'required|email|max:255|unique:users,email,' . $id,
            'user_image' => 'mimes:jpg,jpeg,gif,png'
        );

        $validator = \Validator::make($data, $rule);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator->messages());
        }


        $inputs = $request->all();

        $icon = $request->file('user_image');


        if ($icon) {

            \File::delete(public_path('/upload/') . $user->user_image);

            $tmpFilePath = public_path('/upload/');

            $hardPath =  Str::slug($inputs['name'], '-') . '-' . md5(time());

            $img = Image::make($icon);

            $img->fit(250, 250)->save($tmpFilePath . $hardPath . '-b.jpg');

            $user->user_image = $hardPath . '-b.jpg';
        }


        $user->name = $inputs['name'];
        $user->email = $inputs['email'];
        $user->phone = $inputs['phone'];

        if ($inputs['password']) {
            $user->password = bcrypt($inputs['password']);
        }

        $user->save();

        Session::flash('flash_message', trans('words.successfully_updated'));

        return redirect()->back();
    }

    public function phone_update(Request $request)
    {
        $id = Auth::user()->id;
        $user = User::findOrFail($id);

        $data =  \Request::except(array('_token'));

        $rule = array(
            'phone' => 'required'
        );

        $validator = \Validator::make($data, $rule);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator->messages());
        }


        $inputs = $request->all();

        $user->phone = $inputs['phone'];
        $user->save();

        Session::flash('flash_message', trans('words.successfully_updated'));

        return redirect()->back();
    }

    public function pricing()
    {
        // Show only public (non-custom) plans here
        $plan_list = SubscriptionPlan::where('status', '1')
            ->where(function($q){
                $q->whereNull('is_custom')->orWhere('is_custom', 0);
            })
            ->orderby('id')
            ->get();

        return view('pages.payment.pricing', compact('plan_list'));
    }

    public function payment_method($plan_id)
    {

        if (!Auth::check()) {
            Session::flash('error_flash_message', trans('words.access_denied'));
            return redirect('login');
        }
        if (Auth::User()->usertype == "Admin" or Auth::User()->usertype == "Sub_Admin") {
            return redirect('admin');
        }

        $plan_info = SubscriptionPlan::where('id', $plan_id)->where('status', '1')->first();

        if (!$plan_info) {
            Session::flash('flash_message', 'Select plan!');
            return redirect('pricing');
        }

        // If plan is custom, ensure only the requesting user can access it via CPR link
        $isCustom = (isset($plan_info->is_custom) && (int)$plan_info->is_custom === 1) ? true : false;
        if ($isCustom) {
            $cprId = request()->query('cpr', Session::get('custom_plan_request_id'));
            if (!$cprId) {
                Session::flash('error_flash_message', trans('words.access_denied'));
                return redirect('pricing');
            }

            $cpr = \App\CustomPlanRequest::find($cprId);
            if (!$cpr || $cpr->user_id !== Auth::id() || $cpr->generated_plan_id != $plan_id) {
                Session::flash('error_flash_message', trans('words.access_denied'));
                return redirect('pricing');
            }

            // For online payments, must be awaiting_payment
            if ($cpr->payment_preference === 'online' && $cpr->status !== 'awaiting_payment') {
                Session::flash('error_flash_message', 'This payment link is not valid anymore.');
                return redirect('custom-plan/requests');
            }
        }

        //For free plan
        if ($plan_info->plan_price <= 0) {
            $plan_days = $plan_info->plan_days;
            $plan_amount = $plan_info->plan_price;

            $currency_code = getcong('currency_code') ? getcong('currency_code') : 'USD';

            $user_id = Auth::user()->id;
            $user = User::findOrFail($user_id);

            $user->plan_id = $plan_id;
            $user->start_date = strtotime(date('m/d/Y'));
            // No-expiry: do not set exp_date

            $user->plan_amount = $plan_amount;
            $user->save();


            $payment_trans = new Transactions;

            $payment_trans->user_id = Auth::user()->id;
            $payment_trans->email = Auth::user()->email;
            $payment_trans->plan_id = $plan_id;
            $payment_trans->gateway = 'NA';
            $payment_trans->payment_amount = $plan_amount;
            $payment_trans->payment_id = '-';
            $payment_trans->date = strtotime(date('m/d/Y H:i:s'));
            $payment_trans->save();

            Session::flash('plan_id', Session::get('plan_id'));

            Session::flash('success', trans('words.payment_success'));
            return redirect('dashboard');
        }

        Session::put('plan_id', $plan_id);
        // Track custom plan request (if present)
        $cprId = request()->query('cpr');
        if ($cprId) {
            Session::put('custom_plan_request_id', (int)$cprId);
        }
        Session::flash('razorpay_order_id', Session::get('razorpay_order_id'));


        return view('pages.payment.payment_method', compact('plan_info'));
    }


    public function account_delete()
    {

        if (!Auth::check()) {
            Session::flash('error_flash_message', trans('words.access_denied'));
            return redirect('login');
        }

        $user_id = Auth::user()->id;
        $user = User::findOrFail($user_id);

        $user_name = $user->name;
        $user_email = $user->email;

        //Change Status
        $user_obj = User::findOrFail($user_id);
        $user_obj->status = 0;
        $user_obj->save();

        //Delete session file
        $user_session_name = Session::getId();
        Session::getHandler()->destroy($user_session_name);

        Auth::logout();

        //Account Delete Email

        if (getenv("MAIL_USERNAME")) {

            $data_email = array(
                'name' => $user_name,
                'email' => $user_email
            );

            \Mail::send('emails.account_delete', $data_email, function ($message) use ($user_name, $user_email) {
                $message->to($user_email, $user_name)
                    ->from(getcong('site_email'), getcong('site_name'))
                    ->subject(trans('words.user_dlt_email_subject'));
            });
        }

        $response['status'] = 1;

        echo json_encode($response);
        exit;
    }
}
