<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;

class Banner extends Model
{
    protected $fillable = [
        'user_id',
        'name',
        'slug',
        'image_path',
        'redirect_url',
        'description',
        'cost',
        'status',
        'starts_at',
        'ends_at',
        'lat',
        'lng'
    ];

    protected $casts = [
        'starts_at' => 'datetime',
        'ends_at'   => 'datetime',
        'cost'      => 'decimal:2',
        'status'    => 'integer',
        'description' => 'json',
    ];

    // Only show currently active banners
    public function scopeActive(Builder $q): Builder
    {
        $now = now();
        return $q->where('status', 1)
            ->where(function ($qq) use ($now) {
                $qq->whereNull('starts_at')->orWhere('starts_at', '<=', $now);
            })
            ->where(function ($qq) use ($now) {
                $qq->whereNull('ends_at')->orWhere('ends_at', '>=', $now);
            });
    }

    /**
     * Order by cost DESC, then distance ASC (Haversine in KM).
     * Optional $radiusKm if you want to bound the search.
     */
    public function scopeByLocationAndCost(
        Builder $q,
        float $userLat,
        float $userLng,
        ?float $radiusKm = null
    ): Builder {
        $haversine = "
            6371 * acos(
                cos(radians(?)) * cos(radians(lat)) * cos(radians(lng) - radians(?)) +
                sin(radians(?)) * sin(radians(lat))
            )
        ";

        $q->select('*')->selectRaw("$haversine AS distance_km", [$userLat, $userLng, $userLat]);

        if ($radiusKm !== null) {
            $q->having('distance_km', '<=', $radiusKm);
        }

        return $q->orderByDesc('cost')
            ->orderBy('distance_km'); // tie-breaker
    }
    
    //sj_code
    // app/Models/Banner.php
    public function user()
    {
        return $this->belongsTo(User::class, 'user_id');
    }
}
