<?php

namespace App\Services;

use App\Contracts\OTPServiceInterface;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class Fast2SMSService implements OTPServiceInterface
{
    protected $apiKey;

    public function __construct()
    {
        $this->apiKey = env('FAST2SMS_API_KEY');
        if (!$this->apiKey) {
            throw new \Exception('Fast2SMS API key is not configured');
        }

        // Log API key for debugging (first 4 characters only)
        Log::info('Fast2SMS Service initialized', [
            'api_key_prefix' => substr($this->apiKey, 0, 4) . '...',
            'api_key_length' => strlen($this->apiKey)
        ]);
    }

    /**
     * Send OTP to phone number
     */
    public function sendOtp(string $phoneNumber, ?string $message = null): array
    {
        try {
            // Clean phone number (remove any non-digit characters)
            $cleanPhone = preg_replace('/\D/', '', $phoneNumber);
            // Remove leading plus and zeros
            $cleanPhone = ltrim($cleanPhone, '+');
            $cleanPhone = preg_replace('/^0+/', '', $cleanPhone);

            // Ensure phone number starts with country code (91 for India)
            if (strlen($cleanPhone) == 10) {
                $cleanPhone = '91' . $cleanPhone;
            }

            // Default OTP message if not provided
            if (!$message) {
                $otp = rand(1000, 9999);
                $message = "Your OTP for verification is: {$otp}";
            }

            // Fast2SMS API v2 format - try different authorization formats
            $apiKey = trim($this->apiKey);

            // Try with different authorization header formats
            $headers = [
                'authorization' => $apiKey,
                'Content-Type' => 'application/json'
            ];

            // Some APIs require the key to be prefixed
            if (!preg_match('/^[A-Za-z0-9]/', $apiKey)) {
                $headers['authorization'] = $apiKey;
            }

            // Try alternative authorization formats
            $authFormats = [
                $headers,
                ['Authorization' => $apiKey, 'Content-Type' => 'application/json'], // Capital A
                ['authorization' => 'Bearer ' . $apiKey, 'Content-Type' => 'application/json'], // Bearer token
                ['Content-Type' => 'application/json'], // No auth header
            ];

            // Try alternative endpoint format
            $endpoints = [
                'https://www.fast2sms.com/dev/bulkV2',
                'https://www.fast2sms.com/dev/bulk2',
                'https://www.fast2sms.com/dev/bulk',
                'https://api.fast2sms.com/bulkV2'
            ];

            $response = null;
            foreach ($endpoints as $endpoint) {
                foreach ($authFormats as $authFormat) {
                    try {
                        $response = Http::timeout(30)
                            ->withHeaders($authFormat)
                            ->post($endpoint, [
                                'message' => $message,
                                'language' => 'english',
                                'route' => 'q',
                                'numbers' => $cleanPhone,
                            ]);

                        // Log the attempt
                        Log::debug('Fast2SMS attempt', [
                            'endpoint' => $endpoint,
                            'auth_format' => json_encode(array_keys($authFormat)),
                            'status' => $response->status()
                        ]);

                        if ($response->successful() && strpos($response->body(), '<html') === false) {
                            break 2; // Stop trying other endpoints and auth formats
                        }
                    } catch (\Exception $e) {
                        Log::warning('Fast2SMS attempt failed', [
                            'endpoint' => $endpoint,
                            'auth_format' => json_encode(array_keys($authFormat)),
                            'error' => $e->getMessage()
                        ]);
                        continue;
                    }
                }
            }

            // Log the request for debugging
            $usedEndpoint = $endpoints[0]; // Default to first endpoint
            if ($response) {
                $usedEndpoint = $response->effectiveUri()->getHost() . $response->effectiveUri()->getPath();
            }

            Log::info('Fast2SMS API Request', [
                'url' => $usedEndpoint,
                'headers' => [
                    'authorization' => 'Bearer ' . substr($apiKey, 0, 4) . '...',
                    'Content-Type' => 'application/json'
                ],
                'payload' => [
                    'message' => $message,
                    'language' => 'english',
                    'route' => 'q',
                    'numbers' => $cleanPhone,
                ],
                'response_status' => $response ? $response->status() : 'null',
                'response_body' => $response ? $response->body() : 'null'
            ]);

            if ($response->successful() && strpos($response->body(), '<html') === false) {
                $responseBody = $response->body();

                // Try to parse as JSON
                $result = json_decode($responseBody, true);

                if (json_last_error() === JSON_ERROR_NONE) {
                    if (isset($result['return']) && $result['return'] === true) {
                        Log::info('Fast2SMS OTP sent successfully', [
                            'phone' => $cleanPhone,
                            'message_id' => $result['message_id'] ?? null
                        ]);
                        return [
                            'success' => true,
                            'message' => 'OTP sent successfully',
                            'message_id' => $result['message_id'] ?? null
                        ];
                    } else {
                        Log::error('Fast2SMS API error', ['response' => $result]);
                        return [
                            'success' => false,
                            'message' => 'Failed to send OTP: ' . ($result['message'] ?? 'Unknown error')
                        ];
                    }
                } else {
                    // Response is not JSON, might be success message
                    Log::info('Fast2SMS response (non-JSON)', [
                        'phone' => $cleanPhone,
                        'response' => $responseBody
                    ]);
                    return [
                        'success' => true,
                        'message' => 'OTP sent successfully',
                        'message_id' => null
                    ];
                }
            } else {
                Log::error('Fast2SMS HTTP error', [
                    'status' => $response->status(),
                    'body' => $response->body()
                ]);
                return [
                    'success' => false,
                    'message' => 'HTTP error: ' . $response->status() . ' - ' . $response->body()
                ];
            }
        } catch (\Exception $e) {
            Log::error('Fast2SMS service exception', [
                'message' => $e->getMessage(),
                'phone' => $phoneNumber
            ]);
            return [
                'success' => false,
                'message' => 'Service error: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Send custom message
     */
    public function sendMessage(string $phoneNumber, string $message): array
    {
        return $this->sendOtp($phoneNumber, $message);
    }

    /**
     * Generate and send OTP
     */
    public function sendVerificationOtp(string $phoneNumber): array
    {
        $otp = rand(1000, 9999);
        $message = "Your verification code is: {$otp}. This code will expire in 10 minutes.";

        $result = $this->sendOtp($phoneNumber, $message);

        if ($result['success']) {
            return [
                'success' => true,
                'otp' => $otp,
                'message_id' => $result['message_id']
            ];
        }

        return $result;
    }

    /**
     * Test the Fast2SMS API connection
     */
    public function testConnection(): array
    {
        try {
            $apiKey = trim($this->apiKey);

            // Try different endpoints for balance check
            $endpoints = [
                'https://www.fast2sms.com/dev/wallet/balance',
                'https://www.fast2sms.com/dev/balance',
                'https://www.fast2sms.com/dev/account',
                'https://api.fast2sms.com/wallet/balance'
            ];

            $response = null;
            foreach ($endpoints as $endpoint) {
                try {
                    $response = Http::timeout(30)
                        ->withHeaders([
                            'authorization' => $apiKey,
                            'Content-Type' => 'application/json'
                        ])
                        ->get($endpoint);

                    Log::debug('Fast2SMS balance check attempt', [
                        'endpoint' => $endpoint,
                        'status' => $response->status()
                    ]);

                    // If we get a proper JSON response, use it
                    if ($response->status() !== 404 && strpos($response->body(), '<html') === false) {
                        break;
                    }
                } catch (\Exception $e) {
                    continue;
                }
            }

            Log::info('Fast2SMS Connection Test', [
                'status' => $response->status(),
                'body' => $response->body(),
                'api_key_prefix' => substr($apiKey, 0, 4) . '...'
            ]);

            if ($response->successful() && strpos($response->body(), '<html') === false) {
                $data = $response->json();
                Log::info('Fast2SMS Connection Test Success', [
                    'balance' => $data['balance'] ?? 'unknown',
                    'status' => 'success'
                ]);
                return [
                    'success' => true,
                    'message' => 'Fast2SMS connection successful. Balance: ' . ($data['balance'] ?? 'unknown')
                ];
            } else {
                Log::error('Fast2SMS Connection Test Failed', [
                    'status' => $response->status(),
                    'body' => $response->body()
                ]);
                return [
                    'success' => false,
                    'message' => 'Fast2SMS connection failed: HTTP ' . $response->status() . ' - ' . $response->body()
                ];
            }
        } catch (\Exception $e) {
            return [
                'success' => false,
                'message' => 'Connection test error: ' . $e->getMessage()
            ];
        }
    }
}
