<?php

namespace App\Services;

use App\Contracts\OTPServiceInterface;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class TwilioService implements OTPServiceInterface
{
    protected $sid;
    protected $token;
    protected $fromNumber;
    protected $baseUrl;

    public function __construct()
    {
        $this->sid = env('TWILIO_SID');
        $this->token = env('TWILIO_TOKEN');
        $this->fromNumber = env('TWILIO_FROM_NUMBER');
        $this->baseUrl = 'https://api.twilio.com/2010-04-01';

        if (!$this->sid || !$this->token || !$this->fromNumber) {
            throw new \Exception('Twilio credentials are not configured. Please set TWILIO_SID, TWILIO_TOKEN, and TWILIO_FROM_NUMBER in your .env file');
        }

        Log::info('Twilio Service initialized', [
            'from_number' => $this->fromNumber,
            'sid_prefix' => substr($this->sid, 0, 4) . '...',
            'token_prefix' => substr($this->token, 0, 4) . '...'
        ]);
    }

    /**
     * Send OTP to phone number
     */
    public function sendOtp(string $phoneNumber, string $message = null): array
    {
        try {
            // Clean and format phone number
            $cleanPhone = $this->formatPhoneNumber($phoneNumber);

            // Default OTP message if not provided
            if (!$message) {
                $otp = rand(1000, 9999);
                $message = "Your OTP for verification is: {$otp}";
            }

            // Send message via Twilio HTTP API
            $response = Http::timeout(30)
                ->withBasicAuth($this->sid, $this->token)
                ->asForm()
                ->post("{$this->baseUrl}/Accounts/{$this->sid}/Messages.json", [
                    'From' => $this->fromNumber,
                    'To' => $cleanPhone,
                    'Body' => $message
                ]);

            if ($response->successful()) {
                $data = $response->json();

                Log::info('Twilio OTP sent successfully', [
                    'phone' => $cleanPhone,
                    'message_sid' => $data['sid'],
                    'status' => $data['status']
                ]);

                return [
                    'success' => true,
                    'message' => 'OTP sent successfully',
                    'message_id' => $data['sid']
                ];
            } else {
                $errorData = $response->json();
                Log::error('Twilio HTTP API error', [
                    'status' => $response->status(),
                    'response' => $errorData,
                    'phone' => $cleanPhone
                ]);

                return [
                    'success' => false,
                    'message' => 'Failed to send OTP: ' . ($errorData['message'] ?? 'Unknown error')
                ];
            }
        } catch (\Exception $e) {
            Log::error('Twilio service exception', [
                'message' => $e->getMessage(),
                'phone' => $phoneNumber
            ]);

            return [
                'success' => false,
                'message' => 'Service error: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Send custom message
     */
    public function sendMessage(string $phoneNumber, string $message): array
    {
        return $this->sendOtp($phoneNumber, $message);
    }

    /**
     * Send verification OTP with auto-generated code
     */
    public function sendVerificationOtp(string $phoneNumber): array
    {
        $otp = rand(100000, 999999);
        $message = "Your verification code is: {$otp}. This code will expire in 10 minutes.";

        $result = $this->sendOtp($phoneNumber, $message);

        if ($result['success']) {
            return [
                'success' => true,
                'otp' => $otp,
                'message_id' => $result['message_id']
            ];
        }

        return $result;
    }

    /**
     * Test the Twilio connection
     */
    public function testConnection(): array
    {
        try {
            // Test connection by getting account info
            $response = Http::timeout(30)
                ->withBasicAuth($this->sid, $this->token)
                ->get("{$this->baseUrl}/Accounts/{$this->sid}.json");

            if ($response->successful()) {
                $data = $response->json();

                Log::info('Twilio Connection Test', [
                    'status' => 'success',
                    'account_sid' => $data['sid'],
                    'account_status' => $data['status']
                ]);

                return [
                    'success' => true,
                    'message' => 'Twilio connection successful. Account status: ' . $data['status']
                ];
            } else {
                $errorData = $response->json();
                Log::error('Twilio connection test failed', [
                    'status' => $response->status(),
                    'response' => $errorData
                ]);

                return [
                    'success' => false,
                    'message' => 'Twilio connection failed: ' . ($errorData['message'] ?? 'HTTP ' . $response->status())
                ];
            }
        } catch (\Exception $e) {
            Log::error('Twilio connection test exception', [
                'message' => $e->getMessage()
            ]);

            return [
                'success' => false,
                'message' => 'Connection test error: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Format phone number for Twilio (E.164 format)
     */
    protected function formatPhoneNumber(string $phoneNumber): string
    {
        // Remove all non-digit characters
        $cleanPhone = preg_replace('/\D/', '', $phoneNumber);

        // Remove leading zeros
        $cleanPhone = ltrim($cleanPhone, '0');

        // If phone number is 10 digits (Indian number), add country code
        if (strlen($cleanPhone) === 10) {
            $cleanPhone = '91' . $cleanPhone;
        }

        // Ensure it starts with +
        if (!str_starts_with($cleanPhone, '+')) {
            $cleanPhone = '+' . $cleanPhone;
        }

        return $cleanPhone;
    }
}
