/**
 * MapView Core Module
 * Main class for handling the property map view functionality
 */
class MapView {
    constructor(options) {
        this.options = {
            mapContainer: 'advanced-map',
            apiEndpoint: '/map/data',
            currencySymbol: '₹',
            propertyTypes: [],
            defaultCenter: [20.5937, 78.9629],
            defaultZoom: 6,
            animationDuration: 300,
            debounceDelay: 400,
            maxZoomForClustering: 15,
            ...options
        };
        
        this.map = null;
        this.markers = null;
        this.properties = [];
        this.filteredProperties = [];
        this.currentFilters = {};
        this.drawControl = null;
        this.drawnItems = null;
        this.heatmapLayer = null;
        this.propertyCache = new Map();
        this.searchCache = new Map();
        
        // Performance optimization
        this.requestQueue = [];
        this.isRequestInProgress = false;
        this.lastRequestTime = 0;
        this.minRequestInterval = 1000;
        
        // State management
        this.state = {
            isLoading: false,
            isInitialized: false,
            isInitializing: false,
            initRetryCount: 0,
            maxRetries: 3,
            isDarkMode: localStorage.getItem('map_dark_mode') === 'true',
            isFullscreen: false,
            selectedProperty: null,
            favorites: (function(){
                try { return (JSON.parse(localStorage.getItem('property_favorites') || '[]') || []).map(v => { const n = parseInt(v, 10); return isNaN(n) ? v : n; }); } catch(_) { return []; }
            })(),
            compareList: (function(){
                try { return (JSON.parse(localStorage.getItem('property_compare') || '[]') || []).map(v => { const n = parseInt(v, 10); return isNaN(n) ? v : n; }); } catch(_) { return []; }
            })(),
            autoFit: true,
            viewMode: 'markers', // markers, heatmap, satellite
            filterCount: 0,
            currentSort: 'newest',
            isAnimating: false
        };
        
        // Animation and interaction state
        this.animations = {
            propertyCards: new Map(),
            markers: new Map(),
            filters: null
        };
        
        // Initialize performance monitoring
        this.performance = {
            loadTimes: [],
            renderTimes: [],
            lastUpdate: Date.now()
        };

        // Create debounced load properties method
        this.debouncedLoadProperties = this.debounce(this.loadProperties.bind(this), this.options.debounceDelay);

        // Enhanced debugging
        this.debug = {
            enabled: true,
            logs: [],
            startTime: Date.now(),
            requestCount: 0,
            errorCount: 0
        };

        this.log('MapView initialized with options:', options);
    }

    /**
     * Update filters from external form submission (sidebar/topbar)
     * Accepts a plain object from URLSearchParams/Object.fromEntries
     * and normalizes it into this.currentFilters, then applies filters.
     */
    /**
     * Enhanced logging method for debugging
     */
    log(message, data = null, type = 'info') {
        if (!this.debug.enabled) return;

        const timestamp = new Date().toISOString();
        const logEntry = {
            timestamp,
            message,
            data: data ? JSON.stringify(data, null, 2) : null,
            type,
            elapsed: Date.now() - this.debug.startTime
        };

        this.debug.logs.push(logEntry);

        // Keep only last 100 logs
        if (this.debug.logs.length > 100) {
            this.debug.logs.shift();
        }

        const prefix = `[${timestamp}] [${type.toUpperCase()}]`;
        console.log(prefix, message, data || '');
    }

    /**
     * Get debug information
     */
    getDebugInfo() {
        return {
            logs: this.debug.logs,
            state: this.state,
            options: this.options,
            performance: this.performance,
            elapsed: Date.now() - this.debug.startTime,
            requestCount: this.debug.requestCount,
            errorCount: this.debug.errorCount
        };
    }

    /**
     * Update filters from external form submission (sidebar/topbar)
     * Accepts a plain object from URLSearchParams/Object.fromEntries
     * and normalizes it into this.currentFilters, then applies filters.
     */
    updateFilters(formObj = {}) {
        try {
            this.log('updateFilters called', formObj);

            const f = { ...(this.currentFilters || {}) };

            // Text search (accept both 'search' and legacy 'search_text')
            if (typeof formObj.search === 'string' && formObj.search.trim() !== '') {
                f.search = formObj.search.trim();
            } else if (typeof formObj.search_text === 'string' && formObj.search_text.trim() !== '') {
                f.search = formObj.search_text.trim();
            } else {
                delete f.search;
            }

            // Purpose (Sale/Rent)
            if (typeof formObj.purpose === 'string' && formObj.purpose.trim() !== '') {
                f.purpose = formObj.purpose.trim();
            } else {
                delete f.purpose;
            }

            // Property type -> keep as type_id to match backend
            if (typeof formObj.type_id === 'string' && formObj.type_id !== '') {
                f.type_id = formObj.type_id;
            } else {
                delete f.type_id;
            }

            // Location id (kept for server requests; local filter optional)
            if (typeof formObj.location_id === 'string' && formObj.location_id !== '') {
                f.location_id = formObj.location_id;
            } else {
                delete f.location_id;
            }

            // Verified -> boolean (accept common truthy string variants)
            if (typeof formObj.verified === 'string' && formObj.verified !== '') {
                const v = formObj.verified.trim().toLowerCase();
                const truthy = ['1', 'true', 'yes', 'on'];
                const falsy = ['0', 'false', 'no', 'off'];
                if (truthy.includes(v)) {
                    f.verified = 1; // backend expects boolean-like or 1
                } else if (falsy.includes(v)) {
                    delete f.verified; // treat as not applied
                } else {
                    // Default: attempt boolean parse
                    f.verified = v === 'yes' || v === 'true' ? 1 : undefined;
                    if (f.verified === undefined) delete f.verified;
                }
            } else {
                delete f.verified;
            }

            // Furnishing -> map to furnished boolean if explicitly Furnished
            if (typeof formObj.furnishing === 'string' && formObj.furnishing !== '') {
                const v = formObj.furnishing.toLowerCase();
                // Only treat fully furnished as furnished=true; others unset
                if (v === 'furnished') {
                    f.furnished = true;
                } else {
                    delete f.furnished;
                }
                f.furnishing = formObj.furnishing; // keep original for backend
            } else {
                delete f.furnished;
                delete f.furnishing;
            }

            // Bedrooms & Bathrooms (minimums)
            if (typeof formObj.bedrooms === 'string' && formObj.bedrooms !== '') {
                const b = parseInt(formObj.bedrooms, 10);
                if (!isNaN(b)) f.bedrooms = b; else delete f.bedrooms;
            } else {
                delete f.bedrooms;
            }
            if (typeof formObj.bathrooms === 'string' && formObj.bathrooms !== '') {
                const b = parseInt(formObj.bathrooms, 10);
                if (!isNaN(b)) f.bathrooms = b; else delete f.bathrooms;
            } else {
                delete f.bathrooms;
            }
            // Price range (formatted like "₹500000 - ₹1000000")
            if (typeof formObj.price_range === 'string' && formObj.price_range.trim() !== '') {
                const parts = formObj.price_range.split('-').map(s => s.trim());
                const toNum = (s) => {
                    if (!s) return NaN;
                    const cleaned = s.replace(/[^0-9.]/g, '');
                    return parseFloat(cleaned);
                };
                const min = toNum(parts[0]);
                const max = toNum(parts[1]);
                if (!isNaN(min)) f.price_min = min; else delete f.price_min;
                if (!isNaN(max)) f.price_max = max; else delete f.price_max;
            } else {
                delete f.price_min;
                delete f.price_max;
            }

            // Persist and apply
            this.currentFilters = f;
            this.log('Filters updated', f);
            this.applyFilters();
        } catch (e) {
            this.log('updateFilters failed', e, 'error');
            this.debug.errorCount++;
            console.warn('updateFilters failed:', e);
        }
    }

    /**
     * Format price with Indian numbering system
     */
    formatPrice(price) {
        if (!price || price === 0) return 'Price on request';
        if (typeof price !== 'number') return String(price);

        // Use Indian numbering system
        if (price >= 10000000) {
            return '₹' + (price / 10000000).toFixed(1) + ' Cr';
        } else if (price >= 100000) {
            return '₹' + (price / 100000).toFixed(1) + ' L';
        } else if (price >= 1000) {
            return '₹' + (price / 1000).toFixed(1) + 'K';
        }
        return '₹' + price.toString();
    }

    /**
     * Normalize backend property object so downstream code can rely on latitude/longitude
     */
    normalizeProperty(p) {
        if (!p || typeof p !== 'object') return p;
        // Try several possible keys for coordinates
        const candidates = [
            ['latitude', 'longitude'],
            ['lat', 'lng'],
            ['lat', 'lon'],
            ['map_lat', 'map_lng'],
            ['map_latitude', 'map_longitude'],
            ['geo_lat', 'geo_lng'],
            ['location_latitude', 'location_longitude']
        ];
        let lat = p.latitude, lng = p.longitude;
        if (!(this.isFiniteNumber(lat) && this.isFiniteNumber(lng))) {
            for (const [la, lo] of candidates) {
                const vlat = p[la];
                const vlng = p[lo];
                const flat = vlat != null ? parseFloat(vlat) : NaN;
                const flng = vlng != null ? parseFloat(vlng) : NaN;
                if (this.isFiniteNumber(flat) && this.isFiniteNumber(flng)) {
                    lat = flat; lng = flng; break;
                }
            }
        }
        // Assign normalized values if valid
        if (this.isFiniteNumber(lat) && this.isFiniteNumber(lng)) {
            p.latitude = lat;
            p.longitude = lng;
        }
        // Coerce common numeric fields to numbers to make comparisons stable
        if (p.price != null) {
            const n = parseFloat(p.price);
            if (this.isFiniteNumber(n)) p.price = n;
        }
        if (p.bedrooms != null) {
            const n = parseInt(p.bedrooms);
            if (this.isFiniteNumber(n)) p.bedrooms = n;
        }
        if (p.bathrooms != null) {
            const n = parseInt(p.bathrooms);
            if (this.isFiniteNumber(n)) p.bathrooms = n;
        }
        return p;
    }

    /**
     * Check if a value is a finite number
     */
    isFiniteNumber(v) {
        return typeof v === 'number' && isFinite(v);
    }

    /**
     * Toggle filters panel visibility
     */
    toggleFiltersPanel() {
        const panel = document.getElementById('map-right-panel');
        if (panel) {
            panel.classList.toggle('collapsed');
            this.log('Filters panel toggled', { collapsed: panel.classList.contains('collapsed') });
        }
    }

    /**
     * Toggle dark mode
     */
    toggleDarkMode() {
        this.state.isDarkMode = !this.state.isDarkMode;
        localStorage.setItem('map_dark_mode', this.state.isDarkMode);

        if (this.state.isDarkMode) {
            document.body.classList.add('dark-mode');
            if (this.map) {
                this.map.setOptions({ styles: this.darkMapStyle });
            }
        } else {
            document.body.classList.remove('dark-mode');
            if (this.map) {
                this.map.setOptions({ styles: null });
            }
        }

        this.showNotification(
            this.state.isDarkMode ? 'Dark mode enabled' : 'Light mode enabled',
            'success'
        );
    }

    /**
     * Clear all filters
     */
    clearAllFilters() {
        this.currentFilters = {};
        this.applyFilters();
        this.showNotification('All filters cleared', 'info');
    }

    /**
     * Sort filtered properties
     */
    sortFilteredProperties() {
        if (!this.filteredProperties || this.filteredProperties.length === 0) return;

        switch (this.state.currentSort) {
            case 'price-low':
                this.filteredProperties.sort((a, b) => a.price - b.price);
                break;
            case 'price-high':
                this.filteredProperties.sort((a, b) => b.price - a.price);
                break;
            case 'newest':
                this.filteredProperties.sort((a, b) => b.id - a.id);
                break;
            case 'oldest':
                this.filteredProperties.sort((a, b) => a.id - b.id);
                break;
            case 'area':
                this.filteredProperties.sort((a, b) => b.area - a.area);
                break;
            default:
                this.filteredProperties.sort((a, b) => b.id - a.id);
        }
    }

    /**
     * Locate user on map
     */
    locateUser() {
        if (!navigator.geolocation) {
            this.showNotification('Geolocation is not supported by this browser', 'error');
            return;
        }

        navigator.geolocation.getCurrentPosition(
            (position) => {
                const { latitude, longitude } = position.coords;
                const userLocation = { lat: latitude, lng: longitude };

                // Update current user location
                window.currentUserLocation = userLocation;

                // Center map on user location
                if (this.map) {
                    this.map.setCenter(userLocation);
                    this.map.setZoom(14);
                }

                this.showNotification('Location found and map centered', 'success');
            },
            (error) => {
                console.error('Geolocation error:', error);
                this.showNotification('Unable to get your location', 'error');
            },
            {
                enableHighAccuracy: true,
                timeout: 10000,
                maximumAge: 300000
            }
        );
    }

    /**
     * Handle window resize
     */
    handleResize() {
        // Trigger map resize event to recalculate tiles
        if (this.map) {
            google.maps.event.trigger(this.map, 'resize');
        }
    }
    
    /**
     * Initialize the map view - simplified for use with MapInitializer
     */
    async init() {
        // Prevent multiple simultaneous initializations
        if (this.state.isInitialized || this.state.isInitializing) {
            console.log('Map already initialized or initializing');
            return;
        }

        this.state.isInitializing = true;

        try {
            // Initialize core components (Google Maps API already verified by MapInitializer)
            await this.initializeMap();
            this.initializeControls();
            this.initializeEventListeners();
            this.initializePerformanceMonitoring();

            // Apply saved dark mode preference
            if (this.state.isDarkMode) {
                try {
                    document.body.classList.add('dark-mode');
                    this.map?.setOptions?.({ styles: this.darkMapStyle });
                } catch (_) {}
            }

            // Initialize modules with error handling
            await this.initializeModules();

            // Load initial properties
            await this.loadProperties();

            // Initialize advanced features
            this.initializeAdvancedFeatures();

            this.state.isInitialized = true;
            this.showNotification('Map initialized successfully', 'success');

        } catch (error) {
            console.error('Error initializing map:', error);
            throw error; // Let MapInitializer handle the error
        } finally {
            this.state.isInitializing = false;
        }
    }
    
    /**
     * Simple loading state management
     */
    showLoadingState(message = 'Loading...') {
        // Loading state is now handled by MapInitializer
        console.log('MapView:', message);
    }
    
    hideLoadingState() {
        // Loading state is now handled by MapInitializer
        console.log('MapView: Loading complete');
    }
    
    /**
     * Initialize modules with proper error handling
     */
    async initializeModules() {
        const modules = [
            { name: 'MapUtils', class: MapUtils, property: 'utils' },
            { name: 'MapFilters', class: MapFilters, property: 'filters' },
            { name: 'MapMarkers', class: MapMarkers, property: 'markersManager' },
            { name: 'MapInteractions', class: MapInteractions, property: 'interactions' },
            { name: 'PropertyDetails', class: PropertyDetails, property: 'propertyDetails' }
        ];

        for (const module of modules) {
            try {
                if (typeof module.class !== 'undefined') {
                    this[module.property] = new module.class(this);
                    this.log(`${module.name} module initialized successfully`);
                } else {
                    console.warn(`${module.name} module not available`);
                    this.log(`${module.name} module not available`, null, 'warn');
                }
            } catch (error) {
                console.error(`Error initializing ${module.name}:`, error);
                this.log(`Error initializing ${module.name}`, error, 'error');
                this.debug.errorCount++;
                // Don't throw - allow map to work with missing modules
            }
        }
    }
    
    /**
     * Initialize performance monitoring
     */
    initializePerformanceMonitoring() {
        // Monitor frame rate
        let frameCount = 0;
        let lastTime = performance.now();
        
        const monitorFrameRate = () => {
            frameCount++;
            const currentTime = performance.now();
            
            if (currentTime - lastTime >= 1000) {
                const fps = Math.round((frameCount * 1000) / (currentTime - lastTime));
                this.performance.fps = fps;
                
                if (fps < 30) {
                    console.warn('Low frame rate detected:', fps);
                }
                
                frameCount = 0;
                lastTime = currentTime;
            }
            
            requestAnimationFrame(monitorFrameRate);
        };
        
        requestAnimationFrame(monitorFrameRate);
    }
    
    /**
     * Initialize advanced features
     */
    initializeAdvancedFeatures() {
        // Initialize keyboard shortcuts
        this.initializeKeyboardShortcuts();

        // Initialize gesture recognition for mobile
        this.initializeGestureRecognition();

        // Initialize real-time updates
        this.initializeRealTimeUpdates();

        // Initialize analytics
        this.initializeAnalytics();
    }
    
    /**
     * Simplified map initialization - Google Maps API verification handled by MapInitializer
     */
    
    /**
     * Initialize the Google Map
     */
    async initializeMap() {
        const container = document.getElementById(this.options.mapContainer);
        if (!container) {
            throw new Error(`Map container not found: ${this.options.mapContainer}`);
        }
        
        // Ensure container is visible and has dimensions
        if (container.offsetWidth === 0 || container.offsetHeight === 0) {
            throw new Error('Map container has no dimensions');
        }
        
        this.map = new google.maps.Map(container, {
            center: { lat: this.options.defaultCenter[0], lng: this.options.defaultCenter[1] },
            zoom: this.options.defaultZoom,
            mapId: 'DEMO_MAP_ID', // Required for Advanced Markers
            mapTypeControl: true,
            streetViewControl: false,
            fullscreenControl: false,
            gestureHandling: 'cooperative',
            mapTypeControlOptions: {
                style: google.maps.MapTypeControlStyle.HORIZONTAL_BAR,
                position: google.maps.ControlPosition.TOP_CENTER
            },
            zoomControlOptions: {
                position: google.maps.ControlPosition.RIGHT_CENTER
            }
        });
        
        // Wait for map to be fully loaded
        await new Promise((resolve, reject) => {
            const timeout = setTimeout(() => {
                reject(new Error('Map load timeout'));
            }, 10000);
            
            google.maps.event.addListenerOnce(this.map, 'idle', () => {
                clearTimeout(timeout);
                resolve();
            });
        });
        
        // Holder for Google markers and clusterer (managed by MapMarkers module)
        this.markers = [];
        this.markerClusterer = null;
        
        // Predefine dark mode style
        this.darkMapStyle = [
            { elementType: 'geometry', stylers: [{ color: '#242f3e' }] },
            { elementType: 'labels.text.stroke', stylers: [{ color: '#242f3e' }] },
            { elementType: 'labels.text.fill', stylers: [{ color: '#746855' }] },
            { featureType: 'administrative.locality', elementType: 'labels.text.fill', stylers: [{ color: '#d59563' }] },
            { featureType: 'poi', elementType: 'labels.text.fill', stylers: [{ color: '#d59563' }] },
            { featureType: 'poi.park', elementType: 'geometry', stylers: [{ color: '#263c3f' }] },
            { featureType: 'poi.park', elementType: 'labels.text.fill', stylers: [{ color: '#6b9a76' }] },
            { featureType: 'road', elementType: 'geometry', stylers: [{ color: '#38414e' }] },
            { featureType: 'road', elementType: 'geometry.stroke', stylers: [{ color: '#212a37' }] },
            { featureType: 'road', elementType: 'labels.text.fill', stylers: [{ color: '#9ca5b3' }] },
            { featureType: 'road.highway', elementType: 'geometry', stylers: [{ color: '#746855' }] },
            { featureType: 'road.highway', elementType: 'geometry.stroke', stylers: [{ color: '#1f2835' }] },
            { featureType: 'road.highway', elementType: 'labels.text.fill', stylers: [{ color: '#f3d19c' }] },
            { featureType: 'transit', elementType: 'geometry', stylers: [{ color: '#2f3948' }] },
            { featureType: 'transit.station', elementType: 'labels.text.fill', stylers: [{ color: '#d59563' }] },
            { featureType: 'water', elementType: 'geometry', stylers: [{ color: '#17263c' }] },
            { featureType: 'water', elementType: 'labels.text.fill', stylers: [{ color: '#515c6d' }] },
            { featureType: 'water', elementType: 'labels.text.stroke', stylers: [{ color: '#17263c' }] }
        ];
    }
    
    /**
     * Initialize custom controls
     */
    initializeControls() {
        // Google Maps shows zoom controls by default. We handle other controls via existing HTML buttons.
        // No-op for custom in-map controls for now.
    }
    
    initializeEventListeners() {
        // Toggle filters panel
        const toggleFilters = document.getElementById('toggle-filters');
        if (toggleFilters) {
            toggleFilters.addEventListener('click', () => {
                this.toggleFiltersPanel();
            });
        }

        // Dark mode toggle
        const toggleDarkMode = document.getElementById('toggle-dark-mode');
        if (toggleDarkMode) {
            toggleDarkMode.addEventListener('click', () => {
                this.toggleDarkMode();
            });
        }

        // Fullscreen toggle
        const fullscreenMap = document.getElementById('fullscreen-map');
        if (fullscreenMap) {
            fullscreenMap.addEventListener('click', () => {
                const el = document.getElementById(this.options.mapContainer);
                if (!document.fullscreenElement) {
                    el.requestFullscreen?.();
                } else {
                    document.exitFullscreen?.();
                }
            });
        }

        // Clear filters
        const clearFilters = document.getElementById('clear-filters');
        if (clearFilters) {
            clearFilters.addEventListener('click', () => {
                this.clearAllFilters();
            });
        }

        // Reset Filters inside list empty state
        const resetFiltersBtn = document.getElementById('reset-filters');
        if (resetFiltersBtn) {
            resetFiltersBtn.addEventListener('click', () => this.clearAllFilters());
        }

        const resetFiltersListBtn = document.getElementById('reset-filters-list');
        if (resetFiltersListBtn) {
            resetFiltersListBtn.addEventListener('click', () => this.clearAllFilters());
        }

        // Sorting dropdown in property list header
        const listSort = document.getElementById('list-sort');
        if (listSort) {
            listSort.addEventListener('change', (e) => {
                this.state.currentSort = e.target.value || 'newest';
                this.sortFilteredProperties();
                this.updatePropertyDisplay();
            });
        }

        // My Location from topbar
        const myLocationBtn = document.getElementById('view-my-location');
        if (myLocationBtn) {
            myLocationBtn.addEventListener('click', () => {
                this.locateUser();
            });
        }

        // In-map controls
        const zoomInBtn = document.getElementById('map-zoom-in');
        if (zoomInBtn) {
            zoomInBtn.addEventListener('click', () => {
                try { this.map.setZoom((this.map.getZoom() || 0) + 1); } catch (_) {}
            });
        }

        const zoomOutBtn = document.getElementById('map-zoom-out');
        if (zoomOutBtn) {
            zoomOutBtn.addEventListener('click', () => {
                try { this.map.setZoom((this.map.getZoom() || 0) - 1); } catch (_) {}
            });
        }

        const locateBtn = document.getElementById('map-locate');
        if (locateBtn) {
            locateBtn.addEventListener('click', () => this.locateUser());
        }

        const layersBtn = document.getElementById('map-layers');
        if (layersBtn) {
            layersBtn.addEventListener('click', () => {
                const current = this.map.getMapTypeId && this.map.getMapTypeId();
                const next = (current === 'roadmap') ? 'satellite' : 'roadmap';
                try { this.map.setMapTypeId(next); } catch (_) {}
                this.showNotification(`Map layer: ${next}`, 'info');
            });
        }

        const fullscreenBtn = document.getElementById('map-fullscreen');
        if (fullscreenBtn) {
            fullscreenBtn.addEventListener('click', () => {
                const el = document.getElementById(this.options.mapContainer);
                if (!document.fullscreenElement) {
                    el.requestFullscreen?.();
                } else {
                    document.exitFullscreen?.();
                }
            });
        }
    }

    /**
     * Handle map move events
     */
    onMapMove() {
        // Ensure state is available
        if (!this.state) {
            this.state = {
                isLoading: false,
                isInitialized: false,
                isInitializing: false,
                initRetryCount: 0,
                maxRetries: 3,
                isDarkMode: localStorage.getItem('map_dark_mode') === 'true',
                isFullscreen: false,
                selectedProperty: null,
                favorites: JSON.parse(localStorage.getItem('property_favorites') || '[]'),
                compareList: JSON.parse(localStorage.getItem('property_compare') || '[]'),
                autoFit: true,
                viewMode: 'markers',
                filterCount: 0,
                currentSort: 'newest',
                isAnimating: false
            };
        }

        this.state.autoFit = false;
        clearTimeout(this.moveTimeout);
        this.moveTimeout = setTimeout(() => {
            const zoom = this.map?.getZoom?.();
            if (typeof zoom === 'number' && zoom >= 8) {
                this.loadProperties();
            }
        }, 700);
    }

    /**
     * Select a property
     */
    selectProperty(property) {
        this.state.selectedProperty = property;

        // Highlight on map
        if (this.markersManager) {
            this.markersManager.highlightProperty(property.id);
        }

        // Show in drawer
        if (this.propertyDetails) {
            this.propertyDetails.showPropertyDetails(property);
        }

        // Center map on property
        this.map.setZoom(16);
        this.map.panTo({ lat: property.latitude, lng: property.longitude });
    }

    /**
     * Handle window resize
     */
    handleResize() {
        // Trigger map resize event to recalculate tiles
        if (this.map) {
            google.maps.event.trigger(this.map, 'resize');
        }
    }

    /**
     * Toggle filters panel visibility
     */
    toggleFiltersPanel() {
        const panel = document.getElementById('map-right-panel');
        if (panel) {
            panel.classList.toggle('collapsed');
            this.log('Filters panel toggled', { collapsed: panel.classList.contains('collapsed') });
        }
    }

    /**
     * Clear all filters
     */
    clearAllFilters() {
        this.currentFilters = {};
        this.applyFilters();
        this.showNotification('All filters cleared', 'info');
    }

    /**
     * Initialize event listeners
     */
    initializeEventListeners() {
        // Map events
        this.map.addListener('idle', this.onMapMove);

        // UI event listeners are handled within this method
        // Event listeners will be added here when needed

        // Window resize -> gracefully notify map to recompute tiles
        window.addEventListener('resize', this.handleResize, { passive: true });

        // Header controls: 3D toggle (tilt and satellite)
        const toggle3d = document.getElementById('toggle-3d-mode');
        if (toggle3d) {
            toggle3d.addEventListener('click', () => {
                try {
                    const currentType = this.map.getMapTypeId && this.map.getMapTypeId();
                    const nextType = (currentType === 'satellite') ? 'roadmap' : 'satellite';
                    this.map.setMapTypeId(nextType);
                    // Toggle tilt for a pseudo-3D effect when available
                    if (typeof this.map.setTilt === 'function') {
                        this.map.setTilt(nextType === 'satellite' ? 45 : 0);
                    }
                    this.showNotification(nextType === 'satellite' ? '3D mode enabled' : '3D mode disabled', 'info');
                } catch (_) {}
            });
        }

        // Pagination nav (Prev/Next). If backend paging not wired, just scroll list for now
        const prevBtn = document.getElementById('prev-page');
        const nextBtn = document.getElementById('next-page');
        const scrollList = (dir) => {
            const list = document.getElementById('property-list-container');
            if (list) {
                list.scrollBy({ top: dir * 300, behavior: 'smooth' });
            } else {
                this.showNotification('List not ready', 'warning');
            }
        };
        if (prevBtn) prevBtn.addEventListener('click', () => scrollList(-1));
        if (nextBtn) nextBtn.addEventListener('click', () => scrollList(1));

        // Favorites panel bindings
        this.bindFavoritesPanelEvents();
    }
    async loadProperties() {
        this.setLoading(true);
        this.debug.requestCount++;

        const startTime = Date.now();
        this.log('loadProperties started', this.currentFilters);

        try {
            const params = new URLSearchParams();
            // Append filters
            Object.entries(this.currentFilters || {}).forEach(([k, v]) => {
                if (v === undefined || v === null || v === '') return;
                if (typeof v === 'object') {
                    // Special handling for userLocation expected by backend as lat/lng
                    if (k === 'userLocation' && typeof v.lat === 'number' && typeof v.lng === 'number') {
                        params.append('lat', v.lat);
                        params.append('lng', v.lng);
                        return;
                    }
                    // Other objects are ignored or handled separately below (e.g., polygon)
                    // Fallback to JSON if needed for custom backends
                    try { params.append(k, JSON.stringify(v)); } catch (_) {}
                } else {
                    params.append(k, v);
                }
            });

            this.log('Sending request to backend', { url: `${this.options.apiEndpoint}?${params}`, params: Object.fromEntries(params) });

            const response = await fetch(`${this.options.apiEndpoint}?${params}`, {
                headers: { 'Accept': 'application/json' }
            });

            if (!response.ok) {
                throw new Error(`HTTP ${response.status}: ${response.statusText}`);
            }

            let data = null;
            try {
                data = await response.json();
            } catch (e) {
                this.log('Non-JSON response received', e, 'warn');
                throw new Error('Invalid JSON response from server');
            }

            this.log('Response received', data);

            // Normalize properties array from various shapes
            let props = [];
            if (Array.isArray(data)) {
                props = data;
            } else if (data && Array.isArray(data.properties)) {
                props = data.properties;
            } else if (data && data.properties && Array.isArray(data.properties.data)) {
                props = data.properties.data; // Laravel pagination style
            } else if (data && Array.isArray(data.data)) {
                props = data.data;
            } else if (data && Array.isArray(data.items)) {
                props = data.items;
            } else if (data && data.success && Array.isArray(data.results)) {
                props = data.results;
            }

            if (!Array.isArray(props)) {
                this.log('No valid properties array found in response', data, 'error');
                throw new Error('Invalid response format: expected array of properties');
            }

            this.log(`Processing ${props.length} properties`, { firstProperty: props[0] });

            if (props && props.length >= 0) {
                // Normalize coordinates and other fields before storing
                this.properties = (props || []).map(p => this.normalizeProperty(p));
                // Diagnostics: count valid coordinates
                const withCoords = (this.properties || []).filter(p => this.isFiniteNumber(p?.latitude) && this.isFiniteNumber(p?.longitude));

                this.log(`Properties processed: ${this.properties.length} total, ${withCoords.length} with valid coordinates`);

                if (withCoords.length === 0 && this.properties.length > 0) {
                    this.log('WARNING: No properties contain valid coordinates. Markers will not render.', null, 'warn');
                    // Inform user once to check data configuration
                    this.showNotification('No properties have valid coordinates. Please check your data.', 'warning');
                }

                // Apply current filters to freshly loaded data for consistent UX
                this.filteredProperties = (this.properties || []).filter(p => this.matchesFilters(p));
                this.updatePropertyDisplay();
                this.updateStats();
                this.updatePropertyList();

                this.log(`Filtered properties: ${this.filteredProperties.length}`, null);
            } else {
                this.log('No properties returned from server', null, 'warn');
                this.showNotification('No properties found matching your criteria', 'info');
                this.properties = [];
                this.filteredProperties = [];
                this.updatePropertyDisplay();
            }

            const duration = Date.now() - startTime;
            this.log(`loadProperties completed in ${duration}ms`);

        } catch (error) {
            this.log('Error loading properties', error, 'error');
            this.debug.errorCount++;
            this.showNotification(`Error loading properties: ${error.message}`, 'error');

            // Fallback: show empty state
            this.properties = [];
            this.filteredProperties = [];
            this.updatePropertyDisplay();
        } finally {
            this.setLoading(false);
        }
    }
    
    /**
     * Apply current filters to properties
     */
    applyFilters() {
        // Re-enable auto fit when filters drive the updates (not user pan)
        this.state.autoFit = true;
        // If we already have local data, update UI immediately for responsiveness
        if (this.properties.length) {
            this.filteredProperties = this.properties.filter(property => this.matchesFilters(property));
            this.updatePropertyDisplay();
            this.updateStats();
        }
        // Always trigger a debounced refresh from server to stay in sync
        this.debouncedLoadProperties();
    }
    
    /**
     * Check if property matches current filters
     */
    matchesFilters(property) {
        // Search filter
        if (this.currentFilters.search) {
            const searchTerm = this.currentFilters.search.toLowerCase();
            const searchFields = [
                property.title,
                property.location,
                property.description,
                property.address
            ].join(' ').toLowerCase();
            
            if (!searchFields.includes(searchTerm)) {
                return false;
            }
        }
        
        // Property type filter (use type_id consistently)
        if (this.currentFilters.type_id && property.type_id != this.currentFilters.type_id) {
            return false;
        }
        
        // Price range filter
        if (this.currentFilters.price_min && property.price < this.currentFilters.price_min) {
            return false;
        }
        
        if (this.currentFilters.price_max && property.price > this.currentFilters.price_max) {
            return false;
        }
        
        // Bedrooms filter
        if (this.currentFilters.bedrooms && property.bedrooms < this.currentFilters.bedrooms) {
            return false;
        }
        
        // Bathrooms filter
        if (this.currentFilters.bathrooms && property.bathrooms < this.currentFilters.bathrooms) {
            return false;
        }
        
        // Purpose filter
        if (this.currentFilters.purpose && property.purpose !== this.currentFilters.purpose) {
            return false;
        }

        // Status filter
        if (this.currentFilters.status && property.status !== this.currentFilters.status) {
            return false;
        }
        
        // County/City/Area filters (if provided by backend data)
        if (this.currentFilters.county && property.county && property.county !== this.currentFilters.county) {
            return false;
        }
        if (this.currentFilters.city && property.city && property.city !== this.currentFilters.city) {
            return false;
        }
        if (this.currentFilters.area && property.area_name && property.area_name !== this.currentFilters.area) {
            return false;
        }
        
        // Verified filter (accept either is_verified or verified truthy)
        if (this.currentFilters.verified && !(property.is_verified || property.verified)) {
            return false;
        }
        
        // Furnished filter (accept either is_furnished or furnishing === 'Furnished')
        if (this.currentFilters.furnished && !(property.is_furnished || (String(property.furnishing || '').toLowerCase() === 'furnished'))) {
            return false;
        }
        
        
        // Polygon filter (point-in-polygon)
        if (Array.isArray(this.currentFilters.polygonPath) && this.currentFilters.polygonPath.length >= 3) {
            const path = this.currentFilters.polygonPath;
            if (!(property.latitude && property.longitude)) {
                return false;
            }
            try {
                // Build cached polygon if path changed
                const key = JSON.stringify(path);
                if (this._polygonCacheKey !== key) {
                    this._polygonCacheKey = key;
                    this._cachedPolygon = new google.maps.Polygon({
                        paths: path.map(p => new google.maps.LatLng(p.lat, p.lng))
                    });
                }
                if (google.maps.geometry && google.maps.geometry.poly && typeof google.maps.geometry.poly.containsLocation === 'function') {
                    const point = new google.maps.LatLng(property.latitude, property.longitude);
                    if (!google.maps.geometry.poly.containsLocation(point, this._cachedPolygon)) {
                        return false;
                    }
                } else {
                    // Fallback: quick bounding-box pre-check
                    const lats = path.map(p => p.lat);
                    const lngs = path.map(p => p.lng);
                    const minLat = Math.min(...lats), maxLat = Math.max(...lats);
                    const minLng = Math.min(...lngs), maxLng = Math.max(...lngs);
                    if (property.latitude < minLat || property.latitude > maxLat || property.longitude < minLng || property.longitude > maxLng) {
                        return false;
                    }
                }
            } catch (_) {
                // On any error, do not filter out
            }
        }
        
        return true;
    }
    

    /**
     * Update property display on map and list
     */
    updatePropertyDisplay() {
        if (this.markersManager) {
            const mappable = (this.filteredProperties || []).filter(p => this.isFiniteNumber(p?.latitude) && this.isFiniteNumber(p?.longitude));
            // Pass selected property ID to keep popup open
            const selectedId = this.state.selectedProperty?.id;
            this.markersManager.updateMarkers(mappable, selectedId);
        }
        
        // Ensure list uses current sort order
        this.sortFilteredProperties();
        this.updatePropertyList();
        this.updatePropertyCount();
        this.updateFavoritesUI();
    }

    /**
     * Sort filtered properties based on current selection
     */
    sortFilteredProperties() {
        if (!Array.isArray(this.filteredProperties) || this.filteredProperties.length === 0) return;
        const sort = this.state.currentSort || 'newest';
        const byNum = (v) => (typeof v === 'number' ? v : parseFloat(v) || 0);
        const byDate = (v) => (v ? new Date(v).getTime() : 0);
        const hasUserLoc = this.currentFilters.userLocation && this.utils && typeof this.utils.calculateDistance === 'function';

        switch (sort) {
            case 'price-low':
                this.filteredProperties.sort((a, b) => byNum(a.price) - byNum(b.price));
                break;
            case 'price-high':
                this.filteredProperties.sort((a, b) => byNum(b.price) - byNum(a.price));
                break;
            case 'distance':
                if (hasUserLoc) {
                    const { lat, lng } = this.currentFilters.userLocation;
                    this.filteredProperties.sort((a, b) => {
                        const da = (a.latitude && a.longitude) ? this.utils.calculateDistance(lat, lng, a.latitude, a.longitude) : Number.MAX_VALUE;
                        const db = (b.latitude && b.longitude) ? this.utils.calculateDistance(lat, lng, b.latitude, b.longitude) : Number.MAX_VALUE;
                        return da - db;
                    });
                }
                break;
            case 'relevance':
                // Placeholder: keep as-is or implement scoring later
                break;
            case 'newest':
            default:
                // Sort by created_at desc if present; fallback to id desc
                this.filteredProperties.sort((a, b) => (byDate(b.created_at) - byDate(a.created_at)) || (byNum(b.id) - byNum(a.id)));
        }
    }
    
    /**
     * Update property list in sidebar
     */
    updatePropertyList() {
        if (typeof window.updatePropertyListHTML === 'function') {
            window.updatePropertyListHTML(this.filteredProperties || []);
        } else {
            console.warn('updatePropertyListHTML function not found');
        }
    }

    /**
     * Favorites panel: render list and update counts
     */
    updateFavoritesUI() {
        const favIds = this.state.favorites || [];
        const favCountEl = document.getElementById('favorites-count');
        if (favCountEl) favCountEl.textContent = String(favIds.length);

        const favListEl = document.getElementById('favorites-list');
        if (!favListEl) return;

        // Map ids to property objects when available
        const items = favIds.map(id => (this.properties || []).find(p => p.id == id)).filter(Boolean);
        if (items.length === 0) {
            favListEl.innerHTML = '<div class="text-muted small">No favorites yet</div>';
            return;
        }

        const html = items.map(p => `
            <div class="favorite-item" data-id="${p.id}">
                <div class="fi-title">${p.title || 'Property #' + p.id}</div>
                <div class="fi-actions">
                    <button class="btn btn-xs btn-outline-danger fi-remove" data-id="${p.id}"><i class="fa fa-trash"></i></button>
                </div>
            </div>
        `).join('');
        favListEl.innerHTML = html;

        // Remove buttons
        favListEl.querySelectorAll('.fi-remove').forEach(btn => {
            btn.addEventListener('click', (e) => {
                const id = btn.getAttribute('data-id');
                const idNum = parseInt(id, 10);
                const ix = this.state.favorites.indexOf(isNaN(idNum) ? id : idNum);
                if (ix > -1) {
                    this.state.favorites.splice(ix, 1);
                    localStorage.setItem('property_favorites', JSON.stringify(this.state.favorites));
                    this.updateFavoritesUI();
                    // Also update any favorite icons in list/cards
                    if (this.propertyDetails) {
                        this.propertyDetails.updateFavoriteButtons(isNaN(idNum) ? id : idNum);
                        this.propertyDetails.updateDrawerFavoriteButton(isNaN(idNum) ? id : idNum);
                    }
                }
            });
        });

        // Update compare button enabled state
        const compareBtn = document.getElementById('compare-favorites');
        if (compareBtn) {
            const count = (this.state.compareList || []).length;
            compareBtn.disabled = count < 2;
            compareBtn.textContent = `Compare Selected (${count})`;
        }
    }

    /**
     * Bind favorites panel buttons and compare action
     */
    bindFavoritesPanelEvents() {
        const toggleFav = document.getElementById('toggle-favorites');
        const favContent = document.getElementById('favorites-content');
        if (toggleFav && favContent) {
            toggleFav.addEventListener('click', () => {
                favContent.classList.toggle('collapsed');
            });
        }

        const clearFav = document.getElementById('clear-favorites');
        if (clearFav) {
            clearFav.addEventListener('click', () => {
                this.state.favorites = [];
                localStorage.setItem('property_favorites', '[]');
                this.updateFavoritesUI();
                this.showNotification('Favorites cleared', 'info');
            });
        }

        const compareBtn = document.getElementById('compare-favorites');
        if (compareBtn) {
            compareBtn.addEventListener('click', async () => {
                try {
                    // Fetch compare data from backend for current compareList
                    const ids = (this.state.compareList || []).join(',');
                    if (!ids || this.state.compareList.length < 2) {
                        this.showNotification('Select at least 2 properties to compare', 'warning');
                        return;
                    }
                    const resp = await fetch(`/map/compare?ids=${encodeURIComponent(ids)}`, { headers: { 'Accept': 'application/json' } });
                    if (!resp.ok) throw new Error('Compare request failed');
                    const data = await resp.json();
                    // Simple preview: open a new tab with JSON or implement a modal renderer if available
                    const win = window.open('', '_blank');
                    win.document.write(`<pre>${JSON.stringify(data, null, 2)}</pre>`);
                    win.document.close();
                } catch (e) {
                    console.error(e);
                    this.showNotification('Unable to load comparison data', 'error');
                }
            });
        }
    }
    
    /**
     * Generate HTML for property list item
     */
    generatePropertyListItem(property) {
        const isFavorite = this.state.favorites.includes(property.id);
        const price = this.formatPrice(property.price);
        const image = property.image || '/assets/images/property-placeholder.jpg';
        
        return `
            <div class="property-list-item" data-property-id="${property.id}">
                <div class="property-image">
                    <img src="${image}" alt="${property.title}" loading="lazy">
                    <div class="property-badges">
                        ${property.is_verified ? '<span class="badge badge-success">Verified</span>' : ''}
                        ${property.is_featured ? '<span class="badge badge-warning">Featured</span>' : ''}
                    </div>
                </div>
                <div class="property-details">
                    <h6 class="property-title">${property.title}</h6>
                    <p class="property-location">
                        <i class="fa fa-map-marker"></i> ${property.location}
                    </p>
                    <div class="property-specs">
                        <span><i class="fa fa-bed"></i> ${property.bedrooms}</span>
                        <span><i class="fa fa-bath"></i> ${property.bathrooms}</span>
                        <span><i class="fa fa-home"></i> ${property.area} sq ft</span>
                    </div>
                    <div class="property-price">
                        <strong>${price}</strong>
                        <span class="price-type">/${property.purpose === 'rent' ? 'month' : 'total'}</span>
                    </div>
                </div>
                <div class="property-actions">
                    <button class="action-btn favorite-btn ${isFavorite ? 'active' : ''}" 
                            data-property-id="${property.id}" title="Add to Favorites">
                        <i class="fa fa-heart${isFavorite ? '' : '-o'}"></i>
                    </button>
                    <button class="action-btn share-btn" 
                            data-property-id="${property.id}" title="Share">
                        <i class="fa fa-share-alt"></i>
                    </button>
                </div>
            </div>
        `;
    }
    
    /**
     * Update property count display
     */
    updatePropertyCount() {
        const count = this.filteredProperties.length;
        const countElements = document.querySelectorAll('#property-count, #list-count');
        
        countElements.forEach(element => {
            if (element.id === 'property-count') {
                element.textContent = `${count} properties`;
            } else {
                element.textContent = `(${count})`;
            }
        });

        // Update pagination info if present
        const showingCount = document.getElementById('showing-count');
        const totalCount = document.getElementById('total-count');
        if (showingCount) showingCount.textContent = String(this.filteredProperties.length);
        if (totalCount) totalCount.textContent = String(this.properties.length);
    }
    
    /**
     * Update map statistics
     */
    updateStats() {
        if (!this.filteredProperties.length) return;
        
        const prices = this.filteredProperties.map(p => p.price).filter(p => p > 0);
        const avgPrice = prices.reduce((a, b) => a + b, 0) / prices.length;
        const minPrice = Math.min(...prices);
        const maxPrice = Math.max(...prices);
        
        const visibleCount = document.getElementById('visible-count');
        const averagePrice = document.getElementById('average-price');
        const priceRange = document.getElementById('price-range');
        
        if (visibleCount) visibleCount.textContent = this.filteredProperties.length;
        if (averagePrice) averagePrice.textContent = this.formatPrice(avgPrice);
        if (priceRange) priceRange.textContent = `${this.formatPrice(minPrice)} - ${this.formatPrice(maxPrice)}`;
    }
    
    /**
     * No-op advanced feature initializers (can be extended later)
     */
    initializeKeyboardShortcuts() { /* no-op */ }
    initializeGestureRecognition() { /* no-op */ }
    initializeRealTimeUpdates() { /* no-op */ }
    initializeAnalytics() { /* no-op */ }

    /**
     * Set loading state
     */
    setLoading(loading) {
        this.state.isLoading = loading;
        const loadingOverlay = document.getElementById('map-loading');
        if (loadingOverlay) {
            loadingOverlay.style.display = loading ? 'flex' : 'none';
        }
    }

    /**
     * Show notification
     */
    showNotification(message, type = 'info') {
        // Create notification element if it doesn't exist
        let notification = document.getElementById('map-notification');
        if (!notification) {
            notification = document.createElement('div');
            notification.id = 'map-notification';
            notification.className = 'map-notification';
            document.body.appendChild(notification);
        }

        notification.className = `map-notification ${type}`;
        notification.textContent = message;
        notification.style.display = 'block';

        // Auto hide after 3 seconds
        setTimeout(() => {
            notification.style.display = 'none';
        }, 3000);
    }

    /**
     * Small debounce utility
     */
    debounce(fn, wait = 300) {
        let t;
        return (...args) => {
            clearTimeout(t);
            t = setTimeout(() => fn.apply(this, args), wait);
        };
    }
}

// Export for use in other modules
window.MapView = MapView;
