/**
 * MapFilters Module
 * Handles all filter-related functionality for the map view
 */
class MapFilters {
    constructor(mapView) {
        this.mapView = mapView;
        this.debounceTimeout = null;
        this.drawingManager = null;
        this.activePolygon = null;
        
        this.init();
    }
    
    /**
     * Initialize filter functionality
     */
    init() {
        this.initializeSearchFilters();
        this.initializePriceFilters();
        this.initializePropertyFilters();
        this.initializeFeatureFilters();
        this.initializeMapTools();
        this.initializeAreaFilters();
        // Removed initializeStatusAndSalePrice() - method doesn't exist
    }
    
    /**
     * Initialize search-related filters
     */
    initializeSearchFilters() {
        // Location search (use 'search' to match backend)
        const locationSearch = document.getElementById('location-search');
        if (locationSearch) {
            locationSearch.addEventListener('input', (e) => {
                this.debounceFilter(() => {
                    this.mapView.currentFilters.search = e.target.value;
                    this.mapView.applyFilters();
                });
            });
        }

        // GPS location button
        const gpsBtn = document.getElementById('use-gps');
        if (gpsBtn) {
            gpsBtn.addEventListener('click', () => {
                this.useCurrentLocation();
            });
        }

        // Search radius
        const radiusSlider = document.getElementById('search-radius');
        if (radiusSlider) {
            radiusSlider.addEventListener('input', (e) => {
                const value = e.target.value;
                document.getElementById('radius-value').textContent = `${value} km`;
                const hiddenRadius = document.getElementById('radius');
                if (hiddenRadius) hiddenRadius.value = value;

                this.debounceFilter(() => {
                    this.mapView.currentFilters.radius = value;
                    this.mapView.applyFilters();
                });
            });
        }
    }
    
    /**
     * Initialize price-related filters
     */
    initializePriceFilters() {
        // Price filters are handled by the noUiSlider in the form
        // No additional setup needed here
    }
    
    /**
     * Initialize property-related filters
     */
    initializePropertyFilters() {
        // Property type
        const propertyType = document.getElementById('map-property-type');
        if (propertyType) {
            propertyType.addEventListener('change', (e) => {
                this.mapView.currentFilters.type_id = e.target.value;
                this.mapView.applyFilters();
            });
        }

        // Location
        const location = document.getElementById('filter-location');
        if (location) {
            location.addEventListener('change', (e) => {
                this.mapView.currentFilters.location_id = e.target.value;
                this.mapView.applyFilters();
            });
        }

        // Bedrooms
        const bedrooms = document.getElementById('map-bedrooms');
        if (bedrooms) {
            bedrooms.addEventListener('change', (e) => {
                this.mapView.currentFilters.bedrooms = e.target.value;
                this.mapView.applyFilters();
            });
        }

        // Bathrooms
        const bathrooms = document.getElementById('map-bathrooms');
        if (bathrooms) {
            bathrooms.addEventListener('change', (e) => {
                this.mapView.currentFilters.bathrooms = e.target.value;
                this.mapView.applyFilters();
            });
        }

        // Purpose (handled by tabs in the form)
        // The purpose tabs are handled by the inline JavaScript in map.blade.php
    }

    /**
     * Initialize area filters
     */
    initializeAreaFilters() {
        // Area filters are not currently implemented
        // This method is a placeholder for future functionality
    }
    
    /**
     * Initialize feature-related filters
     */
    initializeFeatureFilters() {
        // Verified properties
        const verified = document.getElementById('filter-verified');
        if (verified) {
            verified.addEventListener('change', (e) => {
                // Use checked state; send 1 when checked, remove when unchecked
                if (e.target.checked) {
                    this.mapView.currentFilters.verified = 1;
                } else {
                    delete this.mapView.currentFilters.verified;
                }
                this.mapView.applyFilters();
            });
        }

        // Furnished properties
        const furnishing = document.getElementById('filter-furnishing');
        if (furnishing) {
            furnishing.addEventListener('change', (e) => {
                this.mapView.currentFilters.furnishing = e.target.value;
                this.mapView.applyFilters();
            });
        }
    }
    
    /**
     * Initialize map tools
     */
    initializeMapTools() {
        // Map tools are handled by the map controls in map-controls.blade.php
        // No additional setup needed here
    }
    
    /**
     * Handle location search
     */
    handleLocationSearch(location) {
        if (!location.trim()) return;

        // In real implementation, use geocoding service
        // For now, just set as search filter
        this.mapView.currentFilters.search_text = location;
        this.mapView.applyFilters();
    }
    
    /**
     * Use current GPS location
     */
    useCurrentLocation() {
        if (!navigator.geolocation) {
            this.mapView.showNotification('Geolocation not supported', 'error');
            return;
        }
        
        const gpsBtn = document.getElementById('use-gps');
        if (gpsBtn) {
            gpsBtn.innerHTML = '<i class="fa fa-spinner fa-spin"></i>';
        }
        
        navigator.geolocation.getCurrentPosition(
            (position) => {
                const lat = position.coords.latitude;
                const lng = position.coords.longitude;
                
                // Center map on user location (Google Maps)
                if (this.mapView.map) {
                    this.mapView.state.autoFit = false; // prevent jumpy refits
                    try { this.mapView.map.setZoom(14); } catch (_) {}
                    try { this.mapView.map.panTo({ lat, lng }); } catch (_) {}
                }
                
                // Update location filter
                this.mapView.currentFilters.userLocation = { lat, lng };
                this.mapView.applyFilters();
                
                // Update location input with reverse geocoding (simplified)
                const locationInput = document.getElementById('location-search');
                if (locationInput) {
                    locationInput.value = `${lat.toFixed(4)}, ${lng.toFixed(4)}`;
                }
                
                this.mapView.showNotification('Location updated', 'success');
                
                if (gpsBtn) {
                    gpsBtn.innerHTML = '<i class="fa fa-crosshairs"></i>';
                }
            },
            (error) => {
                this.mapView.showNotification('Unable to get location', 'error');
                if (gpsBtn) {
                    gpsBtn.innerHTML = '<i class="fa fa-crosshairs"></i>';
                }
            }
        );
    }
    
    /**
     * Toggle drawing mode
     */
    toggleDrawingMode() {
        // Drawing tools are handled by the map controls
        // This method is not used in the current implementation
    }

    /**
     * Clear drawn polygons
     */
    clearDrawnPolygons() {
        // Drawing tools are handled by the map controls
        // This method is not used in the current implementation
    }

    /**
     * Save current search
     */
    saveCurrentSearch() {
        // Save search functionality could be implemented here if needed
        this.mapView.showNotification('Save search feature not implemented', 'info');
    }

    /**
     * Reset all filters
     */
    resetFilters() {
        // Reset functionality is handled by the form's reset button
        // This method is not used in the current implementation
    }
    
    /**
     * Debounce filter updates
     */
    debounceFilter(callback, delay = 300) {
        clearTimeout(this.debounceTimeout);
        this.debounceTimeout = setTimeout(callback, delay);
    }
}

// Export for use in other modules
window.MapFilters = MapFilters;
