/**
 * MapInteractions Module
 * Handles user interactions with the map and UI elements
 */
class MapInteractions {
    constructor(mapView) {
        this.mapView = mapView;
        this.touchStartTime = 0;
        this.touchStartPos = null;
        
        this.init();
    }
    
    /**
     * Initialize interaction functionality
     */
    init() {
        this.initializeMapControls();
        this.initializeTouchInteractions();
        this.initializeKeyboardShortcuts();
        this.initializeResizeHandler();
    }
    
    /**
     * Initialize map control interactions
     */
    initializeMapControls() {
        // Custom zoom controls
        const zoomInBtn = document.getElementById('map-zoom-in');
        const zoomOutBtn = document.getElementById('map-zoom-out');
        
        if (zoomInBtn) {
            zoomInBtn.addEventListener('click', () => {
                const z = this.mapView.map.getZoom() || 0;
                this.mapView.map.setZoom(z + 1);
            });
        }
        
        if (zoomOutBtn) {
            zoomOutBtn.addEventListener('click', () => {
                const z = this.mapView.map.getZoom() || 0;
                this.mapView.map.setZoom(z - 1);
            });
        }
        
        // Locate control
        const locateBtn = document.getElementById('map-locate');
        if (locateBtn) {
            locateBtn.addEventListener('click', () => {
                this.mapView.locateUser();
            });
        }
        
        // Layers control
        const layersBtn = document.getElementById('map-layers');
        if (layersBtn) {
            layersBtn.addEventListener('click', () => {
                this.toggleLayersPanel();
            });
        }
        
        // Fullscreen control
        const fullscreenBtn = document.getElementById('map-fullscreen');
        if (fullscreenBtn) {
            fullscreenBtn.addEventListener('click', () => {
                const el = document.getElementById(this.mapView.options.mapContainer);
                if (!document.fullscreenElement) {
                    el.requestFullscreen?.();
                    this.mapView.state.isFullscreen = true;
                } else {
                    document.exitFullscreen?.();
                    this.mapView.state.isFullscreen = false;
                }
            });
        }
        
        // Legend toggle
        const legendToggle = document.getElementById('toggle-legend');
        if (legendToggle) {
            legendToggle.addEventListener('click', () => {
                this.toggleLegend();
            });
        }
    }
    
    /**
     * Initialize touch interactions for mobile
     */
    initializeTouchInteractions() {
        const mapContainer = document.getElementById(this.mapView.options.mapContainer);
        if (!mapContainer) return;
        
        // Touch events for mobile gestures
        mapContainer.addEventListener('touchstart', (e) => {
            this.handleTouchStart(e);
        }, { passive: true });
        
        mapContainer.addEventListener('touchmove', (e) => {
            this.handleTouchMove(e);
        }, { passive: true });
        
        mapContainer.addEventListener('touchend', (e) => {
            this.handleTouchEnd(e);
        }, { passive: true });
        
        // Double tap to zoom
        let lastTap = 0;
        mapContainer.addEventListener('touchend', (e) => {
            const currentTime = new Date().getTime();
            const tapLength = currentTime - lastTap;
            
            if (tapLength < 500 && tapLength > 0) {
                e.preventDefault();
                this.handleDoubleTap(e);
            }
            
            lastTap = currentTime;
        });
        
        // Prevent context menu on long press
        mapContainer.addEventListener('contextmenu', (e) => {
            e.preventDefault();
        });
    }
    
    /**
     * Initialize keyboard shortcuts
     */
    initializeKeyboardShortcuts() {
        document.addEventListener('keydown', (e) => {
            // Only handle shortcuts when map is focused or no input is active
            if (document.activeElement.tagName === 'INPUT' || 
                document.activeElement.tagName === 'TEXTAREA') {
                return;
            }
            
            switch (e.key) {
                case 'f':
                case 'F':
                    if (e.ctrlKey || e.metaKey) {
                        e.preventDefault();
                        this.focusSearch();
                    } else {
                        const el = document.getElementById(this.mapView.options.mapContainer);
                        if (!document.fullscreenElement) {
                            el.requestFullscreen?.();
                            this.mapView.state.isFullscreen = true;
                        } else {
                            document.exitFullscreen?.();
                            this.mapView.state.isFullscreen = false;
                        }
                    }
                    break;
                    
                case 'Escape':
                    this.handleEscapeKey();
                    break;
                    
                case '+':
                case '=':
                    e.preventDefault();
                    this.mapView.map.setZoom((this.mapView.map.getZoom() || 0) + 1);
                    break;
                    
                case '-':
                case '_':
                    e.preventDefault();
                    this.mapView.map.setZoom((this.mapView.map.getZoom() || 0) - 1);
                    break;
                    
                case 'l':
                case 'L':
                    this.mapView.locateUser();
                    break;
                    
                case 'd':
                case 'D':
                    this.mapView.toggleDarkMode();
                    break;
                    
                case 'h':
                case 'H':
                    this.toggleHeatmap();
                    break;
                    
                case 'r':
                case 'R':
                    if (e.ctrlKey || e.metaKey) {
                        e.preventDefault();
                        this.mapView.clearAllFilters();
                    }
                    break;
            }
        });
    }
    
    /**
     * Initialize window resize handler
     */
    initializeResizeHandler() {
        let resizeTimeout;
        
        window.addEventListener('resize', () => {
            clearTimeout(resizeTimeout);
            resizeTimeout = setTimeout(() => {
                this.handleResize();
            }, 250);
        });
        
        // Handle orientation change on mobile
        window.addEventListener('orientationchange', () => {
            setTimeout(() => {
                this.handleResize();
            }, 500);
        });
    }
    
    /**
     * Handle touch start
     */
    handleTouchStart(e) {
        this.touchStartTime = Date.now();
        
        if (e.touches.length === 1) {
            this.touchStartPos = {
                x: e.touches[0].clientX,
                y: e.touches[0].clientY
            };
        }
    }
    
    /**
     * Handle touch move
     */
    handleTouchMove(e) {
        // Handle multi-touch gestures
        if (e.touches.length === 2) {
            this.handlePinchGesture(e);
        }
    }
    
    /**
     * Handle touch end
     */
    handleTouchEnd(e) {
        const touchDuration = Date.now() - this.touchStartTime;
        
        // Long press detection (for context menu)
        if (touchDuration > 800 && this.touchStartPos) {
            this.handleLongPress(e);
        }
    }
    
    /**
     * Handle double tap
     */
    handleDoubleTap(e) {
        const rect = e.target.getBoundingClientRect();
        const x = e.changedTouches[0].clientX - rect.left;
        const y = e.changedTouches[0].clientY - rect.top;
        
        // Convert to map coordinates and zoom in
        const point = this.containerPixelToLatLng({ x, y });
        if (point) {
            this.mapView.map.setZoom((this.mapView.map.getZoom() || 0) + 1);
            this.mapView.map.panTo(point);
        }
    }
    
    /**
     * Handle pinch gesture for zoom
     */
    handlePinchGesture(e) {
        // Native Maps handles pinch zoom; nothing custom required
    }
    
    /**
     * Handle long press
     */
    handleLongPress(e) {
        // Show context menu or property options
        this.showContextMenu(e);
    }
    
    /**
     * Handle window resize
     */
    handleResize() {
        // Trigger Google Maps resize
        if (this.mapView.map) {
            google.maps.event.trigger(this.mapView.map, 'resize');
        }
        
        // Adjust mobile layout
        this.adjustMobileLayout();
        
        // Reposition floating elements
        this.repositionFloatingElements();
    }
    
    /**
     * Adjust layout for mobile devices
     */
    adjustMobileLayout() {
        const isMobile = window.innerWidth <= 768;
        const leftPanel = document.getElementById('map-left-panel');
        const propertyDrawer = document.getElementById('property-drawer');
        
        if (isMobile) {
            // Mobile-specific adjustments
            if (leftPanel) {
                leftPanel.classList.add('mobile-panel');
            }
            
            if (propertyDrawer) {
                propertyDrawer.classList.add('mobile-drawer');
            }
            
            // Hide some controls on mobile
            this.hideNonEssentialControls();
        } else {
            // Desktop layout
            if (leftPanel) {
                leftPanel.classList.remove('mobile-panel');
            }
            
            if (propertyDrawer) {
                propertyDrawer.classList.remove('mobile-drawer');
            }
            
            // Show all controls on desktop
            this.showAllControls();
        }
    }
    
    /**
     * Hide non-essential controls on mobile
     */
    hideNonEssentialControls() {
        const controls = document.querySelectorAll('.btn-text');
        controls.forEach(control => {
            control.style.display = 'none';
        });
        
        // Hide map stats panel on mobile
        const statsPanel = document.getElementById('map-stats-panel');
        if (statsPanel) {
            statsPanel.style.display = 'none';
        }
    }
    
    /**
     * Show all controls on desktop
     */
    showAllControls() {
        const controls = document.querySelectorAll('.btn-text');
        controls.forEach(control => {
            control.style.display = '';
        });
        
        // Show map stats panel on desktop
        const statsPanel = document.getElementById('map-stats-panel');
        if (statsPanel) {
            statsPanel.style.display = '';
        }
    }
    
    /**
     * Reposition floating elements
     */
    repositionFloatingElements() {
        // Adjust legend position
        const legend = document.getElementById('map-legend');
        if (legend && window.innerWidth <= 768) {
            legend.style.position = 'relative';
            legend.style.bottom = 'auto';
            legend.style.left = 'auto';
        } else if (legend) {
            legend.style.position = 'absolute';
            legend.style.bottom = '20px';
            legend.style.left = '20px';
        }
    }
    
    /**
     * Show context menu
     */
    showContextMenu(e) {
        // Create context menu if it doesn't exist
        let contextMenu = document.getElementById('map-context-menu');
        if (!contextMenu) {
            contextMenu = document.createElement('div');
            contextMenu.id = 'map-context-menu';
            contextMenu.className = 'map-context-menu';
            document.body.appendChild(contextMenu);
        }
        
        // Set context menu content
        contextMenu.innerHTML = `
            <div class="context-menu-item" data-action="add-marker">
                <i class="fa fa-map-marker"></i> Add Marker Here
            </div>
            <div class="context-menu-item" data-action="get-directions">
                <i class="fa fa-directions"></i> Get Directions
            </div>
            <div class="context-menu-item" data-action="search-nearby">
                <i class="fa fa-search"></i> Search Nearby
            </div>
            <div class="context-menu-item" data-action="street-view">
                <i class="fa fa-street-view"></i> Street View
            </div>
        `;
        
        // Position context menu
        const rect = e.target.getBoundingClientRect();
        const x = e.changedTouches ? e.changedTouches[0].clientX : e.clientX;
        const y = e.changedTouches ? e.changedTouches[0].clientY : e.clientY;
        
        contextMenu.style.left = `${x}px`;
        contextMenu.style.top = `${y}px`;
        contextMenu.style.display = 'block';
        
        // Add event listeners
        contextMenu.querySelectorAll('.context-menu-item').forEach(item => {
            item.addEventListener('click', (e) => {
                const action = e.target.closest('.context-menu-item').dataset.action;
                this.handleContextMenuAction(action, { x, y });
                this.hideContextMenu();
            });
        });
        
        // Hide context menu when clicking elsewhere
        setTimeout(() => {
            document.addEventListener('click', this.hideContextMenu.bind(this), { once: true });
        }, 100);
    }
    
    /**
     * Hide context menu
     */
    hideContextMenu() {
        const contextMenu = document.getElementById('map-context-menu');
        if (contextMenu) {
            contextMenu.style.display = 'none';
        }
    }
    
    /**
     * Handle context menu action
     */
    handleContextMenuAction(action, position) {
        const container = document.getElementById(this.mapView.options.mapContainer);
        const rect = container.getBoundingClientRect();
        const rel = { x: position.x - rect.left, y: position.y - rect.top };
        const point = this.containerPixelToLatLng(rel);
        
        switch (action) {
            case 'add-marker':
                this.addCustomMarker(point);
                break;
                
            case 'get-directions':
                this.getDirections(point);
                break;
                
            case 'search-nearby':
                this.searchNearby(point);
                break;
                
            case 'street-view':
                this.openStreetView(point);
                break;
        }
    }
    
    /**
     * Add custom marker
     */
    addCustomMarker(point) {
        const marker = new google.maps.Marker({
            position: { lat: point.lat, lng: point.lng },
            map: this.mapView.map,
            title: 'Custom Marker'
        });
        const info = new google.maps.InfoWindow({
            content: `
                <div class="custom-marker-popup">
                    <h6>Custom Marker</h6>
                    <p>Lat: ${point.lat.toFixed(6)}</p>
                    <p>Lng: ${point.lng.toFixed(6)}</p>
                    <button class="btn btn-sm btn-danger remove-marker" id="remove-marker-btn">Remove</button>
                </div>
            `
        });
        info.open({ map: this.mapView.map, anchor: marker });
        // Remove handler
        setTimeout(() => {
            const btn = document.getElementById('remove-marker-btn');
            if (btn) btn.addEventListener('click', () => { marker.setMap(null); info.close(); });
        }, 50);
    }
    
    /**
     * Get directions to point
     */
    getDirections(point) {
        if (navigator.geolocation) {
            navigator.geolocation.getCurrentPosition((position) => {
                const userLat = position.coords.latitude;
                const userLng = position.coords.longitude;
                
                // Open Google Maps directions
                const url = `https://www.google.com/maps/dir/${userLat},${userLng}/${point.lat},${point.lng}`;
                window.open(url, '_blank');
            });
        } else {
            // Fallback: open Google Maps without starting point
            const url = `https://www.google.com/maps/place/${point.lat},${point.lng}`;
            window.open(url, '_blank');
        }
    }
    
    /**
     * Search nearby properties
     */
    searchNearby(point) {
        // Set location filter and search radius
        this.mapView.currentFilters.location = `${point.lat},${point.lng}`;
        this.mapView.currentFilters.radius = 2; // 2km radius
        
        // Update location input
        const locationInput = document.getElementById('location-search');
        if (locationInput) {
            locationInput.value = `${point.lat.toFixed(4)}, ${point.lng.toFixed(4)}`;
        }
        
        // Apply filters
        this.mapView.applyFilters();
        
        this.mapView.showNotification('Searching nearby properties...', 'info');
    }
    
    /**
     * Open street view
     */
    openStreetView(point) {
        const url = `https://www.google.com/maps/@?api=1&map_action=pano&viewpoint=${point.lat},${point.lng}`;
        window.open(url, '_blank');
    }
    
    /**
     * Toggle layers panel
     */
    toggleLayersPanel() {
        // Create layers panel if it doesn't exist
        let layersPanel = document.getElementById('layers-panel');
        if (!layersPanel) {
            layersPanel = this.createLayersPanel();
        }
        
        // Toggle visibility
        const isVisible = layersPanel.style.display === 'block';
        layersPanel.style.display = isVisible ? 'none' : 'block';
    }
    
    /**
     * Create layers panel
     */
    createLayersPanel() {
        const panel = document.createElement('div');
        panel.id = 'layers-panel';
        panel.className = 'layers-panel';
        panel.innerHTML = `
            <div class="panel-header">
                <h6>Map Layers</h6>
                <button class="close-panel" onclick="this.parentElement.parentElement.style.display='none'">
                    <i class="fa fa-times"></i>
                </button>
            </div>
            <div class="panel-content">
                <div class="layer-option">
                    <label>
                        <input type="radio" name="base-layer" value="street" checked>
                        <span>Street Map</span>
                    </label>
                </div>
                <div class="layer-option">
                    <label>
                        <input type="radio" name="base-layer" value="satellite">
                        <span>Satellite</span>
                    </label>
                </div>
                <div class="layer-option">
                    <label>
                        <input type="radio" name="base-layer" value="dark">
                        <span>Dark Mode</span>
                    </label>
                </div>
                <hr>
                <div class="layer-option">
                    <label>
                        <input type="checkbox" id="heatmap-toggle">
                        <span>Property Heatmap</span>
                    </label>
                </div>
                <div class="layer-option">
                    <label>
                        <input type="checkbox" id="clusters-toggle" checked>
                        <span>Marker Clustering</span>
                    </label>
                </div>
            </div>
        `;
        
        document.body.appendChild(panel);
        
        // Add event listeners
        panel.querySelectorAll('input[name="base-layer"]').forEach(radio => {
            radio.addEventListener('change', (e) => {
                this.switchBaseLayer(e.target.value);
            });
        });
        
        const heatmapToggle = panel.querySelector('#heatmap-toggle');
        if (heatmapToggle) {
            heatmapToggle.addEventListener('change', (e) => {
                this.toggleHeatmap();
            });
        }
        
        const clustersToggle = panel.querySelector('#clusters-toggle');
        if (clustersToggle) {
            clustersToggle.addEventListener('change', (e) => {
                this.toggleClustering(e.target.checked);
            });
        }
        
        return panel;
    }
    
    /**
     * Switch base layer
     */
    switchBaseLayer(layerType) {
        switch (layerType) {
            case 'street':
                this.mapView.map.setMapTypeId(google.maps.MapTypeId.ROADMAP);
                this.mapView.map.setOptions({ styles: null });
                break;
            case 'satellite':
                this.mapView.map.setMapTypeId(google.maps.MapTypeId.HYBRID);
                this.mapView.map.setOptions({ styles: null });
                break;
            case 'dark':
                this.mapView.map.setMapTypeId(google.maps.MapTypeId.ROADMAP);
                this.mapView.map.setOptions({ styles: this.mapView.darkMapStyle });
                break;
        }
    }
    
    /**
     * Toggle heatmap
     */
    toggleHeatmap() {
        if (this.mapView.markersManager) {
            const isVisible = this.mapView.markersManager.toggleHeatmap();
            this.mapView.showNotification(
                isVisible ? 'Heatmap enabled' : 'Heatmap disabled',
                'info'
            );
        }
    }
    
    /**
     * Toggle clustering
     */
    toggleClustering(enabled) {
        // This would require rebuilding the marker layer
        // For now, just show notification
        this.mapView.showNotification(
            enabled ? 'Clustering enabled' : 'Clustering disabled',
            'info'
        );
    }
    
    /**
     * Toggle legend
     */
    toggleLegend() {
        const legend = document.getElementById('map-legend');
        const legendContent = legend?.querySelector('.legend-content');
        const toggleBtn = document.getElementById('toggle-legend');
        
        if (legendContent && toggleBtn) {
            const isVisible = legendContent.style.display !== 'none';
            legendContent.style.display = isVisible ? 'none' : 'block';
            
            const icon = toggleBtn.querySelector('i');
            if (icon) {
                icon.className = isVisible ? 'fa fa-chevron-down' : 'fa fa-chevron-up';
            }
        }
    }
    
    /**
     * Focus search input
     */
    focusSearch() {
        const searchInput = document.getElementById('map-search');
        if (searchInput) {
            searchInput.focus();
            searchInput.select();
        }
    }
    
    /**
     * Handle escape key
     */
    handleEscapeKey() {
        // Close any open modals or panels
        this.hideContextMenu();
        
        const layersPanel = document.getElementById('layers-panel');
        if (layersPanel) {
            layersPanel.style.display = 'none';
        }
        
        // Close property drawer
        const propertyDrawer = document.getElementById('property-drawer');
        if (propertyDrawer && propertyDrawer.classList.contains('open')) {
            propertyDrawer.classList.remove('open');
        }
        
        // Exit fullscreen
        if (document.fullscreenElement) {
            document.exitFullscreen?.();
            this.mapView.state.isFullscreen = false;
        }
    }

    /**
     * Convert container pixel coordinates to LatLng using OverlayView projection
     */
    containerPixelToLatLng(pixel) {
        if (!this.projectionHelper) {
            // Create a lightweight OverlayView to access projection
            const overlay = new google.maps.OverlayView();
            overlay.onAdd = function() {};
            overlay.draw = function() {};
            overlay.onRemove = function() {};
            overlay.setMap(this.mapView.map);
            this.projectionHelper = overlay;
        }
        const proj = this.projectionHelper.getProjection();
        if (!proj) return null;
        const point = new google.maps.Point(pixel.x, pixel.y);
        return proj.fromDivPixelToLatLng(point);
    }
}

// Export for use in other modules
window.MapInteractions = MapInteractions;
