/**
 * MapMarkers Module
 * Handles marker creation, clustering, and interactions on the map
 */
class MapMarkers {
    constructor(mapView) {
        this.mapView = mapView;
        this.propertyMarkers = new Map();
        this.highlightedMarker = null;
        this.infoWindow = new google.maps.InfoWindow();
        this.heatmapLayer = null;
        this.clusterer = null;
        this._clusterLibLoaded = false;
        this._clusterLibLoading = false;
        
        this.init();
    }
    
    /**
     * Initialize marker functionality
     */
    init() {
        this.createCustomIcons();
    }

    /**
     * Get and cache the popup HTML template from DOM
     */
    getPopupTemplate() {
        if (this._popupTemplate !== undefined) return this._popupTemplate;
        const tplEl = document.getElementById('property-popup-template');
        this._popupTemplate = tplEl ? tplEl.innerHTML : null;
        return this._popupTemplate;
    }
    
    /**
     * Create custom marker icons
     */
    createCustomIcons() {
        // Using default Google markers for now; can be replaced with custom SVGs if desired
        this.icons = {
            sale: null,
            rent: null,
            verified: null,
            featured: null,
            highlighted: null
        };
    }
    
    /**
     * Create custom marker content for AdvancedMarkerElement
     */
    createMarkerContent(property) {
        const markerDiv = document.createElement('div');
        markerDiv.className = 'custom-marker';

        const price = this.mapView.formatPrice ? this.mapView.formatPrice(property.price) : `₹${property.price}`;
        const purposeClass = property.purpose === 'rent' ? 'marker-rent' : 'marker-sale';

        markerDiv.innerHTML = `
            <div class="marker-content ${purposeClass}" data-purpose="${property.purpose || 'sale'}">
                <span class="marker-price">${price}</span>
                ${property.is_verified ? '<i class="fa fa-check-circle marker-verified" aria-hidden="true"></i>' : ''}
                <div class="marker-caret"></div>
            </div>
        `;

        // Hover effects via class toggle (handled by CSS)
        markerDiv.addEventListener('mouseenter', () => {
            const content = markerDiv.querySelector('.marker-content');
            content && content.classList.add('hover');
        });

        markerDiv.addEventListener('mouseleave', () => {
            const content = markerDiv.querySelector('.marker-content');
            content && content.classList.remove('hover');
        });

        return markerDiv;
    }
    
    /**
     * Update markers on the map
     * @param {Array} properties
     * @param {string|number} [selectedPropertyId] - Optional property ID to keep popup open
     */
    updateMarkers(properties, selectedPropertyId) {
        // Clear existing markers
        this.clearMarkers();
        this.propertyMarkers.clear();
        if (!properties || properties.length === 0) {
            return;
        }
        // Create markers for each property
        const googleMarkers = [];
        properties.forEach(property => {
            const marker = this.createPropertyMarker(property);
            if (marker) {
                this.propertyMarkers.set(property.id, marker);
                googleMarkers.push(marker);
            }
        });
        this.mapView.markers = googleMarkers;
        // Update heatmap if enabled
        this.updateHeatmap(properties);
        // Apply clustering if library is available (loaded lazily)
        this.applyClustering(googleMarkers);
        // --- Keep InfoWindow open for selected property ---
        if (selectedPropertyId) {
            this.highlightProperty(selectedPropertyId);
        }
    }

    /**
     * Dynamically load MarkerClusterer library from CDN once
     */
    async ensureClustererLib() {
        if (this._clusterLibLoaded || window.markerClusterer || window.MarkerClusterer) return true;
        if (this._clusterLibLoading) {
            // Wait until loaded
            await new Promise(r => setTimeout(r, 100));
            return !!(window.markerClusterer || window.MarkerClusterer);
        }
        this._clusterLibLoading = true;
        try {
            await new Promise((resolve, reject) => {
                const s = document.createElement('script');
                s.src = 'https://unpkg.com/@googlemaps/markerclusterer/dist/index.min.js';
                s.async = true;
                s.onload = () => resolve();
                s.onerror = reject;
                document.head.appendChild(s);
            });
            this._clusterLibLoaded = true;
            return true;
        } catch (_) {
            // Fallback: no clustering
            return false;
        } finally {
            this._clusterLibLoading = false;
        }
    }

    /**
     * Apply clustering if the library is available, otherwise keep plain markers
     */
    async applyClustering(markers) {
        if (!Array.isArray(markers) || markers.length === 0) return;
        const ok = await this.ensureClustererLib();
        if (!ok) return;

        try {
            // Support both @googlemaps/markerclusterer IIFE (window.markerClusterer.MarkerClusterer)
            // and older MarkerClustererPlus global (window.MarkerClusterer)
            if (this.clusterer) {
                try { this.clusterer.clearMarkers?.(); } catch (_) {}
                try { this.clusterer.setMap?.(null); } catch (_) {}
                this.clusterer = null;
            }

            if (window.markerClusterer && window.markerClusterer.MarkerClusterer) {
                this.clusterer = new window.markerClusterer.MarkerClusterer({
                    map: this.mapView.map,
                    markers,
                    // Basic options; could be customized via mapView.options later
                    onClusterClick: undefined
                });
            } else if (window.MarkerClusterer) {
                // Legacy API: new MarkerClusterer(map, markers, options)
                this.clusterer = new window.MarkerClusterer(this.mapView.map, markers, {
                    maxZoom: this.mapView?.options?.maxZoomForClustering || 15
                });
            }
        } catch (e) {
            // If clustering fails, just leave markers as-is
            console.warn('Marker clustering not applied:', e);
        }
    }
    
    /**
     * Create a marker for a property using AdvancedMarkerElement
     */
    createPropertyMarker(property) {
        if (!property.latitude || !property.longitude) {
            return null;
        }
        
        // Create custom marker content
        const markerContent = this.createMarkerContent(property);
        const position = { lat: property.latitude, lng: property.longitude };
        
        let marker;
        
        // Use AdvancedMarkerElement if available, fallback to legacy Marker
        if (google.maps.marker && google.maps.marker.AdvancedMarkerElement) {
            marker = new google.maps.marker.AdvancedMarkerElement({
                position,
                map: this.mapView.map,
                title: property.title,
                content: markerContent
            });
        } else {
            // Fallback to legacy marker for older API versions
            marker = new google.maps.Marker({
                position,
                map: this.mapView.map,
                title: property.title
            });
        }
        
        // Click event
        marker.addListener('click', () => {
            this.onMarkerClick(property);
            const popupContent = this.createPopupContent(property);
            this.infoWindow.setContent(popupContent);
            this.infoWindow.open({ map: this.mapView.map, anchor: marker });
            // Attach popup action listeners after open
            setTimeout(() => this.attachPopupEventListeners(property), 50);
        });
        
        // Hover events (limited support in Google Maps)
        marker.addListener('mouseover', () => {
            this.onMarkerHover(property, marker);
        });
        marker.addListener('mouseout', () => {
            this.onMarkerLeave(property, marker);
        });
        
        return marker;
    }
    
    /**
     * Create popup content for property
     */
    createPopupContent(property) {
        const tpl = this.getPopupTemplate();
        const price = this.mapView.formatPrice(property.price);
        const image = property.image || '/assets/images/property-placeholder.jpg';
        const isFavorite = this.mapView.state.favorites.includes(property.id);
        const inCompare = this.mapView.state.compareList.includes(property.id);

        if (!tpl) {
            // Fallback to inline structure if template not found
            const purposeClass = property.purpose === 'rent' ? 'rent' : 'sale';
            const compareCount = this.mapView.state.compareList.length;
            return `
                <div class="property-popup-modern">
                    <div class="popup-img-wrap">
                        <img class="popup-img" src="${image}" alt="${property.title}" loading="lazy">
                        <div class="popup-badges">
                            ${property.is_verified ? '<span class="badge-verified" title="Verified"><i class="fa fa-check-circle"></i> Verified</span>' : ''}
                            ${property.is_featured ? '<span class="badge-featured" title="Featured"><i class="fa fa-star"></i> Featured</span>' : ''}
                        </div>
                        <div class="popup-price-badge">
                            <span class="price">${price}</span>
                            <span class="purpose ${purposeClass}">${purposeClass === 'rent' ? 'For Rent' : 'For Sale'}</span>
                        </div>
                    </div>
                    <div class="popup-main">
                        <div class="popup-header">
                            <div class="popup-title" title="${property.title}">${property.title}</div>
                            <span class="popup-location" title="${property.location}"><i class="fa fa-map-marker"></i> ${property.location}</span>
                        </div>
                        <div class="popup-specs">
                            <span class="spec-chip" title="Bedrooms"><i class="fa fa-bed"></i> ${property.bedrooms || 0} <span class="spec-label">Bed</span></span>
                            <span class="spec-chip" title="Bathrooms"><i class="fa fa-bath"></i> ${property.bathrooms || 0} <span class="spec-label">Bath</span></span>
                            ${property.area ? `<span class="spec-chip" title="Area"><i class="fa fa-home"></i> ${property.area} <span class="spec-label">sqft</span></span>` : ''}
                        </div>
                        <div class="popup-actions">
                            <a class="btn-modern btn-primary view-details-btn" data-property-id="${property.id}" href="${property.url || '#'}" target="_blank" rel="noopener"><i class="fa fa-eye"></i> View</a>
                            <button class="btn-modern btn-fav favorite-btn ${isFavorite ? 'active' : ''}" data-property-id="${property.id}" title="${isFavorite ? 'Remove from favorites' : 'Add to favorites'}"><i class="fa ${isFavorite ? 'fa-heart' : 'fa-heart-o'}"></i></button>
                            <button class="btn-modern btn-share compare-btn ${inCompare ? 'active' : ''}" data-property-id="${property.id}" title="${inCompare ? 'Remove from compare' : 'Add to compare'}"><i class="fa fa-balance-scale"></i></button>
                            <button class="btn-modern btn-share share-btn" data-property-id="${property.id}" title="Share"><i class="fa fa-share-alt"></i></button>
                        </div>
                    </div>
                    <div class="popup-compare-bar">
                        <div class="compare-status">Selected for compare: <strong class="compare-count">${compareCount}</strong></div>
                        <button class="btn-modern btn-primary view-compare-btn" data-property-id="${property.id}" ${compareCount < 2 ? 'disabled' : ''}>
                            Compare (${compareCount})
                        </button>
                    </div>
                </div>`;
        }

        const purposeClass = property.purpose === 'rent' ? 'rent' : 'sale';
        const purposeText = purposeClass === 'rent' ? 'For Rent' : 'For Sale';
        const compareCount = this.mapView.state.compareList.length;
        const verifiedBadge = property.is_verified ? (
            `<span class="badge-verified" title="Verified"><i class="fa fa-check-circle"></i> Verified</span>`)
            : '';
        const featuredBadge = property.is_featured ? (
            `<span class="badge-featured" title="Featured"><i class="fa fa-star"></i> Featured</span>`)
            : '';

        const favoriteActive = isFavorite ? 'active' : '';
        const favoriteTitle = isFavorite ? 'Remove from favorites' : 'Add to favorites';
        const favoriteFill = isFavorite ? '#e74c3c' : 'none';
        const compareActive = inCompare ? 'active' : '';
        const compareTitle = inCompare ? 'Remove from compare' : 'Add to compare';
        const compareDisabled = compareCount < 2 ? 'disabled' : '';

        let html = tpl;
        const replacements = {
            '[[image]]': image,
            '[[title]]': property.title || '',
            '[[location]]': property.location || '',
            '[[price]]': price,
            '[[purpose_class]]': purposeClass,
            '[[purpose_text]]': purposeText,
            '[[verified_badge]]': verifiedBadge,
            '[[featured_badge]]': featuredBadge,
            '[[bedrooms]]': String(property.bedrooms || 0),
            '[[bathrooms]]': String(property.bathrooms || 0),
            '[[area]]': String(property.area || 0),
            '[[id]]': String(property.id),
            '[[url]]': property.url || '#',
            '[[favorite_active]]': favoriteActive,
            '[[favorite_title]]': favoriteTitle,
            '[[favorite_fill]]': favoriteFill,
            '[[compare_active]]': compareActive,
            '[[compare_title]]': compareTitle,
            '[[compare_count]]': String(compareCount),
            '[[compare_disabled]]': compareDisabled
        };

        Object.keys(replacements).forEach(k => {
            html = html.split(k).join(replacements[k]);
        });

        return html;
    }
    
    /**
     * Handle marker click
     */
    onMarkerClick(property) {
        // Select property in main view
        this.mapView.selectProperty(property);
        
        // Update popup action buttons
        setTimeout(() => {
            this.attachPopupEventListeners(property);
        }, 100);
    }
    
    /**
     * Handle marker hover
     */
    onMarkerHover(property, marker) {
        // Add hover effect to corresponding list item
        const listItem = document.querySelector(`[data-property-id="${property.id}"]`);
        if (listItem) {
            listItem.classList.add('hovered');
        }
        
        // Optional: could show a lightweight tooltip; skipping for Google Maps
    }
    
    /**
     * Handle marker leave
     */
    onMarkerLeave(property, marker) {
        // Remove hover effect from list item
        const listItem = document.querySelector(`[data-property-id="${property.id}"]`);
        if (listItem) {
            listItem.classList.remove('hovered');
        }
        
        // No custom tooltip to hide in Google version
    }
    
    /**
     * Highlight a specific property marker
     */
    highlightProperty(propertyId) {
        // Remove previous highlight
        if (this.highlightedMarker) {
            this.unhighlightMarker(this.highlightedMarker);
        }
        
        // Find and highlight new marker
        const marker = this.propertyMarkers.get(propertyId);
        if (marker) {
            this.highlightMarker(marker);
            this.highlightedMarker = marker;
            
            // Open InfoWindow (Google Maps)
            const property = (this.mapView.filteredProperties || []).find(p => p.id == propertyId)
                || (this.mapView.properties || []).find(p => p.id == propertyId);
            if (property) {
                const popupContent = this.createPopupContent(property);
                this.infoWindow.setContent(popupContent);
                this.infoWindow.open({ map: this.mapView.map, anchor: marker });
                // Attach popup actions
                setTimeout(() => this.attachPopupEventListeners(property), 50);
            }
            
            // Animate to marker
            this.animateToMarker(marker);
        }
    }
    
    /**
     * Highlight a marker
     */
    highlightMarker(marker) {
        // Handle both AdvancedMarkerElement and legacy Marker
        if (marker.setZIndex && typeof marker.setZIndex === 'function') {
            // Legacy marker
            marker.setZIndex(google.maps.Marker.MAX_ZINDEX + 1);
        } else if (marker.content) {
            // AdvancedMarkerElement - add highlight class
            const content = marker.content.querySelector('.marker-content');
            if (content) {
                content.classList.add('highlighted');
            }
        }
    }
    
    /**
     * Remove highlight from marker
     */
    unhighlightMarker(marker) {
        // Handle both AdvancedMarkerElement and legacy Marker
        if (marker.setZIndex) {
            // Legacy marker
            marker.setZIndex(undefined);
        } else if (marker.content) {
            // AdvancedMarkerElement - remove highlight class
            const content = marker.content.querySelector('.marker-content');
            if (content) {
                content.classList.remove('highlighted');
            }
        }
    }
    
    /**
     * Get marker position (compatible with both AdvancedMarkerElement and legacy Marker)
     */
    getMarkerPosition(marker) {
        if (marker.position) {
            // AdvancedMarkerElement
            return marker.position;
        } else if (marker.getPosition) {
            // Legacy Marker
            return marker.getPosition();
        }
        return null;
    }
    
    /**
     * Animate to marker with pulse effect
     */
    animateToMarker(marker) {
        // Simple pan; CSS pulse not applicable to Google default markers
        const position = this.getMarkerPosition(marker);
        if (position) {
            this.mapView.map.panTo(position);
        }
    }
    
    /**
     * Show property tooltip
     */
    showPropertyTooltip(property, marker) { /* no-op for Google version */ }
    
    /**
     * Hide property tooltip
     */
    hidePropertyTooltip() { /* no-op for Google version */ }
    
    /**
     * Attach event listeners to popup buttons
     */
    attachPopupEventListeners(property) {
        // View details button
        const viewDetailsBtn = document.querySelector('.view-details-btn[data-property-id="' + property.id + '"]');
        if (viewDetailsBtn) {
            viewDetailsBtn.addEventListener('click', (e) => {
                e.stopPropagation();
                if (this.mapView.propertyDetails) {
                    this.mapView.propertyDetails.showPropertyDetails(property);
                }
            });
        }
        
        // Favorite button
        const favoriteBtn = document.querySelector('.favorite-btn[data-property-id="' + property.id + '"]');
        if (favoriteBtn) {
            favoriteBtn.addEventListener('click', (e) => {
                e.stopPropagation();
                this.toggleFavorite(property.id);
            });
        }
        
        // Share button
        const shareBtn = document.querySelector('.share-btn[data-property-id="' + property.id + '"]');
        if (shareBtn) {
            shareBtn.addEventListener('click', (e) => {
                e.stopPropagation();
                this.shareProperty(property);
            });
        }

        // Compare button
        const compareBtn = document.querySelector('.compare-btn[data-property-id="' + property.id + '"]');
        if (compareBtn) {
            compareBtn.addEventListener('click', (e) => {
                e.stopPropagation();
                if (this.mapView.propertyDetails) {
                    this.mapView.propertyDetails.addToCompare(property.id);
                    const inCompare = this.mapView.state.compareList.includes(property.id);
                    compareBtn.classList.toggle('active', inCompare);
                    this.updatePopupCompareBar(property.id);
                }
            });
        }

        // View Compare button in popup bar
        const viewCompareBtn = document.querySelector('.view-compare-btn[data-property-id="' + property.id + '"]');
        if (viewCompareBtn) {
            viewCompareBtn.addEventListener('click', (e) => {
                e.stopPropagation();
                const list = this.mapView.state.compareList || [];
                if (list.length >= 2) {
                    const ids = list.join(',');
                    try { window.location.href = `/properties/compare?ids=${encodeURIComponent(ids)}`; } catch(_) {}
                }
            });
        }
    }

    /**
     * Update the popup compare bar counts and button state
     */
    updatePopupCompareBar(propertyId) {
        const count = (this.mapView?.state?.compareList || []).length;
        try {
            const iw = document.querySelector('.property-popup-modern');
            if (!iw) return;
            const countEl = iw.querySelector('.compare-count');
            if (countEl) countEl.textContent = count;
            const btn = iw.querySelector('.view-compare-btn');
            if (btn) {
                btn.disabled = count < 2;
                btn.textContent = `Compare (${count})`;
            }
        } catch (_) {}
    }
    
    /**
     * Toggle property favorite status
     */
    toggleFavorite(propertyId) {
        const favorites = this.mapView.state.favorites;
        const index = favorites.indexOf(propertyId);
        
        if (index > -1) {
            favorites.splice(index, 1);
        } else {
            favorites.push(propertyId);
        }
        
        // Update localStorage
        localStorage.setItem('property_favorites', JSON.stringify(favorites));
        
        // Update UI
        this.updateFavoriteButtons(propertyId);
        this.mapView.showNotification(
            index > -1 ? 'Removed from favorites' : 'Added to favorites',
            'success'
        );
    }
    
    /**
     * Update favorite button states
     */
    updateFavoriteButtons(propertyId) {
        const isFavorite = this.mapView.state.favorites.includes(propertyId);
        const buttons = document.querySelectorAll(`[data-property-id="${propertyId}"].favorite-btn`);
        
        buttons.forEach(btn => {
            const icon = btn.querySelector('i');
            if (isFavorite) {
                btn.classList.add('active');
                if (icon) icon.className = 'fa fa-heart';
            } else {
                btn.classList.remove('active');
                if (icon) icon.className = 'fa fa-heart-o';
            }
        });
    }
    
    /**
     * Share property
     */
    shareProperty(property) {
        if (navigator.share) {
            navigator.share({
                title: property.title,
                text: `Check out this property: ${property.title} in ${property.location}`,
                url: window.location.href + `?property=${property.id}`
            });
        } else {
            // Fallback: copy to clipboard
            const url = window.location.href + `?property=${property.id}`;
            navigator.clipboard.writeText(url).then(() => {
                this.mapView.showNotification('Property link copied to clipboard', 'success');
            });
        }
    }
    
    /**
     * Update heatmap layer
     */
    updateHeatmap(properties) {
        // Remove existing heatmap
        if (this.heatmapLayer) {
            this.heatmapLayer.setMap(null);
            this.heatmapLayer = null;
        }
        
        // Create heatmap data for Google Maps
        const heatmapData = properties
            .filter(p => p.latitude && p.longitude)
            .map(p => ({ location: new google.maps.LatLng(p.latitude, p.longitude), weight: this.getHeatmapIntensity(p) }));
        
        if (heatmapData.length > 0 && google.maps.visualization) {
            this.heatmapLayer = new google.maps.visualization.HeatmapLayer({
                data: heatmapData,
                radius: 25
            });
            this.heatmapLayer.setMap(this.mapView.map);
        }
    }
    
    /**
     * Get heatmap intensity for property
     */
    getHeatmapIntensity(property) {
        let intensity = 0.5;
        
        // Increase intensity for higher prices
        if (property.price > 5000000) intensity += 0.3;
        else if (property.price > 2000000) intensity += 0.2;
        else if (property.price > 1000000) intensity += 0.1;
        
        // Increase intensity for verified properties
        if (property.is_verified) intensity += 0.1;
        
        // Increase intensity for featured properties
        if (property.is_featured) intensity += 0.2;
        
        return Math.min(intensity, 1.0);
    }
    
    /**
     * Toggle heatmap visibility
     */
    toggleHeatmap() {
        if (!this.heatmapLayer) return;
        
        if (!this.heatmapLayer) return false;
        const visible = this.heatmapLayer.getMap() != null;
        this.heatmapLayer.setMap(visible ? null : this.mapView.map);
        return !visible;
    }
    
    /**
     * Clear all markers
     */
    clearMarkers() {
        if (Array.isArray(this.mapView.markers)) {
            this.mapView.markers.forEach(m => m.setMap(null));
            this.mapView.markers = [];
        }
        this.propertyMarkers.clear();
        this.highlightedMarker = null;
        try { this.infoWindow?.close?.(); } catch (_) {}
        
        if (this.heatmapLayer) {
            this.heatmapLayer.setMap(null);
            this.heatmapLayer = null;
        }
        // Clear clusterer if present
        if (this.clusterer) {
            try { this.clusterer.clearMarkers?.(); } catch (_) {}
            try { this.clusterer.setMap?.(null); } catch (_) {}
            this.clusterer = null;
        }
    }
}

// Export for use in other modules
window.MapMarkers = MapMarkers;
