/**
 * PropertyDetails Module
 * Handles property detail display and interactions
 */
class PropertyDetails {
    constructor(mapView) {
        this.mapView = mapView;
        this.currentProperty = null;
        
        this.init();
    }
    
    /**
     * Initialize property details functionality
     */
    init() {
        this.initializeDrawer();
        this.initializeModals();
        this.initializeEventListeners();
    }
    
    /**
     * Initialize property drawer
     */
    initializeDrawer() {
        const drawer = document.getElementById('property-drawer');
        if (!drawer) return;
        
        // Close drawer button
        const closeBtn = document.getElementById('close-drawer');
        if (closeBtn) {
            closeBtn.addEventListener('click', () => {
                this.hidePropertyDetails();
            });
        }
        
        // Drawer action buttons
        const favoriteBtn = document.getElementById('drawer-favorite');
        const shareBtn = document.getElementById('drawer-share');
        const compareBtn = document.getElementById('drawer-compare');
        
        if (favoriteBtn) {
            favoriteBtn.addEventListener('click', () => {
                if (this.currentProperty) {
                    this.toggleFavorite(this.currentProperty.id);
                }
            });
        }
        
        if (shareBtn) {
            shareBtn.addEventListener('click', () => {
                if (this.currentProperty) {
                    this.shareProperty(this.currentProperty);
                }
            });
        }
        
        if (compareBtn) {
            compareBtn.addEventListener('click', () => {
                if (this.currentProperty) {
                    this.addToCompare(this.currentProperty.id);
                }
            });
        }
    }
    
    /**
     * Initialize modals
     */
    initializeModals() {
        // Contact agent form
        const contactForm = document.getElementById('contact-agent-form');
        if (contactForm) {
            contactForm.addEventListener('submit', (e) => {
                e.preventDefault();
                this.submitContactForm();
            });
        }
        
        // Contact agent button in modal
        const contactAgentBtn = document.getElementById('contact-agent-btn');
        if (contactAgentBtn) {
            contactAgentBtn.addEventListener('click', () => {
                this.showContactAgentModal();
            });
        }
        
        // Export comparison button
        const exportComparisonBtn = document.getElementById('export-comparison');
        if (exportComparisonBtn) {
            exportComparisonBtn.addEventListener('click', () => {
                this.exportComparison();
            });
        }
    }
    
    /**
     * Initialize event listeners
     */
    initializeEventListeners() {
        // Property list item clicks
        document.addEventListener('click', (e) => {
            const listItem = e.target.closest('.property-list-item');
            if (listItem) {
                const propertyId = listItem.dataset.propertyId;
                const property = this.mapView.filteredProperties.find(p => p.id == propertyId);
                if (property) {
                    this.showPropertyDetails(property);
                }
            }
        });
        
        // Favorite button clicks
        document.addEventListener('click', (e) => {
            if (e.target.closest('.favorite-btn')) {
                const btn = e.target.closest('.favorite-btn');
                const propertyId = btn.dataset.propertyId;
                this.toggleFavorite(propertyId);
            }
        });
        
        // Share button clicks
        document.addEventListener('click', (e) => {
            if (e.target.closest('.share-btn')) {
                const btn = e.target.closest('.share-btn');
                const propertyId = btn.dataset.propertyId;
                const property = this.mapView.filteredProperties.find(p => p.id == propertyId);
                if (property) {
                    this.shareProperty(property);
                }
            }
        });
    }
    
    /**
     * Show property details in drawer
     */
    showPropertyDetails(property) {
        this.currentProperty = property;
        
        const drawer = document.getElementById('property-drawer');
        const drawerContent = document.getElementById('drawer-content');
        const drawerTitle = document.getElementById('drawer-title');
        
        if (!drawer || !drawerContent) return;
        
        // Update drawer title
        if (drawerTitle) {
            drawerTitle.textContent = property.title;
        }
        
        // Load detailed property information
        this.loadPropertyDetails(property.id).then(detailedProperty => {
            if (detailedProperty && !detailedProperty.error) {
                this.renderPropertyDetails(detailedProperty);
            } else {
                this.renderPropertyDetails(property);
            }
        }).catch(error => {
            console.error('Error loading property details:', error);
            this.renderPropertyDetails(property);
        });
        
        // Update favorite button state
        this.updateDrawerFavoriteButton(property.id);
    }
    
    /**
     * Load detailed property information
     */
    async loadPropertyDetails(propertyId) {
        try {
            const response = await fetch(`/map/property/${propertyId}`);
            if (response.ok) {
                const data = await response.json();
                return data;
            }
        } catch (error) {
            console.error('Error fetching property details:', error);
        }
        return null;
    }
    
    /**
     * Hide property details drawer
     */
    hidePropertyDetails() {
        const drawer = document.getElementById('property-drawer');
        if (drawer) {
            drawer.classList.remove('open');
        }
        
        this.currentProperty = null;
        
        // Clear highlighted marker
        if (this.mapView.markersManager && this.mapView.markersManager.highlightedMarker) {
            this.mapView.markersManager.unhighlightMarker(this.mapView.markersManager.highlightedMarker);
            this.mapView.markersManager.highlightedMarker = null;
        }
    }
    
    /**
     * Render property details in drawer
     */
    renderPropertyDetails(property) {
        const drawerContent = document.getElementById('drawer-content');
        if (!drawerContent) return;

        // Format price
        const formattedPrice = this.mapView.formatPrice(property.price);

        // Get property images
        const images = property.galleries || [property.image];
        const mainImage = property.image || this.getPlaceholderImage();

        // Generate HTML
        const html = `
            <div class="property-detail-content">
                <!-- Image Gallery -->
                <div class="property-images">
                    <div class="main-image">
                        <img id="main-property-image" src="${mainImage}" alt="${property.title}">
                        ${property.is_verified ? '<div class="verified-badge"><i class="fa fa-shield"></i> Verified</div>' : ''}
                        ${property.is_featured ? '<div class="featured-badge"><i class="fa fa-star"></i> Featured</div>' : ''}
                    </div>
                    ${images.length > 1 ? `
                        <div class="image-thumbnails">
                            ${images.map((img, index) => `
                                <img src="${img}" alt="Thumbnail ${index + 1}"
                                     class="thumbnail ${index === 0 ? 'active' : ''}"
                                     onclick="window.selectImage(${index})">
                            `).join('')}
                        </div>
                        <div class="gallery-controls">
                            <button class="gallery-btn" onclick="window.previousImage()">
                                <i class="fa fa-chevron-left"></i>
                            </button>
                            <button class="gallery-btn" onclick="window.nextImage()">
                                <i class="fa fa-chevron-right"></i>
                            </button>
                        </div>
                    ` : ''}
                </div>

                <!-- Property Info -->
                <div class="property-info">
                    <div class="property-header">
                        <h3 class="property-title">${property.title}</h3>
                        <div class="property-price">${formattedPrice}</div>
                        <div class="property-subtitle">
                            ${property.purpose === 'rent' ? 'For Rent' : 'For Sale'} • ${property.location}
                        </div>
                    </div>

                    <div class="property-specs">
                        <div class="spec-item">
                            <i class="fa fa-bed"></i>
                            <span>${property.bedrooms} Bedrooms</span>
                        </div>
                        <div class="spec-item">
                            <i class="fa fa-bath"></i>
                            <span>${property.bathrooms} Bathrooms</span>
                        </div>
                        <div class="spec-item">
                            <i class="fa fa-home"></i>
                            <span>${property.area} sq ft</span>
                        </div>
                        <div class="spec-item">
                            <i class="fa fa-building"></i>
                            <span>${property.furnishing || 'Not specified'}</span>
                        </div>
                    </div>

                    <div class="property-description">
                        <h5>Property Details</h5>
                        <p>${property.description || 'No description available.'}</p>
                        <p><strong>Address:</strong> ${property.address}</p>
                        <p><strong>Property Type:</strong> ${property.type || 'Not specified'}</p>
                        <p><strong>Location:</strong> ${property.location}</p>
                    </div>

                    <!-- Agent Information -->
                    ${property.agent ? `
                        <div class="agent-section">
                            <h5>Contact Agent</h5>
                            <div class="agent-card">
                                <img src="${property.agent.image || '/assets/images/property-placeholder.jpg'}"
                                     alt="${property.agent.name}" class="agent-avatar">
                                <div class="agent-details">
                                    <h6>${property.agent.name}</h6>
                                    <p>${property.agent.phone}</p>
                                    <p>${property.agent.email}</p>
                                    <div class="agent-actions">
                                        <button class="btn btn-primary btn-sm contact-agent-btn">
                                            <i class="fa fa-phone"></i> Contact
                                        </button>
                                        <button class="btn btn-outline-primary btn-sm schedule-btn">
                                            <i class="fa fa-calendar"></i> Schedule Visit
                                        </button>
                                    </div>
                                </div>
                            </div>
                        </div>
                    ` : ''}

                    <!-- Action Buttons -->
                    <div class="property-actions">
                        <button id="drawer-favorite" class="action-btn favorite-btn" data-property-id="${property.id}">
                            <i class="fa fa-heart-o"></i> Add to Favorites
                        </button>
                        <button class="action-btn share-btn" onclick="shareProperty('${property.id}')">
                            <i class="fa fa-share-alt"></i> Share
                        </button>
                        <button class="action-btn compare-btn" onclick="addToCompare('${property.id}')">
                            <i class="fa fa-plus"></i> Compare
                        </button>
                        <button class="action-btn directions-btn" onclick="getDirections(${property.lat}, ${property.lng})">
                            <i class="fa fa-directions"></i> Directions
                        </button>
                    </div>
                </div>
            </div>
        `;

        drawerContent.innerHTML = html;

        // Initialize image gallery if multiple images
        if (images.length > 1) {
            this.initializeImageGallery(images);
        }
        // Attach event listeners
        this.attachActionListeners(property);
    }

    /**
     * Get placeholder image as data URI
     */
    getPlaceholderImage() {
        // Create a simple SVG placeholder image
        const svgPlaceholder = `
            <svg width="400" height="300" xmlns="http://www.w3.org/2000/svg">
                <rect width="100%" height="100%" fill="#f5f5f5"/>
                <text x="50%" y="50%" font-family="Arial, sans-serif" font-size="18" fill="#999"
                      text-anchor="middle" dominant-baseline="middle">Property Image</text>
            </svg>
        `;
        return `data:image/svg+xml;base64,${btoa(svgPlaceholder)}`;
    }
    initializeImageGallery(images) {
        let currentImageIndex = 0;
        
        window.selectImage = (index) => {
            currentImageIndex = index;
            document.getElementById('main-property-image').src = images[index];
            document.getElementById('current-image').textContent = index + 1;
            
            // Update thumbnail active state
            document.querySelectorAll('.thumbnail').forEach((thumb, i) => {
                thumb.classList.toggle('active', i === index);
            });
        };
        
        window.previousImage = () => {
            currentImageIndex = (currentImageIndex - 1 + images.length) % images.length;
            window.selectImage(currentImageIndex);
        };
        
        window.nextImage = () => {
            currentImageIndex = (currentImageIndex + 1) % images.length;
            window.selectImage(currentImageIndex);
        };
        
        // Keyboard navigation
        document.addEventListener('keydown', (e) => {
            if (document.getElementById('property-drawer').classList.contains('open')) {
                if (e.key === 'ArrowLeft') {
                    window.previousImage();
                } else if (e.key === 'ArrowRight') {
                    window.nextImage();
                }
            }
        });
    }
    
    /**
     * Attach event listeners to action buttons
     */
    attachActionListeners(property) {
        // Contact agent button
        const contactBtn = document.querySelector('.contact-agent-btn');
        if (contactBtn) {
            contactBtn.addEventListener('click', () => {
                this.showContactAgentModal(property);
            });
        }
        
        // Schedule visit button
        const scheduleBtn = document.querySelector('.schedule-btn');
        if (scheduleBtn) {
            scheduleBtn.addEventListener('click', () => {
                this.scheduleVisit(property);
            });
        }
        
        // Virtual tour button
        const virtualTourBtn = document.querySelector('.virtual-tour-btn');
        if (virtualTourBtn) {
            virtualTourBtn.addEventListener('click', () => {
                this.startVirtualTour(property);
            });
        }
        
        // Report issue button
        const reportBtn = document.querySelector('.report-btn');
        if (reportBtn) {
            reportBtn.addEventListener('click', () => {
                this.reportIssue(property);
            });
        }
        
        // Directions button
        window.getDirections = (lat, lng) => {
            this.getDirections(lat, lng);
        };
    }
    
    /**
     * Show contact agent modal
     */
    showContactAgentModal(property = this.currentProperty) {
        const modal = document.getElementById('contact-agent-modal');
        const agentInfo = document.getElementById('agent-info');
        
        if (!modal || !property) return;
        
        // Populate agent information
        if (agentInfo && property.agent) {
            agentInfo.innerHTML = `
                <div class="agent-card">
                    <img src="${property.agent.image || '/assets/images/property-placeholder.jpg'}"
                         alt="${property.agent.name}" class="agent-avatar">
                    <div class="agent-details">
                        <h6>${property.agent.name}</h6>
                        <p>${property.agent.company || 'Real Estate Agent'}</p>
                        <div class="agent-rating">
                            ${this.generateStarRating(property.agent.rating || 4.5)}
                            <span>(${property.agent.reviews || 0} reviews)</span>
                        </div>
                    </div>
                </div>
            `;
        }
        
        // Pre-fill message
        const messageField = document.getElementById('contact-message');
        if (messageField) {
            messageField.value = `Hi, I'm interested in the property "${property.title}" located at ${property.location}. Could you please provide more information?`;
        }
        
        // Show modal
        $(modal).modal('show');
    }
    
    /**
     * Submit contact form
     */
    async submitContactForm() {
        const form = document.getElementById('contact-agent-form');
        if (!form || !this.currentProperty) return;
        
        const formData = new FormData(form);
        const data = {
            property_id: this.currentProperty.id,
            name: document.getElementById('contact-name').value,
            email: document.getElementById('contact-email').value,
            phone: document.getElementById('contact-phone').value,
            message: document.getElementById('contact-message').value
        };
        
        try {
            const response = await fetch('/api/contact-agent', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content')
                },
                body: JSON.stringify(data)
            });
            
            if (response.ok) {
                this.mapView.showNotification('Message sent successfully!', 'success');
                $('#contact-agent-modal').modal('hide');
                form.reset();
            } else {
                this.mapView.showNotification('Failed to send message. Please try again.', 'error');
            }
        } catch (error) {
            console.error('Error sending message:', error);
            this.mapView.showNotification('Failed to send message. Please try again.', 'error');
        }
    }
    
    /**
     * Toggle favorite status
     */
    toggleFavorite(propertyId) {
        const idNum = parseInt(propertyId, 10);
        const idVal = isNaN(idNum) ? propertyId : idNum;
        const favorites = this.mapView.state.favorites;
        const index = favorites.indexOf(idVal);
        
        if (index > -1) {
            favorites.splice(index, 1);
        } else {
            favorites.push(idVal);
        }
        
        // Update localStorage
        localStorage.setItem('property_favorites', JSON.stringify(favorites));
        
        // Update UI
        this.updateFavoriteButtons(idVal);
        this.updateDrawerFavoriteButton(idVal);
        
        this.mapView.showNotification(
            index > -1 ? 'Removed from favorites' : 'Added to favorites',
            'success'
        );
    }
    
    /**
     * Update favorite button states
     */
    updateFavoriteButtons(propertyId) {
        const isFavorite = this.mapView.state.favorites.includes(propertyId);
        const buttons = document.querySelectorAll(`[data-property-id="${propertyId}"].favorite-btn`);
        
        buttons.forEach(btn => {
            const icon = btn.querySelector('i');
            if (isFavorite) {
                btn.classList.add('active');
                if (icon) icon.className = 'fa fa-heart';
            } else {
                btn.classList.remove('active');
                if (icon) icon.className = 'fa fa-heart-o';
            }
        });
    }
    
    /**
     * Update drawer favorite button
     */
    updateDrawerFavoriteButton(propertyId) {
        const drawerFavoriteBtn = document.getElementById('drawer-favorite');
        if (!drawerFavoriteBtn) return;
        
        const isFavorite = this.mapView.state.favorites.includes(propertyId);
        const icon = drawerFavoriteBtn.querySelector('i');
        
        if (isFavorite) {
            drawerFavoriteBtn.classList.add('active');
            drawerFavoriteBtn.title = 'Remove from Favorites';
            if (icon) icon.className = 'fa fa-heart';
        } else {
            drawerFavoriteBtn.classList.remove('active');
            drawerFavoriteBtn.title = 'Add to Favorites';
            if (icon) icon.className = 'fa fa-heart-o';
        }
    }
    
    /**
     * Share property
     */
    shareProperty(property) {
        const shareData = {
            title: property.title,
            text: `Check out this property: ${property.title} in ${property.location}`,
            url: `${window.location.origin}/property/${property.id}`
        };
        
        if (navigator.share) {
            navigator.share(shareData);
        } else {
            // Fallback: copy to clipboard
            navigator.clipboard.writeText(shareData.url).then(() => {
                this.mapView.showNotification('Property link copied to clipboard', 'success');
            });
        }
    }
    
    /**
     * Add property to comparison
     */
    addToCompare(propertyId) {
        const idNum = parseInt(propertyId, 10);
        const idVal = isNaN(idNum) ? propertyId : idNum;
        const compareList = this.mapView.state.compareList;
        const already = compareList.includes(idVal);

        if (already) {
            // Remove from compare
            const idx = compareList.indexOf(idVal);
            if (idx > -1) compareList.splice(idx, 1);
            localStorage.setItem('property_compare', JSON.stringify(compareList));
            this.mapView.showNotification('Removed from comparison', 'success');
        } else {
            if (compareList.length >= 4) {
                this.mapView.showNotification('Maximum 4 properties can be compared', 'warning');
                return;
            }
            compareList.push(idVal);
            localStorage.setItem('property_compare', JSON.stringify(compareList));
            this.mapView.showNotification('Added to comparison', 'success');
        }

        // Update any global compare UI pieces
        this.updateCompareButton();
        try { this.mapView.markersManager?.updatePopupCompareBar?.(idVal); } catch(_) {}
        // Re-render favorites panel if available to keep counts in sync
        try { window.renderFavoritesPanel?.(this.mapView.filteredProperties || this.mapView.properties || []); } catch (_) {}
    }
    
    /**
     * Update compare button
     */
    updateCompareButton() {
        const compareBtn = document.getElementById('compare-favorites');
        if (compareBtn) {
            const count = this.mapView.state.compareList.length;
            compareBtn.disabled = count < 2;
            compareBtn.textContent = `Compare Selected (${count})`;
        }
    }
    
    /**
     * Schedule visit
     */
    scheduleVisit(property) {
        // This would typically open a scheduling modal or redirect to a booking system
        this.mapView.showNotification('Visit scheduling feature coming soon!', 'info');
    }
    
    /**
     * Start virtual tour
     */
    startVirtualTour(property) {
        // This would typically open a virtual tour viewer
        this.mapView.showNotification('Virtual tour feature coming soon!', 'info');
    }
    
    /**
     * Report issue
     */
    reportIssue(property) {
        // This would typically open a report form
        this.mapView.showNotification('Issue reporting feature coming soon!', 'info');
    }
    
    /**
     * Get directions to property
     */
    getDirections(lat, lng) {
        if (navigator.geolocation) {
            navigator.geolocation.getCurrentPosition((position) => {
                const userLat = position.coords.latitude;
                const userLng = position.coords.longitude;
                
                const url = `https://www.google.com/maps/dir/${userLat},${userLng}/${lat},${lng}`;
                window.open(url, '_blank');
            });
        } else {
            const url = `https://www.google.com/maps/place/${lat},${lng}`;
            window.open(url, '_blank');
        }
    }
    
    /**
     * Generate star rating HTML
     */
    generateStarRating(rating) {
        const fullStars = Math.floor(rating);
        const hasHalfStar = rating % 1 !== 0;
        const emptyStars = 5 - fullStars - (hasHalfStar ? 1 : 0);
        
        let html = '';
        
        for (let i = 0; i < fullStars; i++) {
            html += '<i class="fa fa-star"></i>';
        }
        
        if (hasHalfStar) {
            html += '<i class="fa fa-star-half-o"></i>';
        }
        
        for (let i = 0; i < emptyStars; i++) {
            html += '<i class="fa fa-star-o"></i>';
        }
        
        return html;
    }
    
    /**
     * Export comparison
     */
    exportComparison() {
        // This would generate a PDF or Excel file with property comparison
        this.mapView.showNotification('Export feature coming soon!', 'info');
    }
}

// Export for use in other modules
window.PropertyDetails = PropertyDetails;
