/* Compact phone popup behavior
   - Registers modal instances by uid
   - Uses existing /leads endpoint with action=verify_phone and action=verify_otp
   - Preserves requested_action and property_id/owner_id hidden fields
*/
(function(window){
    window.PhonePopup = window.PhonePopup || (function(){
        const instances = {};
        function registerModal(uid) {
            if (!uid || instances[uid]) return;
            const modalId = '#phonePopupModal_' + uid;
            const modalEl = document.querySelector(modalId);
            const sendBtn = document.getElementById('phonePopup_send_' + uid);
            const verifyBtn = document.getElementById('phonePopup_verify_' + uid);
            const resendBtn = document.getElementById('phonePopup_resend_' + uid);
            const otpSection = document.getElementById('phonePopup_otp_' + uid);
            const inputPhone = document.getElementById('phonePopup_phone_' + uid);
            const inputName = document.getElementById('phonePopup_name_' + uid);
            const otpInput = document.getElementById('phonePopup_otpInput_' + uid);
            const responseEl = document.getElementById('phonePopup_response_' + uid);
            const resendCountdown = document.getElementById('phonePopup_resendCountdown_' + uid);

            let resendCooldown = 30; // seconds
            let resendTimer = null;

            function setSpinner(btn, on){
                if (!btn) return;
                const spinner = btn.querySelector('.btn-spinner');
                const text = btn.querySelector('.btn-text');
                if (on) { if (spinner) spinner.style.display='inline-block'; if (text) text.style.opacity='0.7'; }
                else { if (spinner) spinner.style.display='none'; if (text) text.style.opacity='1'; }
            }

            function showResponse(text, ok){
                if (!responseEl) return;
                responseEl.textContent = text;
                responseEl.style.display = 'block';
                responseEl.classList.remove('alert-success','alert-danger');
                responseEl.classList.add('alert', ok ? 'alert-success' : 'alert-danger');
                try { responseEl.focus(); } catch(e){}
            }

            function clearResponse(){ if (responseEl) { responseEl.style.display='none'; responseEl.textContent=''; } }

            function startResendCountdown(){
                if (!resendCountdown) return;
                let time = resendCooldown;
                resendCountdown.style.display = 'inline-block';
                const span = resendCountdown.querySelector('.countdown');
                if (span) span.textContent = String(time);
                if (resendBtn) resendBtn.disabled = true;
                resendTimer && clearInterval(resendTimer);
                resendTimer = setInterval(()=>{
                    time -= 1;
                    if (span) span.textContent = String(time);
                    if (time <= 0) { clearInterval(resendTimer); resendCountdown.style.display='none'; if (resendBtn) resendBtn.disabled=false; }
                }, 1000);
            }

            function sendOtp(){
                clearResponse();
                const name = (inputName && inputName.value || '').trim();
                const phone = (inputPhone && inputPhone.value || '').trim();
                if (!name || name.length < 2) { showResponse('Please enter your name (min 2 chars)', false); inputName && inputName.focus(); return; }
                if (!phone || phone.length < 6) { showResponse('Please enter a valid phone number', false); inputPhone && inputPhone.focus(); return; }
                sendBtn.disabled = true; setSpinner(sendBtn, true);

                const fd = new FormData();
                fd.append('_token', document.querySelector('meta[name="csrf-token"]')?.content || '');
                fd.append('action','verify_phone');
                fd.append('phone', phone);
                const prop = document.getElementById('phonePopup_property_' + uid);
                const owner = document.getElementById('phonePopup_owner_' + uid);
                if (prop) fd.append('property_id', prop.value||'');
                if (owner) fd.append('owner_id', owner.value||'');
                fd.append('name', name);
                fd.append('requested_action', document.getElementById('phonePopup_action_' + uid).value || 'view_number');

                fetch('/leads', { method:'POST', body: fd, headers: {'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.content || ''} })
                .then(function(r){
                    const ct = r.headers.get('content-type') || '';
                    if (ct.toLowerCase().indexOf('application/json') !== -1) return r.json();
                    return r.text().then(function(text){
                        try { return JSON.parse(text); }
                        catch(_){ return { status: 'error', message: 'Server returned non-JSON response', raw: text }; }
                    });
                }).then(function(json){
                    if (json && json.status === 'ok'){
                        otpSection.style.display = 'block';
                        if (inputPhone) inputPhone.readOnly = true;
                        setSpinner(sendBtn, false);
                        sendBtn.disabled = true;
                        if (verifyBtn) verifyBtn.style.display = 'inline-block';
                        showResponse('OTP sent successfully', true);
                        instances[uid].currentPhone = json.phone || phone;
                        instances[uid].name = name;
                        otpInput && otpInput.focus();
                        startResendCountdown();
                    } else {
                        setSpinner(sendBtn, false);
                        sendBtn.disabled = false;
                        const msg = (json && (json.message || (json.raw ? 'Server returned HTML/invalid response' : undefined))) || 'Failed to send OTP';
                        if (json && json.raw) console.error('Non-JSON response from /leads:', json.raw);
                        showResponse(msg, false);
                    }
                }).catch(function(e){
                    console.error('Network error sending OTP', e);
                    setSpinner(sendBtn, false);
                    sendBtn.disabled = false;
                    showResponse('Network error sending OTP', false);
                });
            }

            function verifyOtp(){
                clearResponse();
                const code = (otpInput && otpInput.value || '').trim();
                if (!code || code.length !== 4) { showResponse('Enter 4-digit OTP', false); otpInput && otpInput.focus(); return; }
                verifyBtn.disabled = true; setSpinner(verifyBtn, true);
                const fd = new FormData();
                fd.append('_token', document.querySelector('meta[name="csrf-token"]')?.content || '');
                fd.append('action','verify_otp');
                fd.append('otp', code);
                fd.append('phone', instances[uid].currentPhone || (inputPhone && inputPhone.value || ''));
                if (instances[uid].name) fd.append('name', instances[uid].name);
                const prop = document.getElementById('phonePopup_property_' + uid);
                const owner = document.getElementById('phonePopup_owner_' + uid);
                if (prop) fd.append('property_id', prop.value||'');
                if (owner) fd.append('owner_id', owner.value||'');
                fd.append('requested_action', document.getElementById('phonePopup_action_' + uid).value || 'view_number');

                fetch('/leads', { method:'POST', body: fd, headers: {'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.content || ''} })
                .then(function(r){
                    const ct = r.headers.get('content-type') || '';
                    if (ct.toLowerCase().indexOf('application/json') !== -1) return r.json();
                    return r.text().then(function(text){
                        try { return JSON.parse(text); }
                        catch(_){ return { status: 'error', message: 'Server returned non-JSON response', raw: text }; }
                    });
                }).then(function(json){
                    if (json && json.status === 'ok'){
                        showResponse(json.message || 'Verified', true);
                        const reveal = json.reveal || {};
                        try {
                            const ev = new CustomEvent('phonePopupRevealed', { detail: { uid: uid, phone: reveal.phone || null, reveal: reveal } });
                            window.dispatchEvent(ev);
                        } catch(e){ console.error('Failed to dispatch reveal event', e); }
                        setTimeout(function(){ try{ $(modalId).modal('hide'); }catch(_){}} , 700);
                    } else {
                        const msg = (json && (json.message || (json.raw ? 'Server returned HTML/invalid response' : undefined))) || 'Invalid OTP';
                        if (json && json.raw) console.error('Non-JSON response from /leads:', json.raw);
                        showResponse(msg, false);
                    }
                }).catch(function(e){ console.error('Network error verifying OTP', e); showResponse('Network error verifying OTP', false); })
                .finally(function(){ verifyBtn.disabled=false; setSpinner(verifyBtn, false); });
            }

            // Event wiring & keyboard handling
            if (sendBtn) sendBtn.addEventListener('click', sendOtp);
            if (verifyBtn) verifyBtn.addEventListener('click', verifyOtp);
            if (resendBtn) resendBtn.addEventListener('click', function(e){ e.preventDefault(); sendOtp(); });

            // Allow Enter key to submit appropriately
            if (inputPhone) inputPhone.addEventListener('keydown', function(e){ if (e.key === 'Enter') { e.preventDefault(); sendOtp(); } });
            if (otpInput) otpInput.addEventListener('keydown', function(e){ if (e.key === 'Enter') { e.preventDefault(); verifyOtp(); } });

            // Focus management when modal shown
            if (modalEl) {
                try {
                    $(modalEl).on('shown.bs.modal', function(){ setTimeout(()=>{ if (inputName) inputName.focus(); },50); clearResponse(); });
                    $(modalEl).on('hidden.bs.modal', function(){ // reset UI
                        otpSection.style.display = 'none';
                        if (inputPhone) { inputPhone.readOnly = false; }
                        if (verifyBtn) verifyBtn.style.display = 'none';
                        if (sendBtn) { sendBtn.disabled = false; setSpinner(sendBtn, false); sendBtn.style.display = 'block'; }
                        if (otpInput) otpInput.value = '';
                        if (inputName) inputName.value = '';
                        clearInterval(resendTimer); resendTimer = null; if (resendCountdown) resendCountdown.style.display='none';
                    });
                } catch(e){
                    // if jQuery/bootstrap not available, fallback to simple focus
                    modalEl.addEventListener('focusin', ()=>{ inputName && inputName.focus(); });
                }
            }

            instances[uid] = { uid: uid, currentPhone: '' };
        }

        return { registerModal, _instances: instances };
    })();
})(window);
