<?php

namespace App\Http\Controllers\Admin;
use App\Models\Banner;
use Auth;
use Session;
use App\Settings;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\File;
use App\Models\User;

class HerobannerController extends Controller
{
    public function index()
    {
        if (Auth::user()->usertype != "Admin" && Auth::user()->usertype != "Sub_Admin") {
            Session::flash('flash_message', trans('words.access_denied'));
            return redirect('dashboard');
        }

        $settings = Settings::findOrFail(1);
        $page_title = trans('words.hero_banner');

        // eager load user

        if (Auth::user()->usertype == "Admin" || Auth::user()->usertype == "Sub_Admin") {
        $banners = Banner::with('user')->orderBy('id', 'desc')->paginate(10);
        } else {
            $banners = Auth::user()->banners()->with('user')->orderBy('id', 'desc')->paginate(10);
        }

        return view('admin.pages.herobanner.hero_banner', compact('page_title', 'settings', 'banners'));
    }

    public function create()
    {

        if (Auth::User()->usertype != "Admin" and Auth::User()->usertype != "Sub_Admin") {
            Session::flash('flash_message', trans('words.access_denied'));
            return redirect('dashboard');
        }

        $page_title = trans('words.add_banner');
        $users = User::orderBy('name')->get();

        return view('admin.pages.herobanner.addedit', compact('page_title', 'users'));
    }

    public function store(Request $request)
    {
        if (Auth::User()->usertype != "Admin" and Auth::User()->usertype != "Sub_Admin") {
            Session::flash('flash_message', trans('words.access_denied'));
            return redirect('dashboard');
        }

        $request->validate([
            'name' => 'required|max:150',
            'image' => 'required|mimes:jpg,jpeg,png,gif,webp', // Max 2MB
            'redirect_url' => 'nullable|url',
            'small_title' => 'nullable|string|max:255',
            'main_content' => 'nullable|string|max:255',
            'bottom_content' => 'nullable|string|max:255',
            'cost' => 'required|numeric',
            'status' => ['required', Rule::in([0, 1])],
            'starts_at' => 'nullable|date',
            'ends_at' => 'nullable|date|after_or_equal:starts_at',
            'lat' => 'nullable|numeric',
            'lng' => 'nullable|numeric',
        ]);

        $banner = new Banner();
        
        if (Auth::user()->usertype == "Admin" || Auth::user()->usertype == "Sub_Admin") {
            $request->validate([
                'user_id' => 'required|exists:users,id',
            ]);
            $banner->user_id = $request->user_id;
        } else {
            $banner->user_id = Auth::User()->id;
        }
        
        $banner->name = $request->name;
        $banner->slug = Str::slug($request->name, '-');
        $banner->redirect_url = $request->redirect_url;
        $banner->description = [
            'small_title' => $request->small_title,
            'main_content' => $request->main_content,
            'bottom_content' => $request->bottom_content,
        ];
        $banner->cost = $request->cost;
        $banner->status = $request->status;
        $banner->starts_at = $request->starts_at;
        $banner->ends_at = $request->ends_at;
        $banner->lat = $request->lat;
        $banner->lng = $request->lng;

        if ($request->hasFile('image')) {
            $image = $request->file('image');
            $image_name = time() . '.' . $image->getClientOriginalExtension();
            $destinationPath = public_path('/site_assets/images/banners/');
            $image->move($destinationPath, $image_name);
            $banner->image_path = 'site_assets/images/banners/' . $image_name;
        }

        $banner->save();

        Session::flash('flash_message', trans('words.added'));
        return redirect()->route('admin.herobanner.index');
    }

    public function edit($id)
    {
        if (Auth::User()->usertype != "Admin" and Auth::User()->usertype != "Sub_Admin") {
            Session::flash('flash_message', trans('words.access_denied'));
            return redirect('dashboard');
        }

        if (Auth::user()->usertype == "Admin" || Auth::user()->usertype == "Sub_Admin") {
            $banner = Banner::findOrFail($id);
            $users = User::orderBy('name')->get();
        } else {
            $banner = Auth::user()->banners()->findOrFail($id);
            $users = collect([Auth::user()]); // Only current user for non-admins
        }
        $page_title = trans('words.edit_banner');

        return view('admin.pages.herobanner.addedit', compact('page_title', 'banner', 'users'));
    }

    public function update(Request $request, $id)
    {
        if (Auth::User()->usertype != "Admin" and Auth::User()->usertype != "Sub_Admin") {
            Session::flash('flash_message', trans('words.access_denied'));
            return redirect('dashboard');
        }

        $request->validate([
            'name' => 'required|max:150',
            'image' => 'nullable|mimes:jpg,jpeg,png,gif|max:2048',
            'redirect_url' => 'nullable|url',
            'small_title' => 'nullable|string|max:255',
            'main_content' => 'nullable|string|max:255',
            'bottom_content' => 'nullable|string|max:255',
            'cost' => 'required|numeric',
            'status' => ['required', Rule::in([0, 1])],
            'starts_at' => 'nullable|date',
            'ends_at' => 'nullable|date|after_or_equal:starts_at',
            'lat' => 'nullable|numeric',
            'lng' => 'nullable|numeric',
        ]);

        if (Auth::user()->usertype == "Admin" || Auth::user()->usertype == "Sub_Admin") {
            $banner = Banner::findOrFail($id);
            $request->validate([
                'user_id' => 'required|exists:users,id',
            ]);
            $banner->user_id = $request->user_id;
        } else {
            $banner = Auth::user()->banners()->findOrFail($id);
        }
        $banner->name = $request->name;
        $banner->slug = Str::slug($request->name, '-');
        $banner->redirect_url = $request->redirect_url;
        $banner->description = [
            'small_title' => $request->small_title,
            'main_content' => $request->main_content,
            'bottom_content' => $request->bottom_content,
        ];
        $banner->cost = $request->cost;
        $banner->status = $request->status;
        $banner->starts_at = $request->starts_at;
        $banner->ends_at = $request->ends_at;
        $banner->lat = $request->lat;
        $banner->lng = $request->lng;

        if ($request->hasFile('image')) {
            // Delete old image
            if (File::exists(public_path($banner->image_path))) {
                File::delete(public_path($banner->image_path));
            }

            $image = $request->file('image');
            $image_name = time() . '.' . $image->getClientOriginalExtension();
            $destinationPath = public_path('/site_assets/images/banners/');
            $image->move($destinationPath, $image_name);
            $banner->image_path = 'site_assets/images/banners/' . $image_name;
        }

        $banner->save();

        Session::flash('flash_message', trans('Updated'));
        return redirect()->route('admin.herobanner.index');
    }

    public function destroy($id)
    {
        if (Auth::User()->usertype != "Admin" and Auth::User()->usertype != "Sub_Admin") {
            Session::flash('flash_message', trans('words.access_denied'));
            return redirect('dashboard');
        }

        if (Auth::user()->usertype == "Admin" || Auth::user()->usertype == "Sub_Admin") {
            $banner = Banner::findOrFail($id);
        } else {
            $banner = Auth::user()->banners()->findOrFail($id);
        }

        // Delete image file
        if (File::exists(public_path($banner->image_path))) {
            File::delete(public_path($banner->image_path));
        }

        $banner->delete();

        return response()->json(['status' => '1', 'message' => trans('words.deleted')]);
    }
}
