<?php

namespace App\Http\Controllers\Admin;

use Auth;
use App\User;
use App\Transactions;
use App\SubscriptionPlan;
use App\PaymentGateway;
use Illuminate\Http\Request;
use Session;
use Carbon\Carbon;

class PaymentController extends MainAdminController
{
    public function index()
    {
        if(Auth::User()->usertype!="Admin" AND Auth::User()->usertype!="Sub_Admin") {
            Session::flash('flash_message', trans('words.access_denied'));
            return redirect('admin/dashboard');
        }

        $page_title = 'Payment History Management';
        
        // Get filter parameters
        $status = request('status');
        $gateway = request('gateway');
        $user_id = request('user_id');
        $date_from = request('date_from');
        $date_to = request('date_to');
        $search = request('s');

        // Build query
        $query = Transactions::with(['user', 'subscriptionPlan']);

        if ($status) {
            $query->where('payment_status', $status);
        }

        if ($gateway) {
            $query->where('gateway', $gateway);
        }

        if ($user_id) {
            $query->where('user_id', $user_id);
        }

        if ($date_from && $date_to) {
            $start_timestamp = strtotime($date_from . ' 00:00:00');
            $end_timestamp = strtotime($date_to . ' 23:59:59');
            $query->whereBetween('date', [$start_timestamp, $end_timestamp]);
        }

        if ($search) {
            $query->where(function($q) use ($search) {
                $q->where('payment_id', 'LIKE', "%$search%")
                  ->orWhere('email', 'LIKE', "%$search%")
                  ->orWhere('reference_id', 'LIKE', "%$search%")
                  ->orWhereHas('user', function($userQuery) use ($search) {
                      $userQuery->where('name', 'LIKE', "%$search%");
                  });
            });
        }

        $payments = $query->orderBy('id', 'DESC')->paginate(15);
        
        // Get filter options
        $gateways = PaymentGateway::orderBy('gateway_name')->get();
        $users = User::where('usertype', 'User')->orderBy('name')->get();
        
        // Payment statistics
        $stats = [
            'total_payments' => Transactions::count(),
            'completed_payments' => Transactions::where('payment_status', 'completed')->count(),
            'pending_payments' => Transactions::where('payment_status', 'pending')->count(),
            'failed_payments' => Transactions::where('payment_status', 'failed')->count(),
            'total_revenue' => Transactions::where('payment_status', 'completed')->sum('payment_amount')
        ];

        return view('admin.pages.payments.index', compact(
            'page_title', 'payments', 'gateways', 'users', 'stats'
        ));
    }

    public function show($id)
    {
        if(Auth::User()->usertype!="Admin" AND Auth::User()->usertype!="Sub_Admin") {
            Session::flash('flash_message', trans('words.access_denied'));
            return redirect('admin/dashboard');
        }

        $page_title = 'Payment Details';
        $payment = Transactions::with(['user', 'subscriptionPlan'])->findOrFail($id);

        return view('admin.pages.payments.show', compact('page_title', 'payment'));
    }

    public function updateStatus(Request $request, $id)
    {
        if(Auth::User()->usertype!="Admin") {
            Session::flash('flash_message', trans('words.access_denied'));
            return redirect('admin/dashboard');
        }

        $payment = Transactions::findOrFail($id);
        
        $request->validate([
            'payment_status' => 'required|in:pending,completed,failed,refunded',
            'notes' => 'nullable|string|max:1000'
        ]);

        $payment->payment_status = $request->payment_status;
        if ($request->notes) {
            $payment->notes = $request->notes;
        }
        $payment->save();

        Session::flash('flash_message', 'Payment status updated successfully');
        return redirect()->back();
    }

    public function addManualPayment(Request $request)
    {
        if(Auth::User()->usertype!="Admin") {
            Session::flash('flash_message', trans('words.access_denied'));
            return redirect('admin/dashboard');
        }

        $request->validate([
            'user_id' => 'required|exists:users,id',
            'plan_id' => 'nullable|exists:subscription_plan,id',
            'payment_amount' => 'required|numeric|min:0',
            'gateway' => 'required|string',
            'payment_id' => 'required|string|unique:transaction,payment_id',
            'transaction_type' => 'required|string',
            'payment_status' => 'required|in:pending,completed,failed,refunded',
            'description' => 'nullable|string',
            'notes' => 'nullable|string'
        ]);

        $user = User::findOrFail($request->user_id);
        $plan = $request->plan_id ? SubscriptionPlan::findOrFail($request->plan_id) : null;

        $payment = new Transactions();
        $payment->user_id = $request->user_id;
        $payment->email = $user->email;
        $payment->plan_id = $request->plan_id;
        $payment->payment_amount = $request->payment_amount;
        $payment->gateway = $request->gateway;
        $payment->payment_id = $request->payment_id;
        $payment->transaction_type = $request->transaction_type;
        $payment->description = $request->description ?: ($plan ? "Manual payment for {$plan->plan_name}" : "Custom manual payment");
        $payment->currency = getcong('currency_code') ?: 'USD';
        $payment->original_amount = $request->payment_amount;
        $payment->payment_status = $request->payment_status;
        $payment->reference_id = 'MANUAL_' . time();
        $payment->payment_date = now();
        $payment->ip_address = $request->ip();
        $payment->notes = $request->notes;
        $payment->date = time();
        $payment->save();

        // Update user subscription if payment is completed and plan is selected
        if ($request->payment_status == 'completed' && $plan) {
            $plan_days = $plan->plan_days;
            
            // If user has existing plan, extend from current expiry, otherwise from today
            if ($user->exp_date && $user->exp_date > time()) {
                $user->exp_date = strtotime("+$plan_days days", $user->exp_date);
            } else {
                $user->exp_date = strtotime("+$plan_days days");
            }
            
            $user->plan_id = $request->plan_id;
            $user->save();
        }

        Session::flash('flash_message', 'Manual payment added successfully');
        return redirect()->back();
    }

    public function userPayments($user_id)
    {
        if(Auth::User()->usertype!="Admin" AND Auth::User()->usertype!="Sub_Admin") {
            Session::flash('flash_message', trans('words.access_denied'));
            return redirect('admin/dashboard');
        }

        $page_title = 'User Payment History';
        $user = User::findOrFail($user_id);
        $payments = Transactions::where('user_id', $user_id)
                                ->with('subscriptionPlan')
                                ->orderBy('id', 'DESC')
                                ->paginate(10);

        return view('admin.pages.payments.user_payments', compact('page_title', 'user', 'payments'));
    }


    public function export(Request $request)
    {
        if(Auth::User()->usertype!="Admin") {
            Session::flash('flash_message', trans('words.access_denied'));
            return redirect('admin/dashboard');
        }

        $request->validate([
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date'
        ]);

        $startDate = strtotime($request->start_date);
        $endDate = strtotime($request->end_date . ' 23:59:59');

        $payments = Transactions::with(['user', 'subscriptionPlan'])
                               ->whereBetween('date', [$startDate, $endDate])
                               ->orderBy('date', 'DESC')
                               ->get();

        $filename = 'payment_history_' . date('Y_m_d') . '.csv';
        
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename=\"$filename\"",
        ];

        $callback = function() use ($payments) {
            $file = fopen('php://output', 'w');
            
            // CSV Headers
            fputcsv($file, [
                'ID', 'User Name', 'Email', 'Plan', 'Amount', 'Currency', 
                'Gateway', 'Payment ID', 'Status', 'Type', 'Date', 'Notes'
            ]);

            foreach ($payments as $payment) {
                fputcsv($file, [
                    $payment->id,
                    $payment->user ? $payment->user->name : 'N/A',
                    $payment->email,
                    $payment->subscriptionPlan ? $payment->subscriptionPlan->plan_name : 'N/A',
                    $payment->payment_amount,
                    $payment->currency,
                    $payment->gateway,
                    $payment->payment_id,
                    $payment->payment_status,
                    $payment->transaction_type,
                    date('Y-m-d H:i:s', $payment->date),
                    $payment->notes
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }
}
