<?php

namespace App;

use Illuminate\Database\Eloquent\Model;

class Transactions extends Model
{
    protected $table = 'transaction';

    protected $fillable = [
        'user_id',
        'email', 
        'plan_id',
        'gateway',
        'payment_id',
        'payment_amount',
        'transaction_type',
        'description',
        'currency',
        'original_amount',
        'payment_status',
        'reference_id',
        'payment_details',
        'payment_date',
        'ip_address',
        'notes',
        'date'
    ];

    protected $casts = [
        'payment_details' => 'array',
        'payment_date' => 'datetime',
        'original_amount' => 'decimal:2',
        'payment_amount' => 'decimal:2'
    ];

    public $timestamps = true;

    // Relationships
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function subscriptionPlan()
    {
        return $this->belongsTo(SubscriptionPlan::class, 'plan_id');
    }

    // Scopes
    public function scopeByStatus($query, $status)
    {
        return $query->where('payment_status', $status);
    }

    public function scopeByGateway($query, $gateway)
    {
        return $query->where('gateway', $gateway);
    }

    public function scopeByDateRange($query, $startDate, $endDate)
    {
        return $query->whereBetween('date', [strtotime($startDate), strtotime($endDate)]);
    }

    // Accessors
    public function getFormattedAmountAttribute()
    {
        return html_entity_decode(getCurrencySymbols(getcong('currency_code'))) . ' ' . number_format($this->payment_amount, 2);
    }

    public function getFormattedDateAttribute()
    {
        return date('M d, Y h:i A', $this->date);
    }

    public function getStatusBadgeAttribute()
    {
        $badges = [
            'completed' => 'badge-success',
            'pending' => 'badge-warning',
            'failed' => 'badge-danger',
            'refunded' => 'badge-info'
        ];

        return $badges[$this->payment_status] ?? 'badge-secondary';
    }
}
